@extends('layouts.app')

@section('title', 'Editar Turma - SGE')

@section('content')
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h4 class="mb-0">Editar Turma</h4>
            </div>
            <div class="card-body">
                <form action="{{ route('admin.turmas.update', $turma) }}" method="POST">
                    @csrf
                    @method('PUT')

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="nivel_ensino" class="form-label">Nível de Ensino <span class="text-danger">*</span></label>
                            <select class="form-select @error('nivel_ensino') is-invalid @enderror"
                                    id="nivel_ensino" name="nivel_ensino" required>
                                <option value="">Selecione o nível...</option>
                                <option value="PRIMARIO" {{ old('nivel_ensino', $turma->nivel_ensino ?? 'PRIMARIO') === 'PRIMARIO' ? 'selected' : '' }}>Primário</option>
                                <option value="SECUNDARIO" {{ old('nivel_ensino', $turma->nivel_ensino ?? '') === 'SECUNDARIO' ? 'selected' : '' }}>Secundário</option>
                            </select>
                            @error('nivel_ensino')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6 mb-3">
                            <label for="codigo" class="form-label">Classe <span class="text-danger">*</span></label>
                            <select class="form-select @error('codigo') is-invalid @enderror" 
                                    id="codigo" name="codigo" required>
                                <option value="">Selecione a classe...</option>
                                @for($i = 1; $i <= 12; $i++)
                                    <option value="{{ $i }}ª Classe" {{ old('codigo', $turma->codigo) == $i.'ª Classe' ? 'selected' : '' }}>
                                        {{ $i }}ª Classe
                                    </option>
                                @endfor
                            </select>
                            @error('codigo')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="nome" class="form-label">Turma <span class="text-danger">*</span></label>
                            <select class="form-select @error('nome') is-invalid @enderror" 
                                    id="nome" name="nome" required>
                                <option value="">Selecione a turma...</option>
                                @php
                                    $nomeAtual = old('nome', $turma->nome ?? '');
                                    $nomeNormalizado = '';
                                    
                                    // Normalizar: extrair apenas a letra (A, B, C, etc.)
                                    if (!empty($nomeAtual)) {
                                        // Remover "Turma" do início (case insensitive)
                                        $temp = preg_replace('/^turma\s*/i', '', trim($nomeAtual));
                                        $temp = trim($temp);
                                        
                                        // Se ainda tiver mais de 1 caractere, extrair apenas a letra
                                        if (strlen($temp) > 1) {
                                            if (preg_match('/\b([A-L])\b/i', $temp, $matches)) {
                                                $nomeNormalizado = strtoupper($matches[1]);
                                            } else {
                                                // Tentar pegar o primeiro caractere se for uma letra
                                                $primeiroChar = strtoupper(substr($temp, 0, 1));
                                                if (in_array($primeiroChar, ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L'])) {
                                                    $nomeNormalizado = $primeiroChar;
                                                }
                                            }
                                        } else {
                                            // Se tiver apenas 1 caractere e for uma letra válida
                                            $tempUpper = strtoupper($temp);
                                            if (in_array($tempUpper, ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L'])) {
                                                $nomeNormalizado = $tempUpper;
                                            }
                                        }
                                    }
                                @endphp
                                @foreach(['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L'] as $letra)
                                    <option value="{{ $letra }}" {{ $nomeNormalizado == $letra ? 'selected' : '' }}>
                                        Turma {{ $letra }}
                                    </option>
                                @endforeach
                            </select>
                            @error('nome')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6 mb-3">
                            <label for="ano_letivo_id" class="form-label">Ano Letivo <span class="text-danger">*</span></label>
                            <select class="form-select @error('ano_letivo_id') is-invalid @enderror" 
                                    id="ano_letivo_id" name="ano_letivo_id" required>
                                <option value="">Selecione...</option>
                                @foreach($anoLetivos as $ano)
                                    <option value="{{ $ano->id }}" {{ old('ano_letivo_id', $turma->ano_letivo_id) == $ano->id ? 'selected' : '' }}>
                                        {{ $ano->ano_inicial }}
                                    </option>
                                @endforeach
                            </select>
                            @error('ano_letivo_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="capacidade_maxima" class="form-label">Capacidade Máxima</label>
                            <input type="number" class="form-control @error('capacidade_maxima') is-invalid @enderror" 
                                   id="capacidade_maxima" name="capacidade_maxima" 
                                   value="{{ old('capacidade_maxima', $turma->capacidade_maxima ?? 35) }}" min="1">
                            @error('capacidade_maxima')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="Activa" name="Activa" 
                                   {{ old('Activa', $turma->ativa) ? 'checked' : '' }}>
                            <label class="form-check-label" for="Activa">
                                Turma Activa
                            </label>
                        </div>
                    </div>

                    <div class="d-flex justify-content-between">
                        <a href="{{ route('admin.turmas.index') }}" class="btn btn-secondary">
                            <i class="bi bi-arrow-left me-1"></i> Voltar
                        </a>
                        <button type="submit" class="btn btn-primary-custom">
                            <i class="bi bi-check-circle me-1"></i> Atualizar Turma
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<script>
function initTurmaEditForm() {
    const nivelSelect = document.getElementById('nivel_ensino');
    const classeSelect = document.getElementById('codigo');
    const turmaSelect = document.getElementById('nome');
    if (!nivelSelect || !classeSelect) return;

    const codigoAtual = @json(old('codigo', $turma->codigo));
    const turmaAtualRaw = @json(old('nome', $turma->nome));
    const numeroClasseAtual = parseInt(String(codigoAtual).replace(/\D+/g, ''), 10);
    const valorClasseNormalizado = !isNaN(numeroClasseAtual) ? `${numeroClasseAtual}ª Classe` : (codigoAtual || '');

    function renderClasses() {
        const nivel = nivelSelect.value || 'PRIMARIO';
        const range = nivel === 'SECUNDARIO' ? { start: 7, end: 12 } : { start: 1, end: 6 };
        const valorAtual = classeSelect.value || valorClasseNormalizado;

        classeSelect.innerHTML = '<option value="">Selecione a classe...</option>';
        for (let i = range.start; i <= range.end; i++) {
            const value = `${i}ª Classe`;
            const option = document.createElement('option');
            option.value = value;
            option.textContent = value;
            if (valorAtual === value) {
                option.selected = true;
            }
            classeSelect.appendChild(option);
        }
    }

    function ajustarNivelPeloCodigo() {
        if (!isNaN(numeroClasseAtual)) {
            nivelSelect.value = numeroClasseAtual >= 7 ? 'SECUNDARIO' : 'PRIMARIO';
        }
    }

    function ajustarTurmaAtual() {
        if (!turmaSelect) return;
        
        // Se já houver um valor selecionado (do PHP), não fazer nada
        if (turmaSelect.value && turmaSelect.value !== '') {
            return;
        }
        
        let valor = (turmaAtualRaw || '').toString().trim();
        
        // Normalizar: remover "Turma" do início e espaços
        valor = valor.replace(/^turma\s*/i, '').trim();
        
        // Se ainda não tiver valor, tentar extrair a última letra
        if (!valor && turmaAtualRaw) {
            const match = turmaAtualRaw.toString().match(/\b([A-L])\b/i);
            if (match) valor = match[1].toUpperCase();
        }
        
        if (!valor) return;

        // Tentar definir diretamente pelo value (deve ser apenas a letra: A, B, C, etc.)
        const letraNormalizada = valor.toUpperCase();
        
        // Procurar pela letra no texto da opção
        Array.from(turmaSelect.options).forEach(option => {
            if (option.value === letraNormalizada) {
                option.selected = true;
                turmaSelect.value = letraNormalizada;
            }
        });
    }

    nivelSelect.addEventListener('change', function() {
        renderClasses();
    });

    // Executar imediatamente, mas ajustar turma apenas se não estiver selecionada
    ajustarNivelPeloCodigo();
    renderClasses();
    
    // Só ajustar turma se não houver valor selecionado
    if (!turmaSelect || !turmaSelect.value || turmaSelect.value === '') {
        ajustarTurmaAtual();
    }
}

window.initTurmaEditForm = initTurmaEditForm;

document.addEventListener('DOMContentLoaded', function() {
    initTurmaEditForm();
});
</script>
@endsection

