@extends('layouts.app')

@section('title', 'Professores - SGE')

@section('content')
<div class="row">
    <div class="col-12">
        <div class="d-flex justify-content-between align-items-center mb-2" style="border-bottom: 1px solid #dee2e6; padding-bottom: 8px;">
            <h5 class="mb-0" style="font-weight: 600;">{{ __('common.teachers') }}</h5>
            <div class="d-flex gap-1">
                <a href="{{ route('admin.cartoes-professores.index') }}" class="btn btn-sm btn-outline-success">Crachás</a>
                <a href="{{ route('admin.professores.create') }}" class="btn btn-sm btn-primary">{{ __('common.new_teacher') }}</a>
            </div>
        </div>

        <form method="GET" action="{{ route('admin.professores.index') }}" class="mb-2" style="display: flex; gap: 8px; align-items: end;">
            <input type="text" class="form-control form-control-sm" id="search" name="search" 
                   value="{{ request('search') }}" placeholder="{{ __('common.search') }}..." style="flex: 1;">
            <select class="form-select form-select-sm" id="classe" name="classe" style="width: 150px;">
                <option value="">{{ __('common.all_classes') }}</option>
                @foreach($classes ?? [] as $classe)
                    <option value="{{ $classe }}" {{ request('classe') == $classe ? 'selected' : '' }}>{{ $classe }}</option>
                @endforeach
            </select>
            <select class="form-select form-select-sm" id="turma" name="turma" style="width: 150px;">
                <option value="">{{ __('common.all_turmas') }}</option>
                @foreach($turmas ?? [] as $turma)
                    <option value="{{ $turma }}" {{ request('turma') == $turma ? 'selected' : '' }}>{{ $turma }}</option>
                @endforeach
            </select>
            <button type="submit" class="btn btn-sm btn-primary">{{ __('common.filter') }}</button>
            @if(request('search') || request('classe') || request('turma'))
                <a href="{{ route('admin.professores.index') }}" class="btn btn-sm btn-secondary">{{ __('common.clear') }}</a>
            @endif
        </form>

        <div class="js-ajax-pagination">
            <div class="table-responsive">
                <table class="table table-sm table-bordered" style="font-size: 0.85rem; margin-bottom: 0;">
                <thead style="background-color: #f8f9fa;">
                    <tr>
                        <th style="padding: 6px;">{{ __('common.code') }}</th>
                        <th style="padding: 6px;">{{ __('common.name') }}</th>
                        <th style="padding: 6px;">Contacto</th>
                        <th style="padding: 6px;">Nível</th>
                        <th style="padding: 6px;">{{ __('common.class_turma') }}</th>
                        <th style="padding: 6px;">Salário Fixo</th>
                        <th style="padding: 6px;">Valor/Hora</th>
                        <th style="padding: 6px;">{{ __('common.status') }}</th>
                        <th style="padding: 6px;">{{ __('common.actions') }}</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($professores as $professor)
                        <tr>
                            <td style="padding: 5px;">{{ $professor->numero_funcionario }}</td>
                            <td style="padding: 5px;">{{ $professor->user->name }}</td>
                            <td style="padding: 5px;">{{ $professor->telefone ?? '-' }}</td>
                            <td style="padding: 5px;">
                                @if(($professor->nivel_ensino ?? 'AMBOS') === 'PRIMARIO')
                                    Primário
                                @elseif(($professor->nivel_ensino ?? 'AMBOS') === 'SECUNDARIO')
                                    Secundário
                                @else
                                    Ambos
                                @endif
                            </td>
                            <td style="padding: 5px;">
                                @if($professor->turmas->count() > 0)
                                    @foreach($professor->turmas as $turma)
                                        <span style="font-size: 0.75rem; padding: 2px 6px; background: #e7f3ff; color: #0066cc; border-radius: 3px;">{{ $turma->codigo }} - {{ $turma->nome }}</span>
                                    @endforeach
                                @else
                                    <span style="color: #999;">{{ __('common.no_class') }}</span>
                                @endif
                            </td>
                            <td style="padding: 5px;">
                                @if($professor->salario_fixo)
                                    <span style="font-size: 0.85rem;">{{ number_format($professor->salario_fixo, 2, ',', '.') }} MT</span>
                                @else
                                    <span style="color: #999; font-size: 0.75rem;">-</span>
                                @endif
                            </td>
                            <td style="padding: 5px;">
                                @if(($professor->nivel_ensino ?? 'AMBOS') === 'SECUNDARIO' || ($professor->nivel_ensino ?? 'AMBOS') === 'AMBOS')
                                    <div style="display: flex; flex-direction: column; gap: 2px;">
                                        <input type="number" 
                                               class="form-control form-control-sm valor-hora-input" 
                                               data-professor-id="{{ $professor->id }}"
                                               value="{{ $professor->valor_hora ?? '' }}"
                                               min="0" 
                                               step="0.01" 
                                               placeholder="0.00"
                                               style="width: 100px; font-size: 0.85rem; padding: 2px 6px;">
                                        <small style="font-size: 0.7rem; color: #666;">MT/hora</small>
                                    </div>
                                @else
                                    @if($professor->valor_hora)
                                        <span style="font-size: 0.85rem;">{{ number_format($professor->valor_hora, 2, ',', '.') }} MT</span>
                                    @else
                                        <span style="color: #999; font-size: 0.75rem;">-</span>
                                    @endif
                                @endif
                            </td>
                            <td style="padding: 5px;">
                                @if($professor->user && $professor->user->is_active)
                                    <span style="font-size: 0.75rem; padding: 2px 6px; background: #d4edda; color: #155724; border-radius: 3px;">{{ __('common.active') }}</span>
                                @else
                                    <span style="font-size: 0.75rem; padding: 2px 6px; background: #f8d7da; color: #721c24; border-radius: 3px;">{{ __('common.blocked') }}</span>
                                @endif
                            </td>
                            <td style="padding: 5px;">
                                <div style="display: flex; gap: 4px;">
                                    <a href="{{ route('admin.professores.show', $professor) }}" class="btn btn-sm" style="padding: 2px 6px; font-size: 0.75rem; background: #e7f3ff; color: #0066cc; border: none;" title="Visualizar">
                                        <i class="bi bi-eye"></i>
                                    </a>
                                    <a href="{{ route('admin.professores.edit', $professor) }}" class="btn btn-sm" style="padding: 2px 6px; font-size: 0.75rem; background: #fff3cd; color: #856404; border: none;" title="Editar">
                                        <i class="bi bi-pencil"></i>
                                    </a>
                                    <form action="{{ route('admin.professores.toggle-status', $professor) }}" method="POST" class="d-inline">
                                        @csrf
                                        @method('PATCH')
                                        <button type="submit" class="btn btn-sm" style="padding: 2px 6px; font-size: 0.75rem; background: {{ $professor->user && $professor->user->is_active ? '#fff3cd' : '#d4edda' }}; color: {{ $professor->user && $professor->user->is_active ? '#856404' : '#155724' }}; border: none;" title="{{ $professor->user && $professor->user->is_active ? __('common.block') : __('common.unblock') }}">
                                            <i class="bi bi-{{ $professor->user && $professor->user->is_active ? 'lock' : 'unlock' }}"></i>
                                        </button>
                                    </form>
                                    <form action="{{ route('admin.professores.destroy', $professor) }}" method="POST" class="d-inline delete-form"
                                          data-id="{{ $professor->id }}"
                                          data-confirm="Tem certeza que deseja eliminar este professor? Esta acção não pode ser desfeita.">
                                        @csrf
                                        @method('DELETE')
                                        <button type="submit" class="btn btn-sm" style="padding: 2px 6px; font-size: 0.75rem; background: #f8d7da; color: #721c24; border: none;" title="Eliminar">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="9" class="text-center">Nenhum professor encontrado</td>
                        </tr>
                    @endforelse
                </tbody>
                </table>
            </div>
            
            @if(method_exists($professores, 'links'))
                <div class="mt-3">
                    {{ $professores->appends(request()->query())->links() }}
                </div>
            @endif
        </div>

    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Salvar valor_hora quando mudar (com debounce)
    let valorHoraTimeouts = {};
    document.querySelectorAll('.valor-hora-input').forEach(function(input) {
        input.addEventListener('input', function() {
            const professorId = this.getAttribute('data-professor-id');
            const valorHora = parseFloat(this.value) || 0;
            
            // Limpar timeout anterior
            if (valorHoraTimeouts[professorId]) {
                clearTimeout(valorHoraTimeouts[professorId]);
            }
            
            // Salvar após 1 segundo sem digitar
            valorHoraTimeouts[professorId] = setTimeout(function() {
                salvarValorHora(professorId, valorHora);
            }, 1000);
            
            // Atualizar cálculo imediatamente
            atualizarCalculo(professorId);
        });
    });
    
    // Calcular total quando o campo de horas mudar
    document.querySelectorAll('.horas-input').forEach(function(input) {
        input.addEventListener('input', function() {
            const professorId = this.getAttribute('data-professor-id');
            atualizarCalculo(professorId);
        });
    });
    
    function salvarValorHora(professorId, valorHora) {
        fetch('{{ route("admin.professores.update-valor-hora", ":id") }}'.replace(':id', professorId), {
            method: 'PATCH',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json'
            },
            body: JSON.stringify({
                valor_hora: valorHora > 0 ? valorHora : null
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Atualizar o atributo data-valor-hora nos inputs relacionados
                const horasInput = document.querySelector('.horas-input[data-professor-id="' + professorId + '"]');
                if (horasInput) {
                    horasInput.setAttribute('data-valor-hora', data.valor_hora || 0);
                }
                atualizarCalculo(professorId);
            }
        })
        .catch(error => {
            console.error('Erro ao salvar valor por hora:', error);
        });
    }
    
    function atualizarCalculo(professorId) {
        const horasInput = document.querySelector('.horas-input[data-professor-id="' + professorId + '"]');
        const valorHoraInput = document.querySelector('.valor-hora-input[data-professor-id="' + professorId + '"]');
        
        if (!horasInput) return;
        
        const valorHora = parseFloat(valorHoraInput ? valorHoraInput.value : horasInput.getAttribute('data-valor-hora')) || 0;
        const salarioFixo = parseFloat(horasInput.getAttribute('data-salario-fixo')) || 0;
        const nivel = horasInput.getAttribute('data-nivel');
        const horas = parseFloat(horasInput.value) || 0;
        
        // Atualizar atributo data-valor-hora
        horasInput.setAttribute('data-valor-hora', valorHora);
        
        const totalElement = document.querySelector('.total-salario[data-professor-id="' + professorId + '"]');
        const criarPagamentoBtn = document.querySelector('.criar-pagamento-btn[data-professor-id="' + professorId + '"]');
        
        if (!totalElement) return;
        
        let total = 0;
        
        if (nivel === 'PRIMARIO') {
            // Primário: apenas salário fixo
            total = salarioFixo;
        } else if (nivel === 'SECUNDARIO') {
            // Secundário: horas * valor_hora
            if (horas > 0 && valorHora > 0) {
                total = horas * valorHora;
            }
        } else if (nivel === 'AMBOS') {
            // Ambos: salário fixo + (horas * valor_hora)
            total = salarioFixo || 0;
            if (horas > 0 && valorHora > 0) {
                total += (horas * valorHora);
            }
        }
        
        if (total > 0) {
            // Formatar com vírgula como separador decimal e ponto como separador de milhar
            const formatted = total.toFixed(2).replace('.', ',').replace(/\B(?=(\d{3})+(?!\d))/g, '.');
            totalElement.textContent = formatted + ' MT';
            totalElement.style.color = '#28a745';
            totalElement.style.fontWeight = '600';
            
            // Mostrar botão de criar pagamento
            if (criarPagamentoBtn) {
                criarPagamentoBtn.style.display = 'inline-block';
                criarPagamentoBtn.style.visibility = 'visible';
            }
        } else {
            totalElement.textContent = '-';
            totalElement.style.color = '#999';
            totalElement.style.fontWeight = 'normal';
            
            // Esconder botão de criar pagamento
            if (criarPagamentoBtn) {
                criarPagamentoBtn.style.display = 'none';
                criarPagamentoBtn.style.visibility = 'hidden';
            }
        }
    }
    
    // Mostrar botão para professores que podem ter pagamento
    document.querySelectorAll('.criar-pagamento-btn').forEach(function(btn) {
        const nivel = btn.getAttribute('data-nivel');
        const salarioFixo = parseFloat(btn.getAttribute('data-salario-fixo')) || 0;
        const valorHora = parseFloat(btn.getAttribute('data-valor-hora')) || 0;
        
        // Mostrar se:
        // - Primário com salário fixo
        // - Ambos com salário fixo
        // - Secundário com valor_hora (mesmo sem horas ainda, pode criar depois)
        if ((nivel === 'PRIMARIO' && salarioFixo > 0) || 
            (nivel === 'AMBOS' && salarioFixo > 0) ||
            (nivel === 'SECUNDARIO' && valorHora > 0)) {
            btn.style.display = 'inline-block';
            btn.style.visibility = 'visible';
        }
    });
    
    // Também atualizar o total para professores primários com salário fixo
    document.querySelectorAll('.total-salario').forEach(function(span) {
        const professorId = span.getAttribute('data-professor-id');
        const criarPagamentoBtn = document.querySelector('.criar-pagamento-btn[data-professor-id="' + professorId + '"]');
        
        if (criarPagamentoBtn && span.textContent.trim() !== '-') {
            // Se já tem um valor exibido, mostrar o botão
            criarPagamentoBtn.style.display = 'inline-block';
            criarPagamentoBtn.style.visibility = 'visible';
        }
    });
    
    // Criar pagamento via AJAX
    document.querySelectorAll('.criar-pagamento-btn').forEach(function(btn) {
        btn.addEventListener('click', function() {
            const professorId = this.getAttribute('data-professor-id');
            const professorNome = this.getAttribute('data-professor-nome');
            const nivel = this.getAttribute('data-nivel');
            const salarioFixo = parseFloat(this.getAttribute('data-salario-fixo')) || 0;
            const valorHora = parseFloat(this.getAttribute('data-valor-hora')) || 0;
            
            // Buscar horas do input se existir
            const horasInput = document.querySelector('.horas-input[data-professor-id="' + professorId + '"]');
            const horas = horasInput ? parseFloat(horasInput.value) || 0 : 0;
            
            // Calcular valor total
            let valorTotal = 0;
            if (nivel === 'PRIMARIO') {
                valorTotal = salarioFixo;
            } else if (nivel === 'SECUNDARIO') {
                valorTotal = horas * valorHora;
            } else if (nivel === 'AMBOS') {
                valorTotal = salarioFixo + (horas * valorHora);
            }
            
            if (valorTotal <= 0) {
                alert('O valor total deve ser maior que zero.');
                return;
            }
            
            // Confirmar criação
            const mesAtual = new Date().getMonth() + 1;
            const anoAtual = new Date().getFullYear();
            const meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
            
            if (!confirm(`Criar pagamento para ${professorNome}?\nMês: ${meses[mesAtual]}/${anoAtual}\nValor: ${valorTotal.toFixed(2).replace('.', ',')} MT${horas > 0 ? '\nHoras: ' + horas + 'h' : ''}`)) {
                return;
            }
            
            // Desabilitar botão
            this.disabled = true;
            this.innerHTML = '<i class="bi bi-hourglass-split"></i>';
            
            // Criar pagamento via AJAX
            fetch('{{ route("admin.professor-payments.store") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    professor_id: professorId,
                    mes: mesAtual,
                    ano: anoAtual,
                    horas_trabalhadas: horas > 0 ? horas : null,
                    valor_total: valorTotal,
                    observacoes: 'Criado diretamente da lista de professores'
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Pagamento criado com sucesso!');
                    // Redirecionar para a lista de pagamentos
                    window.location.href = '{{ route("admin.professor-payments.index") }}';
                } else {
                    alert('Erro: ' + (data.message || 'Não foi possível criar o pagamento.'));
                    this.disabled = false;
                    this.innerHTML = '<i class="bi bi-cash-coin"></i>';
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                alert('Erro ao criar pagamento. Tente novamente.');
                this.disabled = false;
                this.innerHTML = '<i class="bi bi-cash-coin"></i>';
            });
        });
    });

    // Interceptar formulários de delete
    document.querySelectorAll('.delete-form').forEach(function(form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const confirmMsg = this.getAttribute('data-confirm');
            if (!confirm(confirmMsg)) {
                return false;
            }
            
            const formData = new FormData(this);
            const row = this.closest('tr');
            const url = this.action;
            
            // Mostrar loading
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalHTML = submitBtn.innerHTML;
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="bi bi-hourglass-split"></i>';
            
            fetch(url, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': formData.get('_token'),
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                },
                body: formData
            })
            .then(response => {
                if (!response.ok) {
                    return response.json().then(data => {
                        throw new Error(data.message || 'Erro ao processar requisição');
                    });
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    // Remover linha da tabela com animação
                    row.style.transition = 'opacity 0.3s';
                    row.style.opacity = '0';
                    setTimeout(function() {
                        row.remove();
                        // Verificar se a tabela ficou vazia
                        const tbody = document.querySelector('tbody');
                        if (tbody && tbody.children.length === 0) {
                            tbody.innerHTML = '<tr><td colspan="9" class="text-center">Nenhum professor encontrado</td></tr>';
                        }
                    }, 300);
                } else {
                    alert('Erro: ' + (data.message || 'Erro desconhecido'));
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalHTML;
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                alert('Erro: ' + (error.message || 'Erro ao eliminar professor. Por favor, tente novamente.'));
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalHTML;
            });
        });
    });
});
</script>
@endpush
@endsection



