@extends('layouts.app')

@section('title', 'Criar Professor - SGE')

@section('content')
<div class="row">
    <div class="col-md-10">
        <div class="card">
            <div class="card-header bg-primary-custom text-white">
                <h4 class="mb-0">Criar Novo Professor</h4>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ route('admin.professores.store') }}" enctype="multipart/form-data">
                    @csrf

                    <div class="row g-2">
                        <div class="col-md-6">
                            <label for="name" class="form-label">Nome *</label>
                            <input type="text" class="form-control form-control-sm @error('name') is-invalid @enderror" 
                                   id="name" name="name" value="{{ old('name') }}" required>
                            @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="genero" class="form-label">Género</label>
                            <select class="form-select form-select-sm @error('genero') is-invalid @enderror" 
                                    id="genero" name="genero">
                                <option value="">Selecione</option>
                                <option value="M" {{ old('genero') == 'M' ? 'selected' : '' }}>Masculino</option>
                                <option value="F" {{ old('genero') == 'F' ? 'selected' : '' }}>Feminino</option>
                                <option value="Outro" {{ old('genero') == 'Outro' ? 'selected' : '' }}>Outro</option>
                            </select>
                            @error('genero')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="telefone" class="form-label">Contacto</label>
                            <input type="text" class="form-control form-control-sm @error('telefone') is-invalid @enderror" 
                                   id="telefone" name="telefone" value="{{ old('telefone') }}" 
                                   placeholder="Ex: 842528792">
                            @error('telefone')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="nivel_ensino" class="form-label">Nível de Ensino *</label>
                            <select class="form-select form-select-sm @error('nivel_ensino') is-invalid @enderror"
                                    id="nivel_ensino" name="nivel_ensino" required>
                                <option value="">Selecione</option>
                                <option value="PRIMARIO" {{ old('nivel_ensino') == 'PRIMARIO' ? 'selected' : '' }}>Primário</option>
                                <option value="SECUNDARIO" {{ old('nivel_ensino') == 'SECUNDARIO' ? 'selected' : '' }}>Secundário</option>
                                <option value="AMBOS" {{ old('nivel_ensino', 'AMBOS') == 'AMBOS' ? 'selected' : '' }}>Ambos</option>
                            </select>
                            @error('nivel_ensino')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Campos de Salário -->
                        <div class="col-md-6" id="salario_fixo_field" style="display: none;">
                            <label for="salario_fixo" class="form-label">Salário Fixo (MT) *</label>
                            <input type="number" step="0.01" min="0" 
                                   class="form-control form-control-sm @error('salario_fixo') is-invalid @enderror" 
                                   id="salario_fixo" name="salario_fixo" 
                                   value="{{ old('salario_fixo') }}" 
                                   placeholder="Ex: 15000.00">
                            <small class="form-text text-muted">Salário fixo para professores da primária</small>
                            @error('salario_fixo')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6" id="valor_hora_field" style="display: none;">
                            <label for="valor_hora" class="form-label">Valor por Hora (MT) *</label>
                            <input type="number" step="0.01" min="0" 
                                   class="form-control form-control-sm @error('valor_hora') is-invalid @enderror" 
                                   id="valor_hora" name="valor_hora" 
                                   value="{{ old('valor_hora') }}" 
                                   placeholder="Ex: 500.00">
                            <small class="form-text text-muted">Valor por hora para professores do secundário</small>
                            @error('valor_hora')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="numero_dependentes" class="form-label">Número de Dependentes</label>
                            <input type="number" min="0" max="10" 
                                   class="form-control form-control-sm @error('numero_dependentes') is-invalid @enderror" 
                                   id="numero_dependentes" name="numero_dependentes" 
                                   value="{{ old('numero_dependentes', 0) }}" 
                                   placeholder="0">
                            <small class="form-text text-muted">Usado para cálculo de IRPS (máximo 4 para efeitos fiscais)</small>
                            @error('numero_dependentes')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="documento_identificacao" class="form-label">Documento (Opcional)</label>
                            <input type="file" class="form-control form-control-sm @error('documento_identificacao') is-invalid @enderror" 
                                   id="documento_identificacao" name="documento_identificacao" 
                                   accept=".pdf,.jpg,.jpeg,.png">
                            @error('documento_identificacao')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="turma_id" class="form-label">Turma (Opcional)</label>
                            <select class="form-select form-select-sm @error('turma_id') is-invalid @enderror" 
                                    id="turma_id" name="turma_id">
                                <option value="">Selecione uma turma</option>
                                @foreach($turmas ?? [] as $turma)
                                    <option value="{{ $turma->id }}"
                                            data-nivel="{{ $turma->nivel_ensino ?? 'PRIMARIO' }}"
                                            {{ old('turma_id') == $turma->id ? 'selected' : '' }}>
                                        {{ $turma->codigo }} - Turma {{ $turma->nome }}
                                    </option>
                                @endforeach
                            </select>
                            @error('turma_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="disciplinas" class="form-label">Disciplinas (Opcional)</label>
                            <select class="form-select form-select-sm @error('disciplinas') is-invalid @enderror" 
                                    id="disciplinas" name="disciplinas[]" multiple size="4">
                                @foreach($disciplinas ?? [] as $disciplina)
                                    <option value="{{ $disciplina->id }}"
                                            data-nivel="{{ $disciplina->nivel_ensino ?? 'PRIMARIO' }}"
                                            {{ in_array($disciplina->id, old('disciplinas', [])) ? 'selected' : '' }}>
                                        {{ $disciplina->nome }}
                                    </option>
                                @endforeach
                            </select>
                            <small class="form-text text-muted">Mantenha Ctrl pressionado para selecionar múltiplas</small>
                            @error('disciplinas')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <div class="mt-3">
                        <button type="submit" class="btn btn-primary-custom btn-sm">Criar Professor</button>
                        <a href="{{ route('admin.professores.index') }}" class="btn btn-secondary btn-sm">Cancelar</a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

<script>
function initProfessorCreateForm() {
    const nivelSelect = document.getElementById('nivel_ensino');
    const disciplinasSelect = document.getElementById('disciplinas');
    const turmasSelect = document.getElementById('turma_id');

    function filtrarDisciplinas() {
        if (!nivelSelect || !disciplinasSelect) return;
        const nivel = nivelSelect.value || 'AMBOS';
        const options = Array.from(disciplinasSelect.options);
        options.forEach(option => {
            const nivelDisciplina = option.getAttribute('data-nivel') || 'PRIMARIO';
            const visivel = (nivel === 'AMBOS') || (nivelDisciplina === nivel);
            option.hidden = !visivel;
            if (!visivel) {
                option.selected = false;
            }
        });
    }

    function filtrarTurmas() {
        if (!nivelSelect || !turmasSelect) return;
        const nivel = nivelSelect.value || 'AMBOS';
        const options = Array.from(turmasSelect.options);
        options.forEach(option => {
            if (!option.value) return;
            const nivelTurma = option.getAttribute('data-nivel') || 'PRIMARIO';
            const visivel = (nivel === 'AMBOS') || (nivelTurma === nivel);
            option.hidden = !visivel;
            if (!visivel && option.selected) {
                option.selected = false;
            }
        });
    }

    // Função para mostrar/ocultar campos de salário baseado no nível
    function atualizarCamposSalario() {
        const nivel = nivelSelect.value;
        const salarioFixoField = document.getElementById('salario_fixo_field');
        const valorHoraField = document.getElementById('valor_hora_field');
        const salarioFixoInput = document.getElementById('salario_fixo');
        const valorHoraInput = document.getElementById('valor_hora');

        if (nivel === 'PRIMARIO' || nivel === 'AMBOS') {
            if (salarioFixoField) salarioFixoField.style.display = 'block';
            if (salarioFixoInput) salarioFixoInput.required = true;
        } else {
            if (salarioFixoField) salarioFixoField.style.display = 'none';
            if (salarioFixoInput) {
                salarioFixoInput.required = false;
                salarioFixoInput.value = '';
            }
        }

        if (nivel === 'SECUNDARIO' || nivel === 'AMBOS') {
            if (valorHoraField) valorHoraField.style.display = 'block';
            if (valorHoraInput) valorHoraInput.required = true;
        } else {
            if (valorHoraField) valorHoraField.style.display = 'none';
            if (valorHoraInput) {
                valorHoraInput.required = false;
                valorHoraInput.value = '';
            }
        }
    }

    if (nivelSelect) {
        nivelSelect.addEventListener('change', function() {
            filtrarDisciplinas();
            filtrarTurmas();
            atualizarCamposSalario();
        });
        filtrarDisciplinas();
        filtrarTurmas();
        atualizarCamposSalario(); // Executar ao carregar a página
    }
}

window.initProfessorCreateForm = initProfessorCreateForm;

document.addEventListener('DOMContentLoaded', function() {
    initProfessorCreateForm();
});
</script>
