@extends('layouts.app')

@section('title', 'Novo Pagamento - SIGE')

@section('content')
<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0">
                    <i class="bi bi-cash-coin me-2"></i>
                    Novo Pagamento de Professor
                </h5>
            </div>
            <div class="card-body">
                <form method="POST" action="{{ route('admin.professor-payments.store') }}" id="paymentForm">
                    @csrf

                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="professor_id" class="form-label">Professor *</label>
                            <select name="professor_id" id="professor_id" class="form-select @error('professor_id') is-invalid @enderror" required>
                                <option value="">Selecione um professor</option>
                                @foreach($professores as $prof)
                                    @if($prof->user)
                                    <option value="{{ $prof->id }}" 
                                            data-nivel="{{ $prof->nivel_ensino }}"
                                            data-salario-fixo="{{ $prof->salario_fixo ?? 0 }}"
                                            data-valor-hora="{{ $prof->valor_hora ?? 0 }}"
                                            {{ old('professor_id', $professor?->id) == $prof->id ? 'selected' : '' }}>
                                        {{ $prof->user->name }} ({{ $prof->numero_funcionario }})
                                        @if($prof->nivel_ensino === 'PRIMARIO')
                                            - Primário
                                        @elseif($prof->nivel_ensino === 'SECUNDARIO')
                                            - Secundário
                                        @else
                                            - Ambos
                                        @endif
                                    </option>
                                    @endif
                                @endforeach
                            </select>
                            @error('professor_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-3">
                            <label for="mes" class="form-label">Mês *</label>
                            <select name="mes" id="mes" class="form-select @error('mes') is-invalid @enderror" required>
                                <option value="">Selecione</option>
                                @for($i = 1; $i <= 12; $i++)
                                    <option value="{{ $i }}" {{ old('mes', date('n')) == $i ? 'selected' : '' }}>
                                        {{ ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'][$i] }}
                                    </option>
                                @endfor
                            </select>
                            @error('mes')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-3">
                            <label for="ano" class="form-label">Ano *</label>
                            <input type="number" name="ano" id="ano" class="form-control @error('ano') is-invalid @enderror" 
                                   value="{{ old('ano', date('Y')) }}" min="2020" max="2100" required>
                            @error('ano')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6" id="horas_field" style="display: none;">
                            <label for="horas_trabalhadas" class="form-label">Horas Trabalhadas *</label>
                            <input type="number" step="0.01" min="0" name="horas_trabalhadas" id="horas_trabalhadas" 
                                   class="form-control @error('horas_trabalhadas') is-invalid @enderror" 
                                   value="{{ old('horas_trabalhadas') }}" placeholder="Ex: 40.00">
                            <small class="form-text text-muted">Digite o número de horas trabalhadas no mês</small>
                            @error('horas_trabalhadas')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-6">
                            <label for="valor_total" class="form-label">Valor Total (MT) *</label>
                            <input type="number" step="0.01" min="0" name="valor_total" id="valor_total" 
                                   class="form-control @error('valor_total') is-invalid @enderror" 
                                   value="{{ old('valor_total') }}" required readonly>
                            <small class="form-text text-muted" id="valor_help">O valor será calculado automaticamente</small>
                            @error('valor_total')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-12">
                            <label for="observacoes" class="form-label">Observações</label>
                            <textarea name="observacoes" id="observacoes" class="form-control" rows="3" 
                                      placeholder="Observações sobre o pagamento...">{{ old('observacoes') }}</textarea>
                        </div>
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save me-1"></i> Salvar Pagamento
                        </button>
                        <a href="{{ route('admin.professor-payments.index') }}" class="btn btn-secondary">
                            <i class="bi bi-arrow-left me-1"></i> Cancelar
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const professorSelect = document.getElementById('professor_id');
    const horasField = document.getElementById('horas_field');
    const horasInput = document.getElementById('horas_trabalhadas');
    const valorTotalInput = document.getElementById('valor_total');
    const valorHelp = document.getElementById('valor_help');
    const mesSelect = document.getElementById('mes');
    const anoInput = document.getElementById('ano');

    function atualizarCampos() {
        const selectedOption = professorSelect.options[professorSelect.selectedIndex];
        if (!selectedOption || !selectedOption.value) {
            horasField.style.display = 'none';
            valorTotalInput.value = '';
            valorTotalInput.readOnly = true;
            return;
        }

        const nivel = selectedOption.getAttribute('data-nivel');
        const salarioFixo = parseFloat(selectedOption.getAttribute('data-salario-fixo')) || 0;
        const valorHora = parseFloat(selectedOption.getAttribute('data-valor-hora')) || 0;

        if (nivel === 'SECUNDARIO' || nivel === 'AMBOS') {
            horasField.style.display = 'block';
            horasInput.required = true;
            valorHelp.textContent = 'Digite as horas trabalhadas para calcular o valor';
            valorTotalInput.readOnly = true;
        } else {
            horasField.style.display = 'none';
            horasInput.required = false;
            horasInput.value = '';
            valorTotalInput.value = salarioFixo.toFixed(2);
            valorHelp.textContent = 'Salário fixo do professor';
            valorTotalInput.readOnly = false;
        }

        calcularValor();
    }

    function calcularValor() {
        const selectedOption = professorSelect.options[professorSelect.selectedIndex];
        if (!selectedOption || !selectedOption.value) return;

        const nivel = selectedOption.getAttribute('data-nivel');
        const salarioFixo = parseFloat(selectedOption.getAttribute('data-salario-fixo')) || 0;
        const valorHora = parseFloat(selectedOption.getAttribute('data-valor-hora')) || 0;
        const horas = parseFloat(horasInput.value) || 0;

        if (nivel === 'PRIMARIO') {
            valorTotalInput.value = salarioFixo.toFixed(2);
        } else if (nivel === 'SECUNDARIO') {
            if (horas > 0) {
                valorTotalInput.value = (valorHora * horas).toFixed(2);
            } else {
                valorTotalInput.value = '';
            }
        } else if (nivel === 'AMBOS') {
            // Se tem horas, calcula por horas, senão usa salário fixo
            if (horas > 0) {
                valorTotalInput.value = (valorHora * horas).toFixed(2);
            } else if (salarioFixo > 0) {
                valorTotalInput.value = salarioFixo.toFixed(2);
            } else {
                valorTotalInput.value = '';
            }
        }
    }

    professorSelect.addEventListener('change', atualizarCampos);
    horasInput.addEventListener('input', calcularValor);

    // Verificar se há professor pré-selecionado
    if (professorSelect.value) {
        atualizarCampos();
    }
});
</script>
@endsection
