<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\AnoLetivo;
use App\Models\Horario;
use App\Models\Evento;
use App\Models\Avaliacao;
use Illuminate\Support\Facades\DB;

class ResetarTurmasEDisciplinas2026Seeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== APAGANDO TODAS AS TURMAS E DISCIPLINAS ===');
        
        // Desabilitar verificação de foreign keys temporariamente
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        // Apagar horários
        Horario::truncate();
        
        // Apagar eventos
        Evento::truncate();
        
        // Apagar avaliações
        Avaliacao::truncate();
        
        // Apagar relações turma_disciplina_professor
        DB::table('turma_disciplina_professor')->truncate();
        
        // Apagar relações turma_disciplina
        DB::table('turma_disciplina')->truncate();
        
        // Apagar relações turma_aluno
        DB::table('turma_aluno')->truncate();
        
        // Apagar todas as turmas
        Turma::truncate();
        
        // Apagar todas as disciplinas
        Disciplina::truncate();
        
        // Reabilitar verificação de foreign keys
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        
        $this->command->info('Todas as turmas e disciplinas foram apagadas.');
        
        // Buscar ou criar ano letivo 2026
        $anoLetivo2026 = AnoLetivo::where(function($q) {
            $q->where('ano', 'like', '%2026%')
              ->orWhere('ano', '2026');
        })->first();
        
        if (!$anoLetivo2026) {
            $this->command->error('Ano letivo 2026 não encontrado! Criando...');
            $anoLetivo2026 = AnoLetivo::firstOrCreate(
                ['ano' => '2026'],
                [
                    'data_inicio' => '2026-01-01',
                    'data_fim' => '2026-12-31',
                    'ativo' => true,
                ]
            );
        }
        
        $this->command->info("Usando ano letivo: {$anoLetivo2026->ano} (ID: {$anoLetivo2026->id})");
        
        // Criar disciplinas
        $this->command->info('=== CRIANDO DISCIPLINAS ===');
        $disciplinas = $this->criarDisciplinas();
        
        // Criar turmas
        $this->command->info('=== CRIANDO TURMAS ===');
        $turmas = $this->criarTurmas($anoLetivo2026);
        
        $this->command->info("=== CONCLUÍDO: {$disciplinas->count()} disciplinas e {$turmas->count()} turmas criadas para 2026 ===");
    }
    
    private function criarDisciplinas()
    {
        $disciplinas = collect();
        
        // Disciplinas do Ensino Primário (carga horária em minutos: 5 horas = 300 minutos)
        $disciplinasPrimario = [
            ['nome' => 'Língua Portuguesa', 'nivel_ensino' => 'PRIMARIO', 'carga_horaria_semanal' => 300],
            ['nome' => 'Matemática', 'nivel_ensino' => 'PRIMARIO', 'carga_horaria_semanal' => 300],
            ['nome' => 'Estudo do Meio', 'nivel_ensino' => 'PRIMARIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'Educação Física', 'nivel_ensino' => 'PRIMARIO', 'carga_horaria_semanal' => 120],
            ['nome' => 'Educação Visual', 'nivel_ensino' => 'PRIMARIO', 'carga_horaria_semanal' => 120],
            ['nome' => 'Educação Musical', 'nivel_ensino' => 'PRIMARIO', 'carga_horaria_semanal' => 120],
        ];
        
        // Disciplinas do Ensino Secundário (carga horária em minutos)
        $disciplinasSecundario = [
            ['nome' => 'Língua Portuguesa', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 240],
            ['nome' => 'Matemática', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 240],
            ['nome' => 'Física', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'Química', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'Biologia', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'História', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'Geografia', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'Inglês', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'Educação Física', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 120],
            ['nome' => 'Desenho', 'nivel_ensino' => 'SECUNDARIO', 'carga_horaria_semanal' => 120],
        ];
        
        foreach ($disciplinasPrimario as $disc) {
            // Gerar código da disciplina (primeiras letras do nome)
            $codigo = strtoupper(substr($disc['nome'], 0, 3));
            if ($disc['nome'] === 'Língua Portuguesa') $codigo = 'LP';
            if ($disc['nome'] === 'Estudo do Meio') $codigo = 'EM';
            if ($disc['nome'] === 'Educação Física') $codigo = 'EF';
            if ($disc['nome'] === 'Educação Visual') $codigo = 'EV';
            if ($disc['nome'] === 'Educação Musical') $codigo = 'EMU';
            
            $disciplina = Disciplina::create([
                'nome' => $disc['nome'],
                'codigo' => $codigo,
                'nivel_ensino' => $disc['nivel_ensino'],
                'carga_horaria_semanal' => $disc['carga_horaria_semanal'],
                'ativa' => true,
            ]);
            $disciplinas->push($disciplina);
            $this->command->info("Disciplina criada: {$disc['nome']} ({$disc['nivel_ensino']})");
        }
        
        foreach ($disciplinasSecundario as $disc) {
            // Gerar código da disciplina (com sufixo S para secundário)
            $codigo = strtoupper(substr($disc['nome'], 0, 3)) . 'S';
            if ($disc['nome'] === 'Língua Portuguesa') $codigo = 'LPS';
            if ($disc['nome'] === 'Matemática') $codigo = 'MATS';
            if ($disc['nome'] === 'Física') $codigo = 'FIS';
            if ($disc['nome'] === 'Química') $codigo = 'QUI';
            if ($disc['nome'] === 'Biologia') $codigo = 'BIO';
            if ($disc['nome'] === 'História') $codigo = 'HIS';
            if ($disc['nome'] === 'Geografia') $codigo = 'GEO';
            if ($disc['nome'] === 'Inglês') $codigo = 'ING';
            if ($disc['nome'] === 'Educação Física') $codigo = 'EFS';
            if ($disc['nome'] === 'Desenho') $codigo = 'DES';
            
            $disciplina = Disciplina::create([
                'nome' => $disc['nome'],
                'codigo' => $codigo,
                'nivel_ensino' => $disc['nivel_ensino'],
                'carga_horaria_semanal' => $disc['carga_horaria_semanal'],
                'ativa' => true,
            ]);
            $disciplinas->push($disciplina);
            $this->command->info("Disciplina criada: {$disc['nome']} ({$disc['nivel_ensino']})");
        }
        
        return $disciplinas;
    }
    
    private function criarTurmas($anoLetivo)
    {
        $turmas = collect();
        
        // Turmas do Ensino Primário (1ª a 6ª classe)
        $classesPrimario = ['1ª', '2ª', '3ª', '4ª', '5ª', '6ª'];
        $turmasPrimario = ['A', 'B'];
        
        foreach ($classesPrimario as $classe) {
            foreach ($turmasPrimario as $turma) {
                $turmaCriada = Turma::create([
                    'codigo' => $classe . ' Classe',
                    'nome' => $turma,
                    'nivel_ensino' => 'PRIMARIO',
                    'ano_letivo_id' => $anoLetivo->id,
                    'capacidade_maxima' => 30,
                    'ativa' => true,
                ]);
                $turmas->push($turmaCriada);
                $this->command->info("Turma criada: {$classe} Classe - {$turma}");
            }
        }
        
        // Turmas do Ensino Secundário (7ª a 12ª classe)
        $classesSecundario = ['7ª', '8ª', '9ª', '10ª', '11ª', '12ª'];
        $turmasSecundario = ['A', 'B'];
        
        foreach ($classesSecundario as $classe) {
            foreach ($turmasSecundario as $turma) {
                $turmaCriada = Turma::create([
                    'codigo' => $classe . ' Classe',
                    'nome' => $turma,
                    'nivel_ensino' => 'SECUNDARIO',
                    'ano_letivo_id' => $anoLetivo->id,
                    'capacidade_maxima' => 30,
                    'ativa' => true,
                ]);
                $turmas->push($turmaCriada);
                $this->command->info("Turma criada: {$classe} Classe - {$turma}");
            }
        }
        
        return $turmas;
    }
}
