<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Aluno;
use App\Models\Turma;
use App\Models\AnoLetivo;
use App\Models\Financeiro;
use App\Models\Payment;
use App\Models\Invoice;
use App\Models\Avaliacao;
use App\Services\CodigoEstudanteService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class ResetarECriar300Alunos2026Seeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== APAGANDO TODOS OS ALUNOS ===');
        
        // Desabilitar verificação de foreign keys temporariamente
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        // Apagar todas as avaliações
        DB::table('avaliacoes')->truncate();
        
        // Apagar todas as relações turma_aluno
        DB::table('turma_aluno')->truncate();
        
        // Apagar todos os financeiros relacionados
        DB::table('invoices')->truncate();
        DB::table('payments')->truncate();
        DB::table('financeiro')->truncate();
        
        // Reabilitar verificação de foreign keys
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        
        // Apagar todos os alunos e seus usuários
        $alunos = Aluno::with('user')->get();
        foreach ($alunos as $aluno) {
            if ($aluno->user) {
                $aluno->user->delete();
            }
            $aluno->forceDelete();
        }
        
        $this->command->info('Todos os alunos foram apagados.');
        
        // Buscar ano letivo 2026
        $anoLetivo2026 = AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivo2026) {
            $this->command->error('Ano letivo 2026 não encontrado! Criando...');
            $anoLetivo2026 = AnoLetivo::firstOrCreate(
                ['ano' => '2026'],
                [
                    'data_inicio' => '2026-01-01',
                    'data_fim' => '2026-12-31',
                    'ativo' => true,
                ]
            );
        }
        
        $this->command->info("Usando ano letivo: {$anoLetivo2026->ano} (ID: {$anoLetivo2026->id})");
        
        // Buscar todas as turmas ativas de 2026
        $turmas = Turma::where('ano_letivo_id', $anoLetivo2026->id)
            ->where('ativa', true)
            ->orderBy('codigo')
            ->orderBy('nome')
            ->get();
        
        if ($turmas->isEmpty()) {
            $this->command->error('Nenhuma turma encontrada para 2026!');
            $this->command->info('Criando turmas básicas...');
            $turmas = $this->criarTurmasBasicas($anoLetivo2026);
        }
        
        $this->command->info("Encontradas {$turmas->count()} turmas para distribuir os alunos.");
        
        // Criar 300 alunos
        $this->command->info('=== CRIANDO 300 ALUNOS ===');
        $alunosCriados = $this->criar300Alunos($anoLetivo2026, $turmas);
        
        $this->command->info("=== CONCLUÍDO: {$alunosCriados} alunos criados e distribuídos nas turmas de 2026 ===");
    }
    
    private function criarTurmasBasicas($anoLetivo)
    {
        $turmas = collect();
        
        // Criar turmas do ensino primário (1ª a 6ª classe)
        $classesPrimario = ['1ª', '2ª', '3ª', '4ª', '5ª', '6ª'];
        $turmasPrimario = ['A', 'B'];
        
        foreach ($classesPrimario as $classe) {
            foreach ($turmasPrimario as $turma) {
                $turmaCriada = Turma::firstOrCreate(
                    [
                        'codigo' => $classe . ' Classe',
                        'nome' => $turma,
                        'ano_letivo_id' => $anoLetivo->id,
                    ],
                    [
                        'nivel_ensino' => 'PRIMARIO',
                        'capacidade_maxima' => 30,
                        'ativa' => true,
                    ]
                );
                $turmas->push($turmaCriada);
            }
        }
        
        // Criar turmas do ensino secundário (7ª a 12ª classe)
        $classesSecundario = ['7ª', '8ª', '9ª', '10ª', '11ª', '12ª'];
        $turmasSecundario = ['A', 'B'];
        
        foreach ($classesSecundario as $classe) {
            foreach ($turmasSecundario as $turma) {
                $turmaCriada = Turma::firstOrCreate(
                    [
                        'codigo' => $classe . ' Classe',
                        'nome' => $turma,
                        'ano_letivo_id' => $anoLetivo->id,
                    ],
                    [
                        'nivel_ensino' => 'SECUNDARIO',
                        'capacidade_maxima' => 30,
                        'ativa' => true,
                    ]
                );
                $turmas->push($turmaCriada);
            }
        }
        
        return $turmas;
    }
    
    private function criar300Alunos($anoLetivo, $turmas)
    {
        $nomes = $this->gerarNomes(300);
        $anoIngresso = 2026;
        $timestamp = time();
        $index = 1;
        $criados = 0;
        
        // Distribuir alunos de forma equilibrada entre as turmas
        $turmasArray = $turmas->toArray();
        $turmaIndex = 0;
        $alunosPorTurma = [];
        
        // Inicializar contador por turma
        foreach ($turmas as $turma) {
            $alunosPorTurma[$turma->id] = 0;
        }
        
        // Calcular quantos alunos por turma (distribuição equilibrada)
        $totalTurmas = $turmas->count();
        $alunosPorTurmaBase = floor(300 / $totalTurmas);
        $alunosRestantes = 300 % $totalTurmas;
        
        foreach ($turmas as $turma) {
            $alunosPorTurma[$turma->id] = $alunosPorTurmaBase;
            if ($alunosRestantes > 0) {
                $alunosPorTurma[$turma->id]++;
                $alunosRestantes--;
            }
        }
        
        $this->command->info("Distribuição: aproximadamente {$alunosPorTurmaBase} alunos por turma");
        
        foreach ($nomes as $key => $nomeCompleto) {
            if ($criados >= 300) break;
            
            // Selecionar turma de forma rotativa
            $turma = $turmas[$turmaIndex % $turmas->count()];
            
            // Verificar se a turma já atingiu sua cota
            if ($alunosPorTurma[$turma->id] <= 0) {
                $turmaIndex++;
                $turma = $turmas[$turmaIndex % $turmas->count()];
            }
            
            // Verificar capacidade máxima da turma
            $alunosNaTurma = DB::table('turma_aluno')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->count();
            
            if ($alunosNaTurma >= ($turma->capacidade_maxima ?? 30)) {
                // Turma cheia, tentar próxima
                $turmaIndex++;
                $turma = $turmas[$turmaIndex % $turmas->count()];
            }
            
            $email = strtolower(str_replace([' ', 'á', 'à', 'â', 'ã', 'é', 'ê', 'í', 'ó', 'ô', 'õ', 'ú', 'ç'], 
                ['', 'a', 'a', 'a', 'a', 'e', 'e', 'i', 'o', 'o', 'o', 'u', 'c'], 
                $nomeCompleto)) . $timestamp . $index . '@aluno2026.sge.pt';
            
            // Verificar se email já existe
            if (User::where('email', $email)->exists()) {
                $index++;
                continue;
            }
            
            try {
                DB::beginTransaction();
                
                // Criar usuário
                $user = User::create([
                    'name' => $nomeCompleto,
                    'email' => $email,
                    'password' => Hash::make('2026'),
                    'tipo' => 'aluno',
                    'must_change_password' => true,
                    'is_active' => true,
                ]);
                
                // Gerar código de estudante
                $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);
                
                // Criar aluno
                $aluno = Aluno::create([
                    'user_id' => $user->id,
                    'codigo_estudante' => $codigoEstudante,
                    'telefone' => '+258 8' . rand(1000000, 9999999),
                    'endereco' => 'Endereço ' . $index . ', Maputo',
                    'data_nascimento' => now()->subYears(rand(6, 18))->subDays(rand(0, 365)),
                    'genero' => $this->gerarGenero(),
                    'nome_encarregado' => 'Encarregado de ' . $nomeCompleto,
                    'telefone_encarregado' => '+258 8' . rand(1000000, 9999999),
                    'modalidade_pagamento' => ['Mensal', 'Trimestral', 'Anual'][rand(0, 2)],
                ]);
                
                // Alocar aluno à turma
                $aluno->turmas()->attach($turma->id, ['ano_letivo_id' => $anoLetivo->id]);
                
                DB::commit();
                
                $alunosPorTurma[$turma->id]--;
                $turmaIndex++;
                $index++;
                $criados++;
                
                if ($criados % 50 == 0) {
                    $this->command->info("Criados {$criados}/300 alunos...");
                }
                
            } catch (\Exception $e) {
                DB::rollBack();
                $this->command->error("Erro ao criar aluno {$nomeCompleto}: " . $e->getMessage());
                $index++;
                continue;
            }
        }
        
        return $criados;
    }
    
    private function gerarNomes($quantidade)
    {
        $nomes = [];
        $nomesPrimeiro = [
            'Ana', 'Maria', 'Sofia', 'Beatriz', 'Carolina', 'Diana', 'Eva', 'Filipa', 'Gabriela', 'Helena',
            'Inês', 'Joana', 'Laura', 'Mariana', 'Natália', 'Olívia', 'Patrícia', 'Rita', 'Teresa', 'Vera',
            'André', 'Bruno', 'Carlos', 'Daniel', 'Eduardo', 'Filipe', 'Gonçalo', 'Hugo', 'Ivo', 'João',
            'Luís', 'Miguel', 'Nuno', 'Paulo', 'Ricardo', 'Sérgio', 'Tiago', 'Vasco', 'Xavier', 'Zé'
        ];
        
        $nomesUltimo = [
            'Santos', 'Silva', 'Oliveira', 'Costa', 'Pereira', 'Rodrigues', 'Martins', 'Ferreira', 'Alves', 'Fernandes',
            'Gomes', 'Lopes', 'Carvalho', 'Sousa', 'Ribeiro', 'Pinto', 'Correia', 'Mendes', 'Moreira', 'Araújo',
            'Machado', 'Rocha', 'Teixeira', 'Cunha', 'Dias', 'Monteiro', 'Cardoso', 'Nunes', 'Ramos', 'Coelho'
        ];
        
        for ($i = 0; $i < $quantidade; $i++) {
            $nomePrimeiro = $nomesPrimeiro[array_rand($nomesPrimeiro)];
            $nomeUltimo = $nomesUltimo[array_rand($nomesUltimo)];
            
            // Adicionar segundo nome ocasionalmente
            if (rand(0, 1)) {
                $segundoNome = $nomesPrimeiro[array_rand($nomesPrimeiro)];
                $nomes[] = $nomePrimeiro . ' ' . $segundoNome . ' ' . $nomeUltimo;
            } else {
                $nomes[] = $nomePrimeiro . ' ' . $nomeUltimo;
            }
        }
        
        return $nomes;
    }
    
    private function gerarGenero()
    {
        return rand(0, 1) == 0 ? 'M' : 'F';
    }
}
