<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\Turma;
use App\Models\AnoLetivo;
use App\Models\Financeiro;
use App\Services\CodigoEstudanteService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class ResetAndPopulate10Seeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('Limpando dados existentes...');
        
        // Desabilitar verificação de foreign keys temporariamente
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        // Limpar mensalidades
        DB::table('financeiro')->truncate();
        
        // Limpar avaliações
        DB::table('avaliacoes')->truncate();
        DB::table('turma_disciplina_professor')->truncate();
        DB::table('turma_aluno')->truncate();
        DB::table('turma_disciplina')->truncate();
        
        // Limpar alunos e seus usuários
        $alunos = Aluno::with('user')->get();
        foreach ($alunos as $aluno) {
            if ($aluno->user) {
                $aluno->user->delete();
            }
            $aluno->forceDelete();
        }
        
        // Limpar professores e seus usuários
        $professores = Professor::with('user')->get();
        foreach ($professores as $professor) {
            if ($professor->user) {
                $professor->user->delete();
            }
            $professor->forceDelete();
        }
        
        // Limpar turmas
        DB::table('turmas')->truncate();
        
        // Reabilitar verificação de foreign keys
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        
        $this->command->info('Criando 10 alunos...');
        $this->criarAlunos();
        
        $this->command->info('Criando 10 professores...');
        $this->criarProfessores();
        
        $this->command->info('Criando 10 turmas...');
        $this->criarTurmas();
        
        $this->command->info('Criando 10 mensalidades...');
        $this->criarMensalidades();
        
        $this->command->info('Dados populados com sucesso!');
    }
    
    private function criarAlunos()
    {
        $nomes = [
            'Maria Santos', 'Ana Oliveira', 'Sofia Pereira', 'Beatriz Rodrigues', 'Inês Martins',
            'João Silva', 'Pedro Costa', 'Carlos Mendes', 'Miguel Fernandes', 'Tiago Alves'
        ];
        
        $generos = ['F', 'F', 'F', 'F', 'F', 'M', 'M', 'M', 'M', 'M'];
        
        $anoAtual = date('Y');
        $anoIngresso = $anoAtual;
        $index = 1;
        $timestamp = time();
        
        foreach ($nomes as $key => $nome) {
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@aluno.sge.pt';
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make($anoAtual),
                'tipo' => 'aluno',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);
            
            Aluno::create([
                'user_id' => $user->id,
                'codigo_estudante' => $codigoEstudante,
                'telefone' => '+258 8' . rand(1000000, 9999999),
                'endereco' => 'Endereço ' . $index . ', Maputo',
                'data_nascimento' => now()->subYears(rand(10, 18))->subDays(rand(0, 365)),
                'genero' => $generos[$key],
                'nome_encarregado' => 'Encarregado de ' . $nome,
                'telefone_encarregado' => '+258 8' . rand(1000000, 9999999),
            ]);
            
            $index++;
        }
    }
    
    private function criarProfessores()
    {
        $nomes = [
            'António Machado', 'Manuel Santos', 'José Silva', 'Ricardo Mendes', 'Paulo Fernandes',
            'Isabel Ferreira', 'Cristina Oliveira', 'Patrícia Costa', 'Teresa Rodrigues', 'Sandra Pinto'
        ];
        
        $generos = ['M', 'M', 'M', 'M', 'M', 'F', 'F', 'F', 'F', 'F'];
        
        $anoAtual = date('Y');
        $index = 1;
        $timestamp = time();
        
        foreach ($nomes as $key => $nome) {
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@professor.sge.pt';
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make($anoAtual),
                'tipo' => 'professor',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            Professor::create([
                'user_id' => $user->id,
                'numero_funcionario' => 'PROF' . str_pad($index, 4, '0', STR_PAD_LEFT),
                'telefone' => '+258 8' . rand(1000000, 9999999),
                'endereco' => 'Endereço ' . $index . ', Maputo',
                'data_nascimento' => now()->subYears(rand(25, 55))->subDays(rand(0, 365)),
                'genero' => $generos[$key],
            ]);
            
            $index++;
        }
    }
    
    private function criarTurmas()
    {
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        
        if (!$anoLetivo) {
            $this->command->warn('Nenhum ano letivo ativo encontrado.');
            return;
        }
        
        $classes = ['1ª Classe', '2ª Classe', '3ª Classe', '4ª Classe', '5ª Classe'];
        $nomesTurmas = ['A', 'B'];
        
        $index = 1;
        foreach ($classes as $classe) {
            foreach ($nomesTurmas as $nomeTurma) {
                Turma::create([
                    'codigo' => $classe,
                    'nome' => $nomeTurma,
                    'ano_letivo_id' => $anoLetivo->id,
                    'capacidade_maxima' => 30,
                    'ativa' => true,
                ]);
                $index++;
                if ($index > 10) break 2;
            }
        }
    }
    
    private function criarMensalidades()
    {
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        $alunos = Aluno::all();
        
        if (!$anoLetivo || $alunos->isEmpty()) {
            $this->command->warn('Nenhum ano letivo ou aluno encontrado.');
            return;
        }
        
        $meses = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10];
        $ano = date('Y');
        
        foreach ($alunos->take(10) as $aluno) {
            foreach ($meses as $mes) {
                Financeiro::create([
                    'aluno_id' => $aluno->id,
                    'ano_letivo_id' => $anoLetivo->id,
                    'tipo' => 'Mensalidade',
                    'descricao' => "Mensalidade {$mes}/{$ano}",
                    'valor' => rand(500, 2000),
                    'data_vencimento' => date('Y-m-d', mktime(0, 0, 0, $mes, 5, $ano)),
                    'status' => rand(0, 1) ? 'PAGO' : 'PENDENTE',
                ]);
            }
        }
    }
}

