<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\AnoLetivo;
use App\Models\Horario;
use App\Models\Evento;
use App\Models\Avaliacao;
use Illuminate\Support\Facades\DB;

class RecriarTurmasEAlocarSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== APAGANDO TURMAS E RELAÇÕES ===');
        
        // Desabilitar verificação de foreign keys temporariamente
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        // Apagar horários
        Horario::truncate();
        
        // Apagar eventos
        Evento::truncate();
        
        // Apagar avaliações
        Avaliacao::truncate();
        
        // Apagar relações turma_disciplina_professor
        DB::table('turma_disciplina_professor')->truncate();
        
        // Apagar relações turma_disciplina
        DB::table('turma_disciplina')->truncate();
        
        // Apagar relações turma_aluno
        DB::table('turma_aluno')->truncate();
        
        // Apagar todas as turmas
        Turma::truncate();
        
        // Reabilitar verificação de foreign keys
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        
        $this->command->info('Turmas e relações apagadas.');
        
        // Buscar ano letivo 2026
        $anoLetivo2026 = AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivo2026) {
            $this->command->error('Ano letivo 2026 não encontrado! Criando...');
            $anoLetivo2026 = AnoLetivo::firstOrCreate(
                ['ano' => '2026'],
                [
                    'data_inicio' => '2026-01-01',
                    'data_fim' => '2026-12-31',
                    'ativo' => true,
                ]
            );
        }
        
        $this->command->info("Usando ano letivo: {$anoLetivo2026->ano} (ID: {$anoLetivo2026->id})");
        
        // Buscar disciplinas existentes
        $disciplinasPrimario = Disciplina::where('nivel_ensino', 'PRIMARIO')->where('ativa', true)->get();
        $disciplinasSecundario = Disciplina::where('nivel_ensino', 'SECUNDARIO')->where('ativa', true)->get();
        
        if ($disciplinasPrimario->isEmpty() && $disciplinasSecundario->isEmpty()) {
            $this->command->error('Nenhuma disciplina encontrada! Execute primeiro o seeder de disciplinas.');
            return;
        }
        
        $this->command->info("Encontradas {$disciplinasPrimario->count()} disciplinas do primário e {$disciplinasSecundario->count()} do secundário.");
        
        // Criar turmas
        $this->command->info('=== CRIANDO TURMAS ===');
        $turmas = $this->criarTurmas($anoLetivo2026);
        
        // Atribuir disciplinas às turmas
        $this->command->info('=== ATRIBUINDO DISCIPLINAS ÀS TURMAS ===');
        $this->atribuirDisciplinas($turmas, $disciplinasPrimario, $disciplinasSecundario, $anoLetivo2026);
        
        // Alocar alunos
        $this->command->info('=== ALOCANDO ALUNOS ===');
        $this->alocarAlunos($turmas, $anoLetivo2026);
        
        // Alocar professores
        $this->command->info('=== ALOCANDO PROFESSORES ===');
        $this->alocarProfessores($turmas, $anoLetivo2026);
        
        $this->command->info("=== CONCLUÍDO: {$turmas->count()} turmas criadas e configuradas ===");
    }
    
    private function criarTurmas($anoLetivo)
    {
        $turmas = collect();
        
        // Turmas do Ensino Primário (1ª a 6ª classe)
        $classesPrimario = ['1ª', '2ª', '3ª', '4ª', '5ª', '6ª'];
        $turmasPrimario = ['A', 'B'];
        
        foreach ($classesPrimario as $classe) {
            foreach ($turmasPrimario as $turma) {
                $turmaCriada = Turma::create([
                    'codigo' => $classe . ' Classe',
                    'nome' => $turma,
                    'nivel_ensino' => 'PRIMARIO',
                    'ano_letivo_id' => $anoLetivo->id,
                    'capacidade_maxima' => 30,
                    'ativa' => true,
                ]);
                $turmas->push($turmaCriada);
                $this->command->info("Turma criada: {$classe} Classe - {$turma}");
            }
        }
        
        // Turmas do Ensino Secundário (7ª a 12ª classe)
        $classesSecundario = ['7ª', '8ª', '9ª', '10ª', '11ª', '12ª'];
        $turmasSecundario = ['A', 'B'];
        
        foreach ($classesSecundario as $classe) {
            foreach ($turmasSecundario as $turma) {
                $turmaCriada = Turma::create([
                    'codigo' => $classe . ' Classe',
                    'nome' => $turma,
                    'nivel_ensino' => 'SECUNDARIO',
                    'ano_letivo_id' => $anoLetivo->id,
                    'capacidade_maxima' => 30,
                    'ativa' => true,
                ]);
                $turmas->push($turmaCriada);
                $this->command->info("Turma criada: {$classe} Classe - {$turma}");
            }
        }
        
        return $turmas;
    }
    
    private function atribuirDisciplinas($turmas, $disciplinasPrimario, $disciplinasSecundario, $anoLetivo)
    {
        foreach ($turmas as $turma) {
            $disciplinasParaTurma = $turma->nivel_ensino === 'PRIMARIO' 
                ? $disciplinasPrimario 
                : $disciplinasSecundario;
            
            foreach ($disciplinasParaTurma as $disciplina) {
                // Verificar se já está atribuída
                $jaAtribuida = DB::table('turma_disciplina')
                    ->where('turma_id', $turma->id)
                    ->where('disciplina_id', $disciplina->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->exists();
                
                if (!$jaAtribuida) {
                    DB::table('turma_disciplina')->insert([
                        'turma_id' => $turma->id,
                        'disciplina_id' => $disciplina->id,
                        'ano_letivo_id' => $anoLetivo->id,
                        'carga_horaria_semanal' => $disciplina->carga_horaria_semanal,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
            
            $this->command->info("Disciplinas atribuídas à {$turma->codigo} - {$turma->nome}: {$disciplinasParaTurma->count()}");
        }
    }
    
    private function alocarAlunos($turmas, $anoLetivo)
    {
        // Buscar todos os alunos
        $alunos = Aluno::with('user')->get();
        
        if ($alunos->isEmpty()) {
            $this->command->warn('Nenhum aluno encontrado para alocar.');
            return;
        }
        
        $this->command->info("Encontrados {$alunos->count()} alunos para alocar.");
        
        // Distribuir alunos de forma equilibrada entre as turmas
        $alunosArray = $alunos->toArray();
        shuffle($alunosArray); // Embaralhar para distribuição aleatória
        
        $turmaIndex = 0;
        $alunosPorTurma = [];
        
        // Calcular quantos alunos por turma (distribuição equilibrada)
        $totalTurmas = $turmas->count();
        $alunosPorTurmaBase = floor($alunos->count() / $totalTurmas);
        $alunosRestantes = $alunos->count() % $totalTurmas;
        
        foreach ($turmas as $turma) {
            $alunosPorTurma[$turma->id] = $alunosPorTurmaBase;
            if ($alunosRestantes > 0) {
                $alunosPorTurma[$turma->id]++;
                $alunosRestantes--;
            }
        }
        
        $alunoIndex = 0;
        foreach ($alunosArray as $alunoData) {
            if ($alunoIndex >= count($alunosArray)) break;
            
            // Selecionar turma de forma rotativa
            $turma = $turmas[$turmaIndex % $turmas->count()];
            
            // Verificar se a turma já atingiu sua cota
            $alunosNaTurma = DB::table('turma_aluno')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->count();
            
            if ($alunosNaTurma >= $alunosPorTurma[$turma->id] || $alunosNaTurma >= ($turma->capacidade_maxima ?? 30)) {
                $turmaIndex++;
                $turma = $turmas[$turmaIndex % $turmas->count()];
            }
            
            $aluno = Aluno::find($alunoData['id']);
            if ($aluno) {
                // Verificar se já está alocado
                $jaAlocado = DB::table('turma_aluno')
                    ->where('aluno_id', $aluno->id)
                    ->where('turma_id', $turma->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->exists();
                
                if (!$jaAlocado) {
                    $aluno->turmas()->attach($turma->id, ['ano_letivo_id' => $anoLetivo->id]);
                }
            }
            
            $turmaIndex++;
            $alunoIndex++;
        }
        
        // Mostrar resumo
        foreach ($turmas as $turma) {
            $totalAlunos = DB::table('turma_aluno')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->count();
            $this->command->info("{$turma->codigo} - {$turma->nome}: {$totalAlunos} alunos");
        }
    }
    
    private function alocarProfessores($turmas, $anoLetivo)
    {
        // Buscar todos os professores
        $professores = Professor::with('user')->get();
        
        if ($professores->isEmpty()) {
            $this->command->warn('Nenhum professor encontrado para alocar.');
            return;
        }
        
        $this->command->info("Encontrados {$professores->count()} professores para alocar.");
        
        foreach ($turmas as $turma) {
            // Buscar disciplinas atribuídas à turma
            $disciplinasTurma = DB::table('turma_disciplina')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->pluck('disciplina_id')
                ->toArray();
            
            if (empty($disciplinasTurma)) {
                $this->command->warn("Nenhuma disciplina atribuída à {$turma->codigo} - {$turma->nome}");
                continue;
            }
            
            $professoresAlocados = 0;
            foreach ($disciplinasTurma as $disciplinaId) {
                $disciplina = Disciplina::find($disciplinaId);
                if (!$disciplina) continue;
                
                // Filtrar professores compatíveis com o nível de ensino
                $professoresCompativeis = $professores->filter(function($professor) use ($disciplina) {
                    return $professor->nivel_ensino === 'AMBOS' || 
                           $professor->nivel_ensino === $disciplina->nivel_ensino;
                });
                
                if ($professoresCompativeis->isEmpty()) {
                    $this->command->warn("Nenhum professor compatível para {$disciplina->nome} na {$turma->codigo}");
                    continue;
                }
                
                // Selecionar professor aleatório compatível
                $professor = $professoresCompativeis->random();
                
                // Verificar se já está alocado
                $jaAlocado = DB::table('turma_disciplina_professor')
                    ->where('turma_id', $turma->id)
                    ->where('disciplina_id', $disciplinaId)
                    ->where('professor_id', $professor->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->exists();
                
                if (!$jaAlocado) {
                    DB::table('turma_disciplina_professor')->insert([
                        'turma_id' => $turma->id,
                        'disciplina_id' => $disciplinaId,
                        'professor_id' => $professor->id,
                        'ano_letivo_id' => $anoLetivo->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                    $professoresAlocados++;
                }
            }
            
            $this->command->info("Professores alocados à {$turma->codigo} - {$turma->nome}: {$professoresAlocados}");
        }
    }
}
