<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Financeiro;
use App\Models\Payment;
use App\Models\Invoice;
use App\Models\AnoLetivo;
use Illuminate\Support\Facades\DB;

class MarcarJaneiroComoPagoSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== MARCANDO MENSALIDADE DE JANEIRO COMO PAGA PARA TODOS OS ALUNOS ===');
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivo) {
            $this->command->error('Ano letivo ativo não encontrado!');
            return;
        }
        
        $this->command->info("Usando ano letivo: {$anoLetivo->ano} (ID: {$anoLetivo->id})");
        
        $alunos = \App\Models\Aluno::all();
        $this->command->info("Alunos encontrados: {$alunos->count()}");
        
        // Buscar todas as mensalidades de janeiro de 2026
        $mensalidadesJaneiro = Financeiro::where('ano_letivo_id', $anoLetivo->id)
            ->where('tipo', 'Mensalidade')
            ->whereMonth('data_vencimento', 1)
            ->whereYear('data_vencimento', 2026)
            ->whereIn('status', ['PENDENTE', 'VENCIDO'])
            ->get();
        
        $this->command->info("Mensalidades de janeiro encontradas: {$mensalidadesJaneiro->count()}");
        
        // Se não houver mensalidades de janeiro, criar para todos os alunos
        if ($mensalidadesJaneiro->count() === 0) {
            $this->command->info("Criando mensalidades de janeiro para todos os alunos...");
            
            $valorPadrao = \App\Models\Configuracao::get('valor_mensalidade', 5000);
            
            foreach ($alunos as $aluno) {
                $modalidade = $aluno->modalidade_pagamento ?? 'MENSAL';
                $valorMensal = $aluno->valor_mensalidade_personalizado ?? $valorPadrao;
                
                if ($valorMensal > 0) {
                    // Verificar se já existe mensalidade de janeiro
                    $existe = Financeiro::where('aluno_id', $aluno->id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->where('tipo', 'Mensalidade')
                        ->whereMonth('data_vencimento', 1)
                        ->whereYear('data_vencimento', 2026)
                        ->exists();
                    
                    if (!$existe) {
                        $dataVencimento = \Carbon\Carbon::create(2026, 1, 5);
                        
                        Financeiro::create([
                            'aluno_id' => $aluno->id,
                            'ano_letivo_id' => $anoLetivo->id,
                            'tipo' => 'Mensalidade',
                            'descricao' => "Mensalidade 1/2026",
                            'valor' => $valorMensal,
                            'data_vencimento' => $dataVencimento,
                            'status' => 'PENDENTE',
                        ]);
                    }
                }
            }
            
            // Buscar novamente após criar
            $mensalidadesJaneiro = Financeiro::where('ano_letivo_id', $anoLetivo->id)
                ->where('tipo', 'Mensalidade')
                ->whereMonth('data_vencimento', 1)
                ->whereYear('data_vencimento', 2026)
                ->whereIn('status', ['PENDENTE', 'VENCIDO'])
                ->get();
            
            $this->command->info("Mensalidades de janeiro criadas: {$mensalidadesJaneiro->count()}");
        }
        
        $this->command->info("Encontradas {$mensalidadesJaneiro->count()} mensalidades de janeiro para processar");
        
        $processadas = 0;
        $erros = 0;
        
        DB::beginTransaction();
        try {
            foreach ($mensalidadesJaneiro as $financeiro) {
                try {
                    // Verificar se já está pago
                    if ($financeiro->status === 'PAGO') {
                        continue;
                    }
                    
                    // Marcar como pago
                    $financeiro->status = 'PAGO';
                    $financeiro->data_pagamento = now();
                    $financeiro->valor_pago = $financeiro->valor;
                    $financeiro->save();
                    
                    // Criar registro de pagamento presencial
                    $payment = Payment::create([
                        'aluno_id' => $financeiro->aluno_id,
                        'financeiro_id' => $financeiro->id,
                        'reference' => 'SEED-' . time() . '-' . $financeiro->id,
                        'metodo' => 'OUTRO',
                        'valor' => $financeiro->valor,
                        'status' => 'APROVADO',
                        'data_pagamento' => now(),
                        'observacoes' => 'Pagamento simulado via seeder - Janeiro 2026',
                    ]);
                    
                    // Criar factura
                    $invoice = Invoice::create([
                        'aluno_id' => $financeiro->aluno_id,
                        'payment_id' => $payment->id,
                        'ano_letivo_id' => $financeiro->ano_letivo_id,
                        'numero_factura' => Invoice::gerarNumeroFactura(),
                        'codigo_verificacao' => Invoice::gerarCodigoVerificacao(),
                        'valor' => $financeiro->valor,
                        'status' => 'PAGA',
                        'data_emissao' => now(),
                        'data_vencimento' => $financeiro->data_vencimento,
                        'descricao' => $financeiro->descricao,
                        'itens' => [
                            [
                                'descricao' => $financeiro->descricao,
                                'valor' => $financeiro->valor,
                            ]
                        ],
                    ]);
                    
                    $payment->invoice_id = $invoice->id;
                    $payment->save();
                    
                    $processadas++;
                } catch (\Exception $e) {
                    $erros++;
                    $this->command->warn("Erro ao processar mensalidade ID {$financeiro->id}: " . $e->getMessage());
                }
            }
            
            DB::commit();
            
            $this->command->info("=== CONCLUÍDO ===");
            $this->command->info("Mensalidades processadas: {$processadas}");
            if ($erros > 0) {
                $this->command->warn("Erros encontrados: {$erros}");
            }
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error("Erro ao processar: " . $e->getMessage());
        }
    }
}
