<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\User;
use App\Models\Financeiro;
use App\Models\Payment;
use App\Models\Invoice;
use App\Models\Avaliacao;
use App\Models\Horario;
use App\Models\Evento;
use App\Models\Trabalho;
use App\Models\PlanoTrimestral;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Trimestre;
use Illuminate\Support\Facades\DB;

class ApagarTodosDadosExcetoFuncionariosEAdminsSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== APAGANDO TODOS OS DADOS EXCETO FUNCIONÁRIOS E ADMINS ===');
        
        // Desabilitar verificação de foreign keys temporariamente
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        try {
            // 1. Apagar dados relacionados aos alunos
            $this->command->info('Apagando registros financeiros...');
            Financeiro::query()->delete();
            
            $this->command->info('Apagando pagamentos...');
            Payment::query()->delete();
            
            $this->command->info('Apagando facturas...');
            Invoice::query()->delete();
            
            $this->command->info('Apagando avaliações...');
            Avaliacao::query()->delete();
            
            $this->command->info('Apagando horários...');
            Horario::query()->delete();
            
            $this->command->info('Apagando eventos...');
            Evento::query()->delete();
            
            $this->command->info('Apagando trabalhos...');
            Trabalho::query()->delete();
            
            $this->command->info('Apagando planos trimestrais...');
            PlanoTrimestral::query()->delete();
            
            $this->command->info('Apagando relações turma-aluno...');
            DB::table('turma_aluno')->delete();
            
            $this->command->info('Apagando relações turma-disciplina-professor...');
            DB::table('turma_disciplina_professor')->delete();
            
            $this->command->info('Apagando relações turma-disciplina...');
            DB::table('turma_disciplina')->delete();
            
            // 2. Apagar alunos e seus usuários (apenas se não forem admin/funcionário)
            $this->command->info('Apagando alunos...');
            $alunos = Aluno::all();
            $alunoUserIds = $alunos->pluck('user_id')->filter();
            
            if ($alunos->count() > 0) {
                $this->command->info("Encontrados {$alunos->count()} alunos");
                
                // Apagar apenas usuários que não são admin ou funcionário
                if ($alunoUserIds->count() > 0) {
                    $usuariosParaApagar = User::whereIn('id', $alunoUserIds)
                        ->whereDoesntHave('roles', function($q) {
                            $q->whereIn('name', ['admin', 'superadmin', 'funcionario']);
                        })
                        ->pluck('id');
                    
                    if ($usuariosParaApagar->count() > 0) {
                        $this->command->info("Apagando {$usuariosParaApagar->count()} usuários de alunos...");
                        User::whereIn('id', $usuariosParaApagar)->delete();
                    }
                }
                
                Aluno::query()->delete();
            } else {
                $this->command->info('Nenhum aluno encontrado');
            }
            
            // 3. Apagar professores e seus usuários (apenas se não forem admin/funcionário)
            $this->command->info('Apagando professores...');
            $professores = Professor::all();
            $professorUserIds = $professores->pluck('user_id')->filter();
            
            if ($professores->count() > 0) {
                $this->command->info("Encontrados {$professores->count()} professores");
                
                // Apagar apenas usuários que não são admin ou funcionário
                if ($professorUserIds->count() > 0) {
                    $usuariosParaApagar = User::whereIn('id', $professorUserIds)
                        ->whereDoesntHave('roles', function($q) {
                            $q->whereIn('name', ['admin', 'superadmin', 'funcionario']);
                        })
                        ->pluck('id');
                    
                    if ($usuariosParaApagar->count() > 0) {
                        $this->command->info("Apagando {$usuariosParaApagar->count()} usuários de professores...");
                        User::whereIn('id', $usuariosParaApagar)->delete();
                    }
                }
                
                Professor::query()->delete();
            } else {
                $this->command->info('Nenhum professor encontrado');
            }
            
            // 4. Apagar turmas
            $this->command->info('Apagando turmas...');
            $turmasCount = Turma::count();
            if ($turmasCount > 0) {
                $this->command->info("Encontradas {$turmasCount} turmas");
                Turma::query()->delete();
            } else {
                $this->command->info('Nenhuma turma encontrada');
            }
            
            // 5. Apagar disciplinas
            $this->command->info('Apagando disciplinas...');
            $disciplinasCount = Disciplina::count();
            if ($disciplinasCount > 0) {
                $this->command->info("Encontradas {$disciplinasCount} disciplinas");
                Disciplina::query()->delete();
            } else {
                $this->command->info('Nenhuma disciplina encontrada');
            }
            
            // 6. Apagar trimestres
            $this->command->info('Apagando trimestres...');
            $trimestresCount = Trimestre::count();
            if ($trimestresCount > 0) {
                $this->command->info("Encontrados {$trimestresCount} trimestres");
                Trimestre::query()->delete();
            } else {
                $this->command->info('Nenhum trimestre encontrado');
            }
            
            // Reabilitar verificação de foreign keys
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            
            $this->command->info('=== CONCLUÍDO ===');
            $this->command->info('Todos os dados foram apagados, exceto funcionários e admins!');
            
            // Mostrar resumo do que foi mantido
            $funcionariosCount = \App\Models\Funcionario::count();
            $adminsCount = User::whereHas('roles', function($q) {
                $q->whereIn('name', ['admin', 'superadmin']);
            })->count();
            
            $this->command->info("Funcionários mantidos: {$funcionariosCount}");
            $this->command->info("Admins mantidos: {$adminsCount}");
            
        } catch (\Exception $e) {
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            $this->command->error('Erro ao apagar: ' . $e->getMessage());
            throw $e;
        }
    }
}
