<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Índices para tabela 'alunos'
        Schema::table('alunos', function (Blueprint $table) {
            // Índice para busca por código do estudante
            if (!$this->hasIndex('alunos', 'idx_alunos_codigo_estudante')) {
                $table->index('codigo_estudante', 'idx_alunos_codigo_estudante');
            }
            // Índice para relacionamento com users
            if (!$this->hasIndex('alunos', 'idx_alunos_user_id')) {
                $table->index('user_id', 'idx_alunos_user_id');
            }
        });

        // Índices para tabela 'turmas'
        Schema::table('turmas', function (Blueprint $table) {
            // Índice composto para filtros comuns
            if (!$this->hasIndex('turmas', 'idx_turmas_ano_letivo_ativa')) {
                $table->index(['ano_letivo_id', 'ativa'], 'idx_turmas_ano_letivo_ativa');
            }
            // Índice para código
            if (!$this->hasIndex('turmas', 'idx_turmas_codigo')) {
                $table->index('codigo', 'idx_turmas_codigo');
            }
            // Índice para nome (usado em filtros)
            if (!$this->hasIndex('turmas', 'idx_turmas_nome')) {
                $table->index('nome', 'idx_turmas_nome');
            }
            // Índice para nivel_ensino
            if (!$this->hasIndex('turmas', 'idx_turmas_nivel_ensino')) {
                $table->index('nivel_ensino', 'idx_turmas_nivel_ensino');
            }
        });

        // Índices para tabela 'turma_aluno' (pivot)
        Schema::table('turma_aluno', function (Blueprint $table) {
            // Índice composto para queries comuns
            if (!$this->hasIndex('turma_aluno', 'idx_turma_aluno_ano_letivo')) {
                $table->index(['aluno_id', 'ano_letivo_id'], 'idx_turma_aluno_ano_letivo');
            }
            if (!$this->hasIndex('turma_aluno', 'idx_turma_aluno_turma_ano')) {
                $table->index(['turma_id', 'ano_letivo_id'], 'idx_turma_aluno_turma_ano');
            }
        });

        // Índices para tabela 'disciplinas'
        Schema::table('disciplinas', function (Blueprint $table) {
            // Índice para nivel_ensino
            if (!$this->hasIndex('disciplinas', 'idx_disciplinas_nivel_ensino')) {
                $table->index('nivel_ensino', 'idx_disciplinas_nivel_ensino');
            }
            // Índice para nome (usado em ordenação)
            if (!$this->hasIndex('disciplinas', 'idx_disciplinas_nome')) {
                $table->index('nome', 'idx_disciplinas_nome');
            }
        });

        // Índices para tabela 'professores'
        Schema::table('professores', function (Blueprint $table) {
            // Índice para user_id
            if (!$this->hasIndex('professores', 'idx_professores_user_id')) {
                $table->index('user_id', 'idx_professores_user_id');
            }
            // Índice para nivel_ensino
            if (!$this->hasIndex('professores', 'idx_professores_nivel_ensino')) {
                $table->index('nivel_ensino', 'idx_professores_nivel_ensino');
            }
        });

        // Índices para tabela 'financeiro'
        Schema::table('financeiro', function (Blueprint $table) {
            // Índice composto para queries comuns
            if (!$this->hasIndex('financeiro', 'idx_financeiro_aluno_ano_tipo')) {
                $table->index(['aluno_id', 'ano_letivo_id', 'tipo'], 'idx_financeiro_aluno_ano_tipo');
            }
            // Índice para status (usado em filtros)
            if (!$this->hasIndex('financeiro', 'idx_financeiro_status')) {
                $table->index('status', 'idx_financeiro_status');
            }
            // Índice para data_vencimento (usado em ordenação e filtros)
            if (!$this->hasIndex('financeiro', 'idx_financeiro_data_vencimento')) {
                $table->index('data_vencimento', 'idx_financeiro_data_vencimento');
            }
        });

        // Índices para tabela 'users'
        Schema::table('users', function (Blueprint $table) {
            // Índice para name (usado em busca)
            if (!$this->hasIndex('users', 'idx_users_name')) {
                $table->index('name', 'idx_users_name');
            }
        });

        // Índices para tabela 'avaliacoes'
        Schema::table('avaliacoes', function (Blueprint $table) {
            // Índice composto para queries comuns
            if (!$this->hasIndex('avaliacoes', 'idx_avaliacoes_turma_disciplina_ano')) {
                $table->index(['turma_id', 'disciplina_id', 'ano_letivo_id'], 'idx_avaliacoes_turma_disciplina_ano');
            }
            // Índice para aluno_id
            if (!$this->hasIndex('avaliacoes', 'idx_avaliacoes_aluno_id')) {
                $table->index('aluno_id', 'idx_avaliacoes_aluno_id');
            }
        });

        // Índices para tabela 'horarios'
        Schema::table('horarios', function (Blueprint $table) {
            // Índice composto para queries comuns
            if (!$this->hasIndex('horarios', 'idx_horarios_turma_ano')) {
                $table->index(['turma_id', 'ano_letivo_id'], 'idx_horarios_turma_ano');
            }
        });

        // Índices para tabela 'eventos'
        Schema::table('eventos', function (Blueprint $table) {
            // Índice composto para queries comuns
            if (!$this->hasIndex('eventos', 'idx_eventos_ano_letivo')) {
                $table->index('ano_letivo_id', 'idx_eventos_ano_letivo');
            }
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('alunos', function (Blueprint $table) {
            $table->dropIndex('idx_alunos_codigo_estudante');
            $table->dropIndex('idx_alunos_user_id');
        });

        Schema::table('turmas', function (Blueprint $table) {
            $table->dropIndex('idx_turmas_ano_letivo_ativa');
            $table->dropIndex('idx_turmas_codigo');
            $table->dropIndex('idx_turmas_nome');
            $table->dropIndex('idx_turmas_nivel_ensino');
        });

        Schema::table('turma_aluno', function (Blueprint $table) {
            $table->dropIndex('idx_turma_aluno_ano_letivo');
            $table->dropIndex('idx_turma_aluno_turma_ano');
        });

        Schema::table('disciplinas', function (Blueprint $table) {
            $table->dropIndex('idx_disciplinas_nivel_ensino');
            $table->dropIndex('idx_disciplinas_nome');
        });

        Schema::table('professores', function (Blueprint $table) {
            $table->dropIndex('idx_professores_user_id');
            $table->dropIndex('idx_professores_nivel_ensino');
        });

        Schema::table('financeiro', function (Blueprint $table) {
            $table->dropIndex('idx_financeiro_aluno_ano_tipo');
            $table->dropIndex('idx_financeiro_status');
            $table->dropIndex('idx_financeiro_data_vencimento');
        });

        Schema::table('users', function (Blueprint $table) {
            $table->dropIndex('idx_users_name');
        });

        Schema::table('avaliacoes', function (Blueprint $table) {
            $table->dropIndex('idx_avaliacoes_turma_disciplina_ano');
            $table->dropIndex('idx_avaliacoes_aluno_id');
        });

        Schema::table('horarios', function (Blueprint $table) {
            $table->dropIndex('idx_horarios_turma_ano');
        });

        Schema::table('eventos', function (Blueprint $table) {
            $table->dropIndex('idx_eventos_ano_letivo');
        });
    }

    /**
     * Verifica se um índice já existe
     */
    private function hasIndex(string $table, string $indexName): bool
    {
        $connection = Schema::getConnection();
        $databaseName = $connection->getDatabaseName();
        
        $result = $connection->select(
            "SELECT COUNT(*) as count 
             FROM information_schema.statistics 
             WHERE table_schema = ? 
             AND table_name = ? 
             AND index_name = ?",
            [$databaseName, $table, $indexName]
        );
        
        return $result[0]->count > 0;
    }
};
