<?php

namespace App\Services;

use Google_Client;
use Google_Service_Drive;
use Google_Service_Drive_DriveFile;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;

class GoogleDriveService
{
    private $client;
    private $service;
    private $folderId;

    public function __construct()
    {
        $this->folderId = \App\Models\Configuracao::get('google_drive_folder_id', null);
        $this->initializeClient();
    }

    private function initializeClient()
    {
        try {
            if (!class_exists('Google_Client')) {
                throw new \Exception('Biblioteca Google API Client não instalada. Execute: composer require google/apiclient');
            }

            $this->client = new Google_Client();
            $this->client->setApplicationName('SGE - Sistema de Gestão Escolar');
            $this->client->setScopes(Google_Service_Drive::DRIVE_FILE);
            $this->client->setAccessType('offline');
            $this->client->setPrompt('select_account consent');

            // Credenciais do arquivo JSON
            $credentialsPath = storage_path('app/google-credentials.json');
            if (File::exists($credentialsPath)) {
                $this->client->setAuthConfig($credentialsPath);
            } else {
                // Tentar usar credenciais da configuração
                $clientId = \App\Models\Configuracao::get('google_drive_client_id', null);
                $clientSecret = \App\Models\Configuracao::get('google_drive_client_secret', null);
                $refreshToken = \App\Models\Configuracao::get('google_drive_refresh_token', null);
                $accessToken = \App\Models\Configuracao::get('google_drive_access_token', null);

                if ($clientId && $clientSecret) {
                    $this->client->setClientId($clientId);
                    $this->client->setClientSecret($clientSecret);
                    $this->client->setRedirectUri(route('superadmin.backups.google-drive.callback'));
                    
                    // Se tiver access token, usar diretamente
                    if ($accessToken) {
                        $this->client->setAccessToken(['access_token' => $accessToken]);
                    } elseif ($refreshToken) {
                        // Tentar renovar com refresh token
                        try {
                            $this->client->refreshToken($refreshToken);
                        } catch (\Exception $e) {
                            // Se refresh token expirou, precisa reautenticar
                            Log::warning('Refresh token inválido, precisa reautenticar: ' . $e->getMessage());
                        }
                    }
                } else {
                    throw new \Exception('Credenciais do Google Drive não configuradas');
                }
            }

            $this->service = new Google_Service_Drive($this->client);
        } catch (\Exception $e) {
            Log::error('Erro ao inicializar Google Drive: ' . $e->getMessage());
            throw $e;
        }
    }

    public function uploadFile($filePath, $fileName = null)
    {
        try {
            if (!File::exists($filePath)) {
                throw new \Exception("Arquivo não encontrado: {$filePath}");
            }

            $fileName = $fileName ?: basename($filePath);
            $fileSize = File::size($filePath);
            $mimeType = mime_content_type($filePath) ?: 'application/octet-stream';

            $fileMetadata = new Google_Service_Drive_DriveFile([
                'name' => $fileName,
            ]);

            // Se tiver pasta configurada, definir como pai
            if ($this->folderId) {
                $fileMetadata->setParents([$this->folderId]);
            }

            $content = File::get($filePath);

            // Upload em chunks para arquivos grandes (> 5MB)
            if ($fileSize > 5 * 1024 * 1024) {
                $this->client->setDefer(true);
                $request = $this->service->files->create($fileMetadata, [
                    'fields' => 'id',
                    'uploadType' => 'resumable'
                ]);

                $chunkSizeBytes = 1 * 1024 * 1024; // 1MB chunks
                $media = new \Google_Http_MediaFileUpload(
                    $this->client,
                    $request,
                    $mimeType,
                    null,
                    true,
                    $chunkSizeBytes
                );
                $media->setFileSize($fileSize);

                $status = false;
                $handle = fopen($filePath, "rb");
                while (!$status && !feof($handle)) {
                    $chunk = fread($handle, $chunkSizeBytes);
                    $status = $media->nextChunk($chunk);
                }
                fclose($handle);
                $this->client->setDefer(false);

                $fileId = $status->id;
                $file = $status;
            } else {
                // Upload direto para arquivos pequenos
                $file = $this->service->files->create($fileMetadata, [
                    'data' => $content,
                    'mimeType' => $mimeType,
                    'uploadType' => 'multipart',
                    'fields' => 'id, name, size, webViewLink'
                ]);
                $fileId = $file->id;
            }

            Log::info('Arquivo enviado para Google Drive', [
                'arquivo' => $fileName,
                'file_id' => $fileId,
                'tamanho' => $fileSize
            ]);

            return [
                'success' => true,
                'file_id' => $fileId,
                'file_name' => $fileName,
                'web_view_link' => $file->webViewLink ?? null
            ];

        } catch (\Exception $e) {
            Log::error('Erro ao fazer upload para Google Drive: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function isConfigured()
    {
        $credentialsPath = storage_path('app/google-credentials.json');
        $hasCredentialsFile = File::exists($credentialsPath);
        
        $clientId = \App\Models\Configuracao::get('google_drive_client_id', null);
        $clientSecret = \App\Models\Configuracao::get('google_drive_client_secret', null);
        $hasCredentialsConfig = $clientId && $clientSecret;

        return $hasCredentialsFile || $hasCredentialsConfig;
    }

    public function getAuthUrl()
    {
        try {
            $this->initializeClient();
            return $this->client->createAuthUrl();
        } catch (\Exception $e) {
            Log::error('Erro ao gerar URL de autenticação: ' . $e->getMessage());
            return null;
        }
    }

    public function handleCallback($code)
    {
        try {
            $this->initializeClient();
            $accessToken = $this->client->fetchAccessTokenWithAuthCode($code);
            
            if (array_key_exists('error', $accessToken)) {
                throw new \Exception($accessToken['error_description'] ?? 'Erro ao obter token');
            }

            $this->client->setAccessToken($accessToken);

            // Salvar refresh token (importante para renovação automática)
            if (isset($accessToken['refresh_token'])) {
                \App\Models\Configuracao::set('google_drive_refresh_token', $accessToken['refresh_token'], 'string', 'Refresh token do Google Drive');
            }

            // Salvar access token
            if (isset($accessToken['access_token'])) {
                \App\Models\Configuracao::set('google_drive_access_token', $accessToken['access_token'], 'string', 'Access token do Google Drive');
            }

            // Salvar token completo (para renovação)
            if (isset($accessToken['expires_in'])) {
                $accessToken['created'] = time();
                \App\Models\Configuracao::set('google_drive_token_data', json_encode($accessToken), 'json', 'Dados completos do token do Google Drive');
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Erro ao processar callback: ' . $e->getMessage());
            return false;
        }
    }

    public function listFolders()
    {
        try {
            $this->initializeClient();
            $response = $this->service->files->listFiles([
                'q' => "mimeType='application/vnd.google-apps.folder' and trashed=false",
                'fields' => 'files(id, name)',
                'pageSize' => 100
            ]);

            return $response->getFiles();
        } catch (\Exception $e) {
            Log::error('Erro ao listar pastas: ' . $e->getMessage());
            return [];
        }
    }
}
