<?php

namespace App\Services;

use App\Models\Aluno;
use App\Models\Financeiro;
use App\Models\AnoLetivo;

class FinanceiroService
{
    /**
     * Obtém o valor padrão da mensalidade baseado na classe do aluno
     * 
     * Valores por classe:
     * - 1ª - 3ª Classe: 4.320,00 MT
     * - 4ª - 6ª Classe: 4.725,00 MT
     * - 7ª - 9ª Classe: 5.750,00 MT
     * - 10ª - 12ª Classe: 5.225,00 MT
     */
    public function getValorMensalidadePorClasse($aluno, $anoLetivoId = null): float
    {
        // Se o aluno tem valor personalizado, usar esse
        if ($aluno->valor_mensalidade_personalizado && $aluno->valor_mensalidade_personalizado > 0) {
            return $aluno->valor_mensalidade_personalizado;
        }
        
        // Buscar turma do aluno no ano letivo
        if (!$anoLetivoId) {
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();
            $anoLetivoId = $anoLetivo ? $anoLetivo->id : null;
        }
        
        $turma = null;
        $numeroClasse = null;
        
        // Tentar buscar turma no ano letivo especificado
        if ($anoLetivoId) {
            $turma = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivoId)
                ->first();
        }
        
        // Se não encontrou turma no ano letivo especificado, tentar em outros anos letivos
        if (!$turma) {
            $turma = $aluno->turmas()
                ->orderBy('turma_aluno.ano_letivo_id', 'desc')
                ->first();
        }
        
        // Se encontrou turma, extrair número da classe
        if ($turma) {
            $numeroClasse = $this->extrairNumeroClasse($turma);
        }
        
        // Se ainda não conseguiu determinar a classe, tentar pela última mensalidade paga
        if (!$numeroClasse || $numeroClasse < 1 || $numeroClasse > 12) {
            $ultimaMensalidade = Financeiro::where('aluno_id', $aluno->id)
                ->where('tipo', 'Mensalidade')
                ->where('status', 'PAGO')
                ->orderBy('data_pagamento', 'desc')
                ->first();
            
            if ($ultimaMensalidade && $ultimaMensalidade->valor > 0) {
                // Tentar determinar a classe pelo valor da última mensalidade paga
                $valor = $ultimaMensalidade->valor;
                if ($valor >= 5225 && $valor <= 6200) {
                    // Provavelmente 10ª-12ª classe
                    $numeroClasse = 10;
                } elseif ($valor >= 5750 && $valor <= 6000) {
                    // Provavelmente 7ª-9ª classe
                    $numeroClasse = 7;
                } elseif ($valor >= 4725 && $valor <= 5000) {
                    // Provavelmente 4ª-6ª classe
                    $numeroClasse = 4;
                } elseif ($valor >= 4320 && $valor <= 4500) {
                    // Provavelmente 1ª-3ª classe
                    $numeroClasse = 1;
                }
            }
        }
        
        // Determinar valor baseado na classe encontrada
        if ($numeroClasse && $numeroClasse >= 1 && $numeroClasse <= 12) {
            if ($numeroClasse >= 1 && $numeroClasse <= 3) {
                return 4320.00;
            } elseif ($numeroClasse >= 4 && $numeroClasse <= 6) {
                return 4725.00;
            } elseif ($numeroClasse >= 7 && $numeroClasse <= 9) {
                return 5750.00;
            } elseif ($numeroClasse >= 10 && $numeroClasse <= 12) {
                return 5225.00;
            }
        }
        
        // Se não conseguiu determinar a classe de forma alguma, usar valor médio
        // (média dos valores: 4320 + 4725 + 5750 + 5225) / 4 = 5005
        return 5005.00;
    }
    
    /**
     * Extrai o número da classe do nome ou código da turma
     */
    private function extrairNumeroClasse($turma): int
    {
        // Tentar extrair do nome (ex: "1ª Classe-A" -> 1, "10ª Classe-A" -> 10)
        if (preg_match('/(\d+)ª/', $turma->nome, $matches)) {
            return (int) $matches[1];
        }
        
        // Tentar extrair do código (ex: "1A" -> 1, "10A" -> 10)
        if (preg_match('/^(\d+)/', $turma->codigo, $matches)) {
            return (int) $matches[1];
        }
        
        // Se não conseguir, retornar 1 como padrão
        return 1;
    }
    
    /**
     * Obtém o valor da mensalidade baseado no número da classe
     */
    public function getValorMensalidadePorNumeroClasse(int $numeroClasse): float
    {
        if ($numeroClasse >= 1 && $numeroClasse <= 3) {
            return 4320.00;
        } elseif ($numeroClasse >= 4 && $numeroClasse <= 6) {
            return 4725.00;
        } elseif ($numeroClasse >= 7 && $numeroClasse <= 9) {
            return 5750.00;
        } elseif ($numeroClasse >= 10 && $numeroClasse <= 12) {
            return 5225.00;
        }
        
        // Valor padrão se não conseguir determinar
        return 4320.00;
    }
    
    /**
     * Obtém o valor da mensalidade baseado na turma
     */
    public function getValorMensalidadePorTurma($turma): float
    {
        $numeroClasse = $this->extrairNumeroClasse($turma);
        return $this->getValorMensalidadePorNumeroClasse($numeroClasse);
    }
    
    /**
     * Obtém o valor da taxa de inscrição
     */
    public function getValorTaxaInscricao(): float
    {
        return 2500.00;
    }

    /**
     * Obtém a situação financeira do aluno
     * Considera apenas mensalidades vencidas há mais de 5 dias (a partir do dia 6)
     */
    public function getSituacaoFinanceira(Aluno $aluno, $anoLetivoId = null): string
    {
        if (!$anoLetivoId) {
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();
            if (!$anoLetivo) {
                return 'REGULAR';
            }
            $anoLetivoId = $anoLetivo->id;
        }

        // Data limite: 5 dias atrás (após o período de graça)
        $dataLimite = now()->subDays(5);

        // IMPORTANTE: Considerar apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
        $dividas = Financeiro::where('aluno_id', $aluno->id)
            ->where('ano_letivo_id', $anoLetivoId)
            ->where('tipo', 'Mensalidade') // Apenas mensalidades
            ->whereIn('status', ['PENDENTE', 'VENCIDO']) // Apenas não pagas
            ->where('data_vencimento', '<=', $dataLimite) // Apenas vencidas há mais de 4 dias (a partir do dia 6)
            ->sum('valor');

        return $dividas > 0 ? 'EM_DÍVIDA' : 'REGULAR';
    }

    /**
     * Atualiza status de itens financeiros vencidos
     * Considera vencido apenas após 4 dias da data de vencimento (dia 6+)
     * Última data sem multa: dia 5. A partir do dia 6 começa a multa.
     */
    public function atualizarVencidos(): void
    {
        // Data limite: 4 dias após a data de vencimento (dia 5 é última data sem multa, dia 6+ começa multa)
        $dataLimite = now()->subDays(4);
        
        Financeiro::where('status', 'PENDENTE')
            ->where('data_vencimento', '<=', $dataLimite)
            ->update(['status' => 'VENCIDO']);
    }

    /**
     * Registra um pagamento
     */
    public function registrarPagamento(Financeiro $financeiro, float $valor, $dataPagamento = null): Financeiro
    {
        $financeiro->status = 'PAGO';
        $financeiro->valor_pago = $valor;
        $financeiro->data_pagamento = $dataPagamento ?? now();
        $financeiro->save();

        return $financeiro;
    }

    /**
     * Calcula multa baseada no atraso de pagamento
     * Período de graça: até dia 5 (não há multa)
     * Dia 6-15: 10% do valor
     * Dia 16-30: 15% do valor
     * Dia 31+: 25% do valor
     */
    public function calcularMulta(Financeiro $financeiro): float
    {
        if ($financeiro->status === 'PAGO') {
            return 0;
        }

        // Calcular dias de atraso (diferença entre hoje e data de vencimento)
        $hoje = now()->startOfDay();
        $vencimento = \Carbon\Carbon::parse($financeiro->data_vencimento)->startOfDay();
        
        // Se a data de vencimento é no futuro, não há multa
        if ($vencimento >= $hoje) {
            return 0;
        }
        
        $diasAtraso = $hoje->diffInDays($vencimento);
        
        // Período de graça: até dia 5 (última data sem multa)
        // Se vence no dia 1, pode pagar até dia 5 sem multa
        // A partir do dia 6 (5 dias de atraso) começa a multa
        if ($diasAtraso <= 4) {
            return 0;
        }
        
        // A partir do dia 6 (5 dias de atraso), calcular multa
        $valorBase = $financeiro->valor;
        $percentualMulta = 0;

        // Dia 6-15 (5-15 dias de atraso): 10%
        if ($diasAtraso >= 5 && $diasAtraso <= 15) {
            $percentualMulta = 0.10;
        } 
        // Dia 16-30 (16-30 dias de atraso): 15%
        elseif ($diasAtraso >= 16 && $diasAtraso <= 30) {
            $percentualMulta = 0.15;
        } 
        // Dia 31+ (31+ dias de atraso): 25%
        elseif ($diasAtraso > 30) {
            $percentualMulta = 0.25;
        }

        return round($valorBase * $percentualMulta, 2);
    }

    /**
     * Obtém o valor total com multa
     */
    public function getValorComMulta(Financeiro $financeiro): float
    {
        $multa = $this->calcularMulta($financeiro);
        return $financeiro->valor + $multa;
    }

    /**
     * Verifica se o aluno está incapacitado (3 ou mais pagamentos em falta)
     * Considera apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
     * Última data sem multa: dia 5. A partir do dia 6 começa a multa.
     */
    public function isIncapacitado(Aluno $aluno, $anoLetivoId = null): bool
    {
        if (!$anoLetivoId) {
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();
            if (!$anoLetivo) {
                return false;
            }
            $anoLetivoId = $anoLetivo->id;
        }

        // Data limite: 4 dias atrás (após o período de graça - dia 5 é última data sem multa)
        $dataLimite = now()->subDays(4);

        $pagamentosEmFalta = Financeiro::where('aluno_id', $aluno->id)
            ->where('ano_letivo_id', $anoLetivoId)
            ->where('tipo', 'Mensalidade')
            ->whereIn('status', ['PENDENTE', 'VENCIDO'])
            ->where('data_vencimento', '<=', $dataLimite) // Apenas vencidas há mais de 4 dias (a partir do dia 6)
            ->count();

        return $pagamentosEmFalta >= 3;
    }
}



