<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Invoice extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'aluno_id',
        'payment_id',
        'ano_letivo_id',
        'numero_factura',
        'codigo_verificacao',
        'valor',
        'status',
        'data_emissao',
        'data_vencimento',
        'descricao',
        'itens',
        'caminho_pdf',
    ];

    protected $casts = [
        'valor' => 'decimal:2',
        'itens' => 'array',
        'data_emissao' => 'date',
        'data_vencimento' => 'date',
    ];

    /**
     * Boot do modelo
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($invoice) {
            if (empty($invoice->codigo_verificacao)) {
                $invoice->codigo_verificacao = self::gerarCodigoVerificacao();
            }
        });
    }

    public function aluno()
    {
        return $this->belongsTo(Aluno::class);
    }

    public function payment()
    {
        return $this->belongsTo(Payment::class);
    }

    public function anoLetivo()
    {
        return $this->belongsTo(AnoLetivo::class);
    }

    /**
     * Gera um número único de factura
     */
    public static function gerarNumeroFactura(): string
    {
        $ano = date('Y');
        $ultimaFactura = self::where('numero_factura', 'like', "FACT{$ano}%")
            ->orderBy('numero_factura', 'desc')
            ->first();

        if ($ultimaFactura) {
            $ultimoNumero = (int) substr($ultimaFactura->numero_factura, -6);
            $novoNumero = $ultimoNumero + 1;
        } else {
            $novoNumero = 1;
        }

        return 'FACT' . $ano . str_pad($novoNumero, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Gera um código único de verificação
     */
    public static function gerarCodigoVerificacao(): string
    {
        do {
            $codigo = strtoupper(substr(md5(uniqid(rand(), true)), 0, 8));
        } while (self::where('codigo_verificacao', $codigo)->exists());

        return $codigo;
    }

    /**
     * Verifica se a factura está paga
     */
    public function isPaga(): bool
    {
        return $this->status === 'PAGA';
    }
}
