<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Disciplina extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'disciplinas';

    protected $fillable = [
        'nome',
        'codigo',
        'descricao',
        'modelo_avaliacao',
        'nivel_ensino',
        'carga_horaria_semanal',
        'ativa',
    ];

    protected $casts = [
        'ativa' => 'boolean',
        'carga_horaria_semanal' => 'integer',
    ];

    /**
     * Retorna a carga horária formatada (ex: 2h 30min)
     */
    public function getCargaHorariaFormatadaAttribute(): string
    {
        if (!$this->carga_horaria_semanal) {
            return '-';
        }
        
        $horas = floor($this->carga_horaria_semanal / 60);
        $minutos = $this->carga_horaria_semanal % 60;
        
        if ($horas > 0 && $minutos > 0) {
            return "{$horas}h {$minutos}min";
        } elseif ($horas > 0) {
            return "{$horas}h";
        } else {
            return "{$minutos}min";
        }
    }

    /**
     * Calcula o total de minutos alocados nos horários
     */
    public function getMinutosAlocadosAttribute(): int
    {
        return $this->horarios->sum(function ($horario) {
            $inicio = \Carbon\Carbon::parse($horario->hora_inicio);
            $fim = \Carbon\Carbon::parse($horario->hora_fim);
            return $fim->diffInMinutes($inicio);
        });
    }

    /**
     * Verifica se a carga horária está completa
     */
    public function cargaHorariaCompleta(): bool
    {
        if (!$this->carga_horaria_semanal) {
            return true; // Se não tem carga definida, considera completa
        }
        
        return $this->minutos_alocados >= $this->carga_horaria_semanal;
    }

    public function turmas()
    {
        return $this->belongsToMany(Turma::class, 'turma_disciplina_professor')
            ->withPivot('professor_id', 'ano_letivo_id')
            ->withTimestamps();
    }

    public function professores()
    {
        return $this->belongsToMany(Professor::class, 'turma_disciplina_professor')
            ->withPivot('turma_id', 'ano_letivo_id')
            ->withTimestamps();
    }

    public function avaliacoes()
    {
        return $this->hasMany(Avaliacao::class);
    }

    public function trabalhos()
    {
        return $this->hasMany(Trabalho::class);
    }

    public function horarios()
    {
        return $this->hasMany(Horario::class);
    }

    // Cambridge removido
}



