<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AnoLetivo extends Model
{
    use HasFactory;

    protected $table = 'ano_letivo';

    protected $fillable = [
        'ano',
        'data_inicio',
        'data_fim',
        'ativo',
    ];

    protected $casts = [
        'data_inicio' => 'date',
        'data_fim' => 'date',
        'ativo' => 'boolean',
    ];

    public function trimestres()
    {
        return $this->hasMany(Trimestre::class);
    }

    public function turmas()
    {
        return $this->hasMany(Turma::class);
    }

    public function eventos()
    {
        return $this->hasMany(Evento::class);
    }

    /**
     * Retorna apenas o ano inicial (ex: 2025/2026 -> 2025)
     */
    public function getAnoInicialAttribute()
    {
        if (strpos($this->ano, '/') !== false) {
            return explode('/', $this->ano)[0];
        }
        return $this->ano;
    }

    /**
     * Retorna o ano letivo atual (sempre 2026)
     * Este é o ano letivo usado para operações normais do sistema
     */
    public static function getAnoLetivoAtual()
    {
        // Sempre buscar 2026 primeiro
        $ano2026 = static::where(function($q) {
            $q->where('ano', 'like', '%2026%')
              ->orWhere('ano', '2026');
        })->first();
        
        if ($ano2026) {
            return $ano2026;
        }
        
        // Se não encontrar 2026, criar
        return static::firstOrCreate(
            ['ano' => '2026'],
            [
                'data_inicio' => '2026-01-01',
                'data_fim' => '2026-12-31',
                'ativo' => true,
            ]
        );
    }

    /**
     * Retorna apenas anos letivos disponíveis para uso no sistema
     * Por padrão mostra apenas 2026, mas permite outros anos se existirem
     * (exclui 2025 e anteriores, exceto se adicionados manualmente)
     */
    public static function getAnosLetivosDisponiveis()
    {
        return static::where(function($q) {
            $q->where('ano', 'like', '%2026%')
              ->orWhere('ano', '2026');
        })->orderBy('ano', 'desc')->get();
    }

    /**
     * Retorna anos letivos disponíveis para novos ingressos e renovações
     * - 2026 sempre disponível
     * - 2027 disponível a partir de outubro (mês 10)
     */
    public static function getAnosLetivosParaMatricula()
    {
        $mesAtual = (int) date('m');
        
        $query = static::query();
        
        // Sempre incluir 2026
        $query->where(function($q) {
            $q->where('ano', 'like', '%2026%')
              ->orWhere('ano', '2026');
        });
        
        // Se estamos em outubro ou depois, incluir 2027
        if ($mesAtual >= 10) {
            $query->orWhere(function($q) {
                $q->where('ano', 'like', '%2027%')
                  ->orWhere('ano', '2027');
            });
        }
        
        return $query->orderBy('ano', 'desc')->get();
    }

    /**
     * Verifica se o ano letivo está disponível para matrícula
     * 2026 sempre disponível
     * 2027 disponível a partir de outubro
     */
    public function isDisponivelParaMatricula()
    {
        $anoRaw = $this->getRawOriginal('ano');
        $mesAtual = (int) date('m');
        
        // 2026 sempre disponível
        if (strpos($anoRaw, '2026') !== false || $anoRaw === '2026') {
            return true;
        }
        
        // 2027 disponível a partir de outubro
        if ((strpos($anoRaw, '2027') !== false || $anoRaw === '2027') && $mesAtual >= 10) {
            return true;
        }
        
        return false;
    }

    /**
     * Accessor para exibir apenas o ano inicial quando acessado como 'ano'
     */
    public function getAnoAttribute($value)
    {
        // Se não há valor, retorna null
        if ($value === null) {
            return null;
        }
        
        // Se o valor contém '/', retorna apenas o primeiro ano
        if (strpos($value, '/') !== false) {
            return explode('/', $value)[0];
        }
        
        return $value;
    }
}



