<?php

namespace App\Http\Controllers;

use App\Models\Trabalho;
use App\Models\PlanoTrimestral;
use App\Models\Disciplina;
use App\Models\Turma;
use App\Models\AnoLetivo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class TrabalhoController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        $user = auth()->user();
        $anoLetivo = AnoLetivo::where('ativo', true)->first();

        // Admin, Superadmin e funcionários com permissão veem os planos trimestrais dos professores
        if ($user->isAdmin() || $user->isSuperadmin() || $user->podeGerir('gerir_planos')) {
            $planos = PlanoTrimestral::where('ano_letivo_id', $anoLetivo?->id)
                ->with(['disciplina', 'turma', 'trimestre', 'professor.user'])
                ->orderBy('created_at', 'desc')
                ->get();
            return view('trabalhos.index', compact('planos', 'anoLetivo'));
        } elseif ($user->isProfessor()) {
            $professor = $user->professor;
            $trabalhos = $professor ? $professor->trabalhos()->where('ano_letivo_id', $anoLetivo?->id)->with(['disciplina', 'turma'])->get() : collect();
            return view('trabalhos.index', compact('trabalhos', 'anoLetivo'));
        } elseif ($user->isAluno()) {
            $aluno = $user->aluno;
            $trabalhos = $aluno ? $aluno->trabalhos()->where('ano_letivo_id', $anoLetivo?->id)->with(['disciplina', 'turma', 'professor'])->get() : collect();
            return view('trabalhos.index', compact('trabalhos', 'anoLetivo'));
        }

        abort(403);
    }

    public function create()
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403);
        }

        $professor = $user->professor;
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        $turmas = $professor ? $professor->turmas()->wherePivot('ano_letivo_id', $anoLetivo?->id)->get() : collect();
        $disciplinas = $professor ? $professor->disciplinas()->get() : collect();

        return view('trabalhos.create', compact('turmas', 'disciplinas', 'anoLetivo'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403);
        }

        $request->validate([
            'titulo' => 'required|string',
            'descricao' => 'nullable|string',
            'disciplina_id' => 'required|exists:disciplinas,id',
            'turma_id' => 'required|exists:turmas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'data_entrega' => 'required|date',
            'valor' => 'nullable|numeric|min:0|max:20',
            'tipo' => 'required|in:Trabalho,TPC',
        ]);

        $professor = $user->professor;
        
        // Verifica se o professor leciona esta disciplina nesta turma
        $alocacao = $professor->turmas()
            ->wherePivot('turma_id', $request->turma_id)
            ->wherePivot('disciplina_id', $request->disciplina_id)
            ->first();

        if (!$alocacao) {
            abort(403, 'Você não leciona esta disciplina nesta turma.');
        }

        $dados = $request->all();
        $dados['professor_id'] = $professor->id;
        
        $trabalho = Trabalho::create($dados);

        // Associa todos os alunos da turma
        $turma = \App\Models\Turma::find($request->turma_id);
        $alunos = $turma->alunos()->get();
        foreach ($alunos as $aluno) {
            $trabalho->alunos()->attach($aluno->id);
        }

        return redirect()->route('trabalhos.index')->with('success', 'Trabalho criado com sucesso!');
    }

    public function visualizarPlano(PlanoTrimestral $plano)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_planos')) {
            abort(403, 'Você não tem permissão para visualizar este plano.');
        }

        if (!Storage::disk('public')->exists($plano->arquivo)) {
            abort(404, 'Arquivo não encontrado.');
        }

        $caminhoArquivo = Storage::disk('public')->path($plano->arquivo);
        $mimeType = Storage::disk('public')->mimeType($plano->arquivo);
        
        return response()->file($caminhoArquivo, [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'inline; filename="' . $plano->nome_arquivo . '"',
        ]);
    }
}



