<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;

class InvoiceController extends Controller
{
    public function __construct()
    {
        // Rotas de aluno
        $this->middleware(['auth', 'role:aluno'])->except(['verificar']);
        // Rota de verificação (acessível por funcionários/admin)
        $this->middleware(['auth', 'role:admin,superadmin,funcionario'])->only(['verificar']);
    }

    /**
     * Lista facturas do aluno
     */
    public function index()
    {
        $user = auth()->user();
        $aluno = $user->aluno;

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Perfil de estudante não encontrado.']);
        }

        $invoices = $aluno->invoices()
            ->with('payment')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('aluno.invoice.index', compact('invoices'));
    }

    /**
     * Mostra uma factura específica
     */
    public function show(Invoice $invoice)
    {
        $user = auth()->user();
        
        if ($invoice->aluno_id !== $user->aluno->id) {
            abort(403);
        }

        $invoice->load(['aluno.user', 'payment', 'anoLetivo']);

        // Preparar logotipo para a view
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoUrl = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoUrl = asset('storage/' . $logotipo);
        }

        return view('aluno.invoice.show', compact('invoice', 'logotipoUrl'));
    }

    /**
     * Download PDF da factura
     */
    public function download(Invoice $invoice)
    {
        $user = auth()->user();
        
        if ($invoice->aluno_id !== $user->aluno->id) {
            abort(403);
        }

        $invoice->load(['aluno.user', 'payment', 'anoLetivo']);

        // Preparar logotipo em base64 para o PDF
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $logotipoBase64 = 'data:' . mime_content_type($logotipoPath) . ';base64,' . base64_encode($logotipoData);
        }

        $pdf = Pdf::loadView('aluno.invoice.pdf', compact('invoice', 'logotipoBase64'));
        
        return $pdf->download('factura-' . $invoice->numero_factura . '.pdf');
    }

    /**
     * Verificar factura por código (para funcionários)
     */
    public function verificar(Request $request)
    {
        $codigo = $request->input('codigo');

        if (!$codigo) {
            return view('admin.invoice.verificar');
        }

        $invoice = Invoice::where('codigo_verificacao', $codigo)
            ->with(['aluno.user', 'payment', 'anoLetivo'])
            ->first();

        if (!$invoice) {
            return view('admin.invoice.verificar', [
                'erro' => 'Factura não encontrada com o código fornecido.',
                'codigo' => $codigo
            ]);
        }

        // Preparar logotipo para a view
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoUrl = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoUrl = asset('storage/' . $logotipo);
        }

        return view('admin.invoice.verificar', compact('invoice', 'logotipoUrl', 'codigo'));
    }
}
