<?php

namespace App\Http\Controllers;

use App\Models\Horario;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Professor;
use App\Models\AnoLetivo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;

class HorarioController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        // Buscar todas as turmas para o filtro
        $queryTurmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true);
        
        // Filtro por nivel_ensino
        if ($request->filled('nivel_ensino')) {
            $queryTurmas->where('nivel_ensino', $request->nivel_ensino);
        }
        
        $todasTurmas = $queryTurmas->orderBy('codigo')->get();
        
        // Filtro por turma
        $turmaFiltro = $request->get('turma_id');
        $nivelEnsinoFiltro = $request->get('nivel_ensino');
        
        // Para horários, vamos paginar por turma (não por horário individual)
        // Primeiro buscar turmas com horários
        $queryTurmasComHorarios = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->whereHas('horarios', function($q) use ($anoLetivo) {
                $q->where('ano_letivo_id', $anoLetivo?->id);
            })
            ->when($turmaFiltro, function($query) use ($turmaFiltro) {
                return $query->where('id', $turmaFiltro);
            })
            ->when($nivelEnsinoFiltro, function($query) use ($nivelEnsinoFiltro) {
                return $query->where('nivel_ensino', $nivelEnsinoFiltro);
            });
        
        // Otimização: Paginação e select específico
        $turmasComHorarios = $queryTurmasComHorarios->select('turmas.id', 'turmas.nome', 'turmas.codigo', 'turmas.nivel_ensino', 'turmas.ano_letivo_id')
            ->orderBy('codigo')
            ->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());
        
        // Buscar horários das turmas da página atual
        $turmaIds = $turmasComHorarios->pluck('id');
        $horarios = Horario::where('ano_letivo_id', $anoLetivo?->id)
            ->whereIn('turma_id', $turmaIds)
            ->with(['turma', 'disciplina', 'professor'])
            ->orderBy('hora_inicio')
            ->orderBy('dia_semana')
            ->get();

        // Organizar por turma
        $horariosPorTurma = $horarios->groupBy('turma_id');
        
        // Para cada turma, organizar em grade (dia x hora)
        $grades = [];
        $diasSemana = ['Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta'];
        
        foreach ($turmasComHorarios as $turma) {
            $turmaId = $turma->id;
            $horariosTurma = $horariosPorTurma->get($turmaId, collect());
            $grade = [];
            
            // Obter todos os horários únicos para esta turma (com início e fim)
            $horariosUnicos = $horariosTurma->map(function($h) {
                return [
                    'inicio' => \Carbon\Carbon::parse($h->hora_inicio)->format('H:i'),
                    'fim' => \Carbon\Carbon::parse($h->hora_fim)->format('H:i'),
                    'hora_inicio' => $h->hora_inicio,
                ];
            })->unique(function($item) {
                return $item['inicio'];
            })->sortBy('hora_inicio')->values();
            
            // Criar estrutura de grade e identificar intervalo
            $horariosComIntervalo = [];
            
            foreach ($horariosUnicos as $index => $horarioUnico) {
                $horaFormatada = $horarioUnico['inicio'];
                $horaCarbon = \Carbon\Carbon::parse($horarioUnico['hora_inicio']);
                $grade[$horaFormatada] = [];
                
                foreach ($diasSemana as $dia) {
                    $horario = $horariosTurma->first(function($h) use ($dia, $horaCarbon) {
                        $horaInicio = \Carbon\Carbon::parse($h->hora_inicio);
                        return $h->dia_semana === $dia && 
                               $horaInicio->format('H:i') === $horaCarbon->format('H:i');
                    });
                    
                    $grade[$horaFormatada][$dia] = $horario;
                }
                
                // Adicionar horário à lista
                $horariosComIntervalo[] = [
                    'inicio' => $horarioUnico['inicio'],
                    'fim' => $horarioUnico['fim'],
                    'display' => $horarioUnico['inicio'] . ' - ' . $horarioUnico['fim'],
                    'tipo' => 'aula',
                ];
                
                // Verificar se há intervalo após a 3ª aula (índice 2, após criar a 3ª)
                if ($index == 2 && isset($horariosUnicos[$index + 1])) {
                    $fimTerceira = \Carbon\Carbon::parse($horarioUnico['fim']);
                    $inicioQuarta = \Carbon\Carbon::parse($horariosUnicos[$index + 1]['inicio']);
                    $diferencaMinutos = $fimTerceira->diffInMinutes($inicioQuarta);
                    
                    // Se a diferença for 15 minutos, é o intervalo de lanche
                    if ($diferencaMinutos == 15) {
                        $horariosComIntervalo[] = [
                            'inicio' => $fimTerceira->format('H:i'),
                            'fim' => $inicioQuarta->format('H:i'),
                            'display' => 'Intervalo: ' . $fimTerceira->format('H:i') . ' - ' . $inicioQuarta->format('H:i'),
                            'tipo' => 'intervalo',
                        ];
                    }
                }
            }
            
            $grades[$turmaId] = [
                'turma' => $turma,
                'grade' => $grade,
                'horarios' => $horariosComIntervalo,
            ];
        }

        return view('horarios.index', compact('grades', 'anoLetivo', 'diasSemana', 'todasTurmas', 'turmaFiltro', 'nivelEnsinoFiltro', 'turmasComHorarios'));
    }

    public function create()
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)->where('ativa', true)->get();
        $disciplinas = Disciplina::where('ativa', true)->get();
        $professores = Professor::with('user')->get();

        return view('horarios.create', compact('turmas', 'disciplinas', 'professores', 'anoLetivo'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'disciplina_id' => 'required|exists:disciplinas,id',
            'professor_id' => 'required|exists:professores,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'dia_semana' => 'required|in:Segunda,Terça,Quarta,Quinta,Sexta',
            'hora_inicio' => 'required|date_format:H:i',
            'hora_fim' => 'required|date_format:H:i|after:hora_inicio',
            'sala' => 'nullable|string',
        ]);

        // Verificar conflito de horário do professor
        $conflito = $this->verificarConflitoProfessor(
            $request->professor_id,
            $request->dia_semana,
            $request->hora_inicio,
            $request->hora_fim,
            $request->ano_letivo_id,
            null // Não há horário existente (criação)
        );

        if ($conflito) {
            $turmaConflito = Turma::find($conflito->turma_id);
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'hora_inicio' => "Conflito de horário! O professor já está alocado para dar aulas na turma '{$turmaConflito->codigo} - {$turmaConflito->nome}' no mesmo dia e horário."
                ]);
        }

        // Verificar carga horária da disciplina (por turma)
        $disciplina = Disciplina::findOrFail($request->disciplina_id);
        $cargaHorariaTurma = DB::table('turma_disciplina')
            ->where('turma_id', $request->turma_id)
            ->where('disciplina_id', $request->disciplina_id)
            ->where('ano_letivo_id', $request->ano_letivo_id)
            ->value('carga_horaria_semanal');
        $cargaHoraria = $cargaHorariaTurma ?? $disciplina->carga_horaria_semanal;

        if ($cargaHoraria) {
            $minutosAula = \Carbon\Carbon::parse($request->hora_inicio)
                ->diffInMinutes(\Carbon\Carbon::parse($request->hora_fim));
            
            // Calcular minutos já alocados para esta disciplina nesta turma e ano letivo
            $minutosAlocados = Horario::where('disciplina_id', $request->disciplina_id)
                ->where('turma_id', $request->turma_id)
                ->where('ano_letivo_id', $request->ano_letivo_id)
                ->get()
                ->sum(function($horario) {
                    return \Carbon\Carbon::parse($horario->hora_inicio)
                        ->diffInMinutes(\Carbon\Carbon::parse($horario->hora_fim));
                });
            
            $novosMinutos = $minutosAlocados + $minutosAula;
            
            if ($novosMinutos > $cargaHoraria) {
                $horasDisponiveis = floor(($cargaHoraria - $minutosAlocados) / 60);
                $minutosDisponiveis = ($cargaHoraria - $minutosAlocados) % 60;
                return redirect()->back()
                    ->withInput()
                    ->withErrors([
                        'hora_fim' => "A carga horária semanal desta disciplina (" . $this->formatarCargaHoraria($cargaHoraria) . ") seria excedida. Restam apenas {$horasDisponiveis}h {$minutosDisponiveis}min disponíveis."
                    ]);
            }
        }

        // Permitir no máximo uma dupla por dia (em qualquer período)
        $erroDupla = $this->validarDuplaConsecutiva(
            $request->turma_id,
            $request->dia_semana,
            $request->ano_letivo_id,
            $request->disciplina_id,
            $request->hora_inicio,
            $request->hora_fim
        );
        if ($erroDupla) {
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'disciplina_id' => $erroDupla
                ]);
        }

        // Não permitir a mesma disciplina no mesmo horário em dias seguidos
        if ($this->mesmoHorarioEmDiasSeguidos(
            $request->turma_id,
            $request->disciplina_id,
            $request->ano_letivo_id,
            $request->dia_semana,
            $request->hora_inicio,
            $request->hora_fim
        )) {
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'hora_inicio' => 'Não é permitido repetir a mesma disciplina no mesmo horário em dias seguidos.'
                ]);
        }

        Horario::create($request->all());

        return redirect()->route('horarios.index')->with('success', 'Horário criado com sucesso!');
    }

    public function edit(Horario $horario)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)->where('ativa', true)->get();
        $disciplinas = Disciplina::where('ativa', true)->get();
        $professores = Professor::with('user')->get();

        return view('horarios.edit', compact('horario', 'turmas', 'disciplinas', 'professores', 'anoLetivo'));
    }

    public function update(Request $request, Horario $horario)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'disciplina_id' => 'required|exists:disciplinas,id',
            'professor_id' => 'required|exists:professores,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'dia_semana' => 'required|in:Segunda,Terça,Quarta,Quinta,Sexta',
            'hora_inicio' => 'required|date_format:H:i',
            'hora_fim' => 'required|date_format:H:i|after:hora_inicio',
            'sala' => 'nullable|string',
        ]);

        // Verificar conflito de horário do professor
        $conflito = $this->verificarConflitoProfessor(
            $request->professor_id,
            $request->dia_semana,
            $request->hora_inicio,
            $request->hora_fim,
            $request->ano_letivo_id,
            $horario->id // Excluir o próprio horário da verificação
        );

        if ($conflito) {
            $turmaConflito = Turma::find($conflito->turma_id);
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'hora_inicio' => "Conflito de horário! O professor já está alocado para dar aulas na turma '{$turmaConflito->codigo} - {$turmaConflito->nome}' no mesmo dia e horário."
                ]);
        }

        // Verificar carga horária da disciplina (por turma)
        $disciplina = Disciplina::findOrFail($request->disciplina_id);
        $cargaHorariaTurma = DB::table('turma_disciplina')
            ->where('turma_id', $request->turma_id)
            ->where('disciplina_id', $request->disciplina_id)
            ->where('ano_letivo_id', $request->ano_letivo_id)
            ->value('carga_horaria_semanal');
        $cargaHoraria = $cargaHorariaTurma ?? $disciplina->carga_horaria_semanal;

        if ($cargaHoraria) {
            $minutosAula = \Carbon\Carbon::parse($request->hora_inicio)
                ->diffInMinutes(\Carbon\Carbon::parse($request->hora_fim));
            
            // Calcular minutos já alocados para esta disciplina nesta turma e ano letivo (excluindo o horário atual)
            $minutosAlocados = Horario::where('disciplina_id', $request->disciplina_id)
                ->where('turma_id', $request->turma_id)
                ->where('ano_letivo_id', $request->ano_letivo_id)
                ->where('id', '!=', $horario->id)
                ->get()
                ->sum(function($h) {
                    return \Carbon\Carbon::parse($h->hora_inicio)
                        ->diffInMinutes(\Carbon\Carbon::parse($h->hora_fim));
                });
            
            $novosMinutos = $minutosAlocados + $minutosAula;
            
            if ($novosMinutos > $cargaHoraria) {
                $horasDisponiveis = floor(($cargaHoraria - $minutosAlocados) / 60);
                $minutosDisponiveis = ($cargaHoraria - $minutosAlocados) % 60;
                return redirect()->back()
                    ->withInput()
                    ->withErrors([
                        'hora_fim' => "A carga horária semanal desta disciplina (" . $this->formatarCargaHoraria($cargaHoraria) . ") seria excedida. Restam apenas {$horasDisponiveis}h {$minutosDisponiveis}min disponíveis."
                    ]);
            }
        }

        // Permitir no máximo uma dupla por dia (em qualquer período)
        $erroDupla = $this->validarDuplaConsecutiva(
            $request->turma_id,
            $request->dia_semana,
            $request->ano_letivo_id,
            $request->disciplina_id,
            $request->hora_inicio,
            $request->hora_fim,
            $horario->id
        );
        if ($erroDupla) {
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'disciplina_id' => $erroDupla
                ]);
        }

        // Não permitir a mesma disciplina no mesmo horário em dias seguidos
        if ($this->mesmoHorarioEmDiasSeguidos(
            $request->turma_id,
            $request->disciplina_id,
            $request->ano_letivo_id,
            $request->dia_semana,
            $request->hora_inicio,
            $request->hora_fim,
            $horario->id
        )) {
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'hora_inicio' => 'Não é permitido repetir a mesma disciplina no mesmo horário em dias seguidos.'
                ]);
        }

        $horario->update($request->all());

        return redirect()->route('horarios.index')->with('success', 'Horário atualizado com sucesso!');
    }

    public function destroy(Horario $horario)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $horario->delete();

        return redirect()->route('horarios.index')->with('success', 'Horário eliminado com sucesso!');
    }

    public function gerarAutomatico()
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->get();

        return view('horarios.gerar-automatico', compact('turmas', 'anoLetivo'));
    }

    public function storeAutomatico(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'hora_inicio_manha' => 'required|date_format:H:i',
            'hora_fim_manha' => 'required|date_format:H:i|after:hora_inicio_manha',
            'duracao_aula' => 'required|integer|min:30|max:120',
            'dias_semana' => 'required|array|min:1',
            'dias_semana.*' => 'in:Segunda,Terça,Quarta,Quinta,Sexta',
            'sala_padrao' => 'nullable|string',
        ]);

        $turma = Turma::findOrFail($request->turma_id);
        $anoLetivo = AnoLetivo::findOrFail($request->ano_letivo_id);

        // Primeiro: Buscar disciplinas atribuídas à turma (tabela turma_disciplina)
        $disciplinasAtribuidas = DB::table('turma_disciplina')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->toArray();

        if (empty($disciplinasAtribuidas)) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['turma_id' => 'Esta turma não tem disciplinas atribuídas. Por favor, atribua disciplinas primeiro na página de turmas.']);
        }

        // Segundo: Buscar apenas professores que lecionam essas disciplinas específicas nesta turma
        $alocacoes = DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->whereIn('disciplina_id', $disciplinasAtribuidas)
            ->get();

        if ($alocacoes->isEmpty()) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['turma_id' => 'Esta turma não tem professores atribuídos às disciplinas. Por favor, aloque professores às disciplinas primeiro.']);
        }

        // Mapa de carga horária por disciplina (por turma)
        $cargasPorDisciplina = DB::table('turma_disciplina')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('carga_horaria_semanal', 'disciplina_id')
            ->toArray();

        // Calcular horários com intervalo - exatamente 6 aulas por dia
        $horarios = $this->calcularHorariosComIntervalo(
            $request->hora_inicio_manha,
            $request->hora_fim_manha,
            $request->duracao_aula,
            15 // Intervalo de 15 minutos
        );

        // Deve ter exatamente 6 aulas
        if (count($horarios) < 6) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['hora_inicio_manha' => 'Os horários configurados não permitem criar 6 aulas por dia. Ajuste os horários de início/fim e duração da aula.']);
        }

        // Limitar a exatamente 6 aulas
        $horarios = array_slice($horarios, 0, 6);

        // Filtrar apenas horários de aula (excluir intervalo)
        $horariosAulas = array_filter($horarios, function($h) {
            return !isset($h['tipo']) || $h['tipo'] !== 'intervalo';
        });
        $horariosAulas = array_values($horariosAulas); // Reindexar
        
        $duracaoAulaMinutos = $request->duracao_aula;
        $horariosCriados = 0;
        
        // Preparar alocações com informações de carga horária
        $alocacoesComCarga = [];
        foreach ($alocacoes as $alocacao) {
            $disciplina = Disciplina::find($alocacao->disciplina_id);
            $cargaTurma = $cargasPorDisciplina[$alocacao->disciplina_id] ?? null;
            $cargaBase = $cargaTurma ?? ($disciplina?->carga_horaria_semanal);

            if ($disciplina && $cargaBase) {
                // Calcular quantas aulas são necessárias para esta disciplina
                $aulasNecessarias = ceil($cargaBase / $duracaoAulaMinutos);
                $alocacoesComCarga[] = [
                    'alocacao' => $alocacao,
                    'disciplina' => $disciplina,
                    'aulas_necessarias' => $aulasNecessarias,
                    'carga_horaria' => $cargaBase,
                    'minutos_alocados' => 0, // Será atualizado conforme criamos horários
                ];
            } else {
                // Se não tem carga horária definida, usar padrão de 2 aulas por semana
                $alocacoesComCarga[] = [
                    'alocacao' => $alocacao,
                    'disciplina' => $disciplina,
                    'aulas_necessarias' => 2,
                    'carga_horaria' => null,
                    'minutos_alocados' => 0,
                ];
            }
        }
        
        if (empty($alocacoesComCarga)) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['turma_id' => 'Nenhuma disciplina com professor atribuído encontrada para esta turma.']);
        }
        
        // Criar horários distribuindo baseado na carga horária de cada disciplina
        foreach ($request->dias_semana as $diaSemana) {
            // Embaralhar alocações para cada dia ter ordem diferente (mas manter referência ao array original)
            $indicesEmbaralhados = range(0, count($alocacoesComCarga) - 1);
            shuffle($indicesEmbaralhados);
            $disciplinasUsadasDia = [];
            
            // Para cada horário disponível no dia
            for ($i = 0; $i < count($horariosAulas); $i++) {
                $horarioAula = $horariosAulas[$i];
                $horarioCriado = false;
                
                // Tentar encontrar uma disciplina que ainda precisa de aulas
                foreach ($indicesEmbaralhados as $indice) {
                    if ($horarioCriado) break;
                    
                    $alocacaoInfo = &$alocacoesComCarga[$indice];
                    $alocacao = $alocacaoInfo['alocacao'];
                    $disciplina = $alocacaoInfo['disciplina'];

                    $disciplinasUsadasDia[$alocacao->disciplina_id] = $disciplinasUsadasDia[$alocacao->disciplina_id] ?? 0;

                    // Não permitir mais de duas aulas da mesma disciplina no dia
                    if ($disciplinasUsadasDia[$alocacao->disciplina_id] >= 2) {
                    continue;
                }
                
                    // Verificar se ainda precisa de mais aulas
                    if ($alocacaoInfo['carga_horaria']) {
                        // Se já atingiu a carga horária, pular
                        if ($alocacaoInfo['minutos_alocados'] >= $alocacaoInfo['carga_horaria']) {
                            continue;
                        }
                        
                        // Verificar se criar esta aula não excederia a carga horária
                        $novosMinutos = $alocacaoInfo['minutos_alocados'] + $duracaoAulaMinutos;
                        if ($novosMinutos > $alocacaoInfo['carga_horaria']) {
                            continue; // Esta disciplina já atingiu sua carga horária
                        }
                    } else {
                        // Se não tem carga horária definida, verificar se já tem aulas suficientes
                        $aulasExistentes = Horario::where('turma_id', $turma->id)
                            ->where('disciplina_id', $alocacao->disciplina_id)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->count();
                        
                        if ($aulasExistentes >= $alocacaoInfo['aulas_necessarias']) {
                            continue;
                        }
                    }
                    
                    // Não permitir a mesma disciplina no mesmo horário em dias seguidos
                    if ($this->mesmoHorarioEmDiasSeguidos(
                        $turma->id,
                        $alocacao->disciplina_id,
                        $anoLetivo->id,
                        $diaSemana,
                        $horarioAula['inicio'],
                        $horarioAula['fim']
                    )) {
                        continue;
                    }

                    // Verificar se já existe este horário
                    $horarioExistente = Horario::where('turma_id', $turma->id)
                    ->where('disciplina_id', $alocacao->disciplina_id)
                    ->where('professor_id', $alocacao->professor_id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->where('dia_semana', $diaSemana)
                        ->where('hora_inicio', $horarioAula['inicio'])
                    ->first();
                
                    if ($horarioExistente) {
                        continue; // Já existe este horário
                    }
                    
                    // Verificar conflito de horário do professor
                    $conflito = $this->verificarConflitoProfessor(
                        $alocacao->professor_id,
                        $diaSemana,
                        $horarioAula['inicio'],
                        $horarioAula['fim'],
                        $anoLetivo->id,
                        null
                    );
                    
                    if (!$conflito) {
                        $criouDupla = false;
                        $podeDupla = ($i + 1 < count($horariosAulas)) &&
                            ($alocacaoInfo['carga_horaria'] ? ($alocacaoInfo['minutos_alocados'] + (2 * $duracaoAulaMinutos) <= $alocacaoInfo['carga_horaria']) : true);

                        $horarioProximo = null;
                        if ($podeDupla) {
                            $horarioProximo = $horariosAulas[$i + 1];
                            if ($this->mesmoHorarioEmDiasSeguidos(
                                $turma->id,
                                $alocacao->disciplina_id,
                                $anoLetivo->id,
                                $diaSemana,
                                $horarioProximo['inicio'],
                                $horarioProximo['fim']
                            )) {
                                $podeDupla = false;
                            }
                        }

                        if ($podeDupla) {
                            $conflitoDupla = $this->verificarConflitoProfessor(
                                $alocacao->professor_id,
                                $diaSemana,
                                $horarioProximo['inicio'],
                                $horarioProximo['fim'],
                                $anoLetivo->id,
                                null
                            );

                            $horarioExistenteDupla = Horario::where('turma_id', $turma->id)
                                ->where('disciplina_id', $alocacao->disciplina_id)
                                ->where('professor_id', $alocacao->professor_id)
                                ->where('ano_letivo_id', $anoLetivo->id)
                                ->where('dia_semana', $diaSemana)
                                ->where('hora_inicio', $horarioProximo['inicio'])
                                ->first();

                            if (!$conflitoDupla && !$horarioExistenteDupla) {
                        Horario::create([
                            'turma_id' => $turma->id,
                            'disciplina_id' => $alocacao->disciplina_id,
                            'professor_id' => $alocacao->professor_id,
                            'ano_letivo_id' => $anoLetivo->id,
                            'dia_semana' => $diaSemana,
                                    'hora_inicio' => $horarioAula['inicio'],
                                    'hora_fim' => $horarioAula['fim'],
                            'sala' => $request->sala_padrao,
                        ]);
                                Horario::create([
                                    'turma_id' => $turma->id,
                                    'disciplina_id' => $alocacao->disciplina_id,
                                    'professor_id' => $alocacao->professor_id,
                                    'ano_letivo_id' => $anoLetivo->id,
                                    'dia_semana' => $diaSemana,
                                    'hora_inicio' => $horarioProximo['inicio'],
                                    'hora_fim' => $horarioProximo['fim'],
                                    'sala' => $request->sala_padrao,
                                ]);

                                $alocacoesComCarga[$indice]['minutos_alocados'] += (2 * $duracaoAulaMinutos);
                                $horariosCriados += 2;
                                $horarioCriado = true;
                                $disciplinasUsadasDia[$alocacao->disciplina_id] += 2;
                                $criouDupla = true;
                                $i++; // pular o próximo horário, já usado na dupla
                            }
                        }

                        if ($criouDupla) {
                            continue;
                        }

                        Horario::create([
                            'turma_id' => $turma->id,
                            'disciplina_id' => $alocacao->disciplina_id,
                            'professor_id' => $alocacao->professor_id,
                            'ano_letivo_id' => $anoLetivo->id,
                            'dia_semana' => $diaSemana,
                            'hora_inicio' => $horarioAula['inicio'],
                            'hora_fim' => $horarioAula['fim'],
                            'sala' => $request->sala_padrao,
                        ]);
                        
                        // Atualizar minutos alocados (atualiza o array original)
                        $alocacoesComCarga[$indice]['minutos_alocados'] += $duracaoAulaMinutos;
                        $horariosCriados++;
                        $horarioCriado = true;
                        $disciplinasUsadasDia[$alocacao->disciplina_id]++;
                    }
                }
            }
        }

        return redirect()->route('horarios.index')
            ->with('success', "{$horariosCriados} horários criados automaticamente para a turma {$turma->codigo} - {$turma->nome}!");
    }

    public function gerarTodasTurmasView()
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->get();

        return view('horarios.gerar-todas-turmas', compact('turmas', 'anoLetivo'));
    }

    public function gerarTodasTurmas(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $request->validate([
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'hora_inicio_manha' => 'required|date_format:H:i',
            'hora_fim_manha' => 'required|date_format:H:i|after:hora_inicio_manha',
            'duracao_aula' => 'required|integer|min:30|max:120',
            'dias_semana' => 'required|array|min:1',
            'dias_semana.*' => 'in:Segunda,Terça,Quarta,Quinta,Sexta',
            'sala_padrao' => 'nullable|string',
            'nivel_ensino' => 'nullable|in:PRIMARIO,SECUNDARIO',
        ]);

        $anoLetivo = AnoLetivo::findOrFail($request->ano_letivo_id);
        
        // Buscar turmas com disciplinas e professores atribuídos
        $queryTurmas = Turma::where('ano_letivo_id', $anoLetivo->id)
            ->where('ativa', true);
        
        if ($request->filled('nivel_ensino')) {
            $queryTurmas->where('nivel_ensino', $request->nivel_ensino);
        }
        
        $turmas = $queryTurmas->get();
        
        $totalHorariosCriados = 0;
        $turmasProcessadas = 0;
        $turmasSemDisciplinas = [];
        $turmasSemProfessores = [];
        
        foreach ($turmas as $turma) {
            // Verificar se tem disciplinas atribuídas
            $disciplinasAtribuidas = DB::table('turma_disciplina')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->pluck('disciplina_id')
                ->toArray();
            
            if (empty($disciplinasAtribuidas)) {
                $turmasSemDisciplinas[] = $turma->codigo . ' - ' . $turma->nome;
                continue;
            }
            
            // Verificar se tem professores atribuídos
            $alocacoes = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->whereIn('disciplina_id', $disciplinasAtribuidas)
                ->get();
            
            if ($alocacoes->isEmpty()) {
                $turmasSemProfessores[] = $turma->codigo . ' - ' . $turma->nome;
                continue;
            }
            
            // Criar horários para esta turma usando a mesma lógica do storeAutomatico
            $cargasPorDisciplina = DB::table('turma_disciplina')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->pluck('carga_horaria_semanal', 'disciplina_id')
                ->toArray();
            
            $horarios = $this->calcularHorariosComIntervalo(
                $request->hora_inicio_manha,
                $request->hora_fim_manha,
                $request->duracao_aula,
                15
            );
            
            if (count($horarios) < 6) {
                continue;
            }
            
            $horarios = array_slice($horarios, 0, 6);
            $horariosAulas = array_filter($horarios, function($h) {
                return !isset($h['tipo']) || $h['tipo'] !== 'intervalo';
            });
            $horariosAulas = array_values($horariosAulas);
            
            $duracaoAulaMinutos = $request->duracao_aula;
            $horariosCriadosTurma = 0;
            
            $alocacoesComCarga = [];
            foreach ($alocacoes as $alocacao) {
                $disciplina = Disciplina::find($alocacao->disciplina_id);
                $cargaTurma = $cargasPorDisciplina[$alocacao->disciplina_id] ?? null;
                $cargaBase = $cargaTurma ?? ($disciplina?->carga_horaria_semanal);
                
                if ($disciplina && $cargaBase) {
                    $aulasNecessarias = ceil($cargaBase / $duracaoAulaMinutos);
                    $alocacoesComCarga[] = [
                        'alocacao' => $alocacao,
                        'disciplina' => $disciplina,
                        'aulas_necessarias' => $aulasNecessarias,
                        'carga_horaria' => $cargaBase,
                        'minutos_alocados' => 0,
                    ];
                } else {
                    $alocacoesComCarga[] = [
                        'alocacao' => $alocacao,
                        'disciplina' => $disciplina,
                        'aulas_necessarias' => 2,
                        'carga_horaria' => null,
                        'minutos_alocados' => 0,
                    ];
                }
            }
            
            // Criar horários para esta turma
            foreach ($request->dias_semana as $diaSemana) {
                $indicesEmbaralhados = range(0, count($alocacoesComCarga) - 1);
                shuffle($indicesEmbaralhados);
                $disciplinasUsadasDia = [];
                
                for ($i = 0; $i < count($horariosAulas); $i++) {
                    $horarioAula = $horariosAulas[$i];
                    $horarioCriado = false;
                    
                    foreach ($indicesEmbaralhados as $indice) {
                        if ($horarioCriado) break;
                        
                        $alocacaoInfo = &$alocacoesComCarga[$indice];
                        $alocacao = $alocacaoInfo['alocacao'];
                        $disciplina = $alocacaoInfo['disciplina'];
                        
                        $disciplinasUsadasDia[$alocacao->disciplina_id] = $disciplinasUsadasDia[$alocacao->disciplina_id] ?? 0;
                        
                        if ($disciplinasUsadasDia[$alocacao->disciplina_id] >= 2) {
                            continue;
                        }
                        
                        if ($alocacaoInfo['carga_horaria']) {
                            if ($alocacaoInfo['minutos_alocados'] >= $alocacaoInfo['carga_horaria']) {
                                continue;
                            }
                            $novosMinutos = $alocacaoInfo['minutos_alocados'] + $duracaoAulaMinutos;
                            if ($novosMinutos > $alocacaoInfo['carga_horaria']) {
                                continue;
                            }
                        }
                        
                        if ($this->mesmoHorarioEmDiasSeguidos(
                            $turma->id,
                            $alocacao->disciplina_id,
                            $anoLetivo->id,
                            $diaSemana,
                            $horarioAula['inicio'],
                            $horarioAula['fim']
                        )) {
                            continue;
                        }
                        
                        $horarioExistente = Horario::where('turma_id', $turma->id)
                        ->where('disciplina_id', $alocacao->disciplina_id)
                        ->where('professor_id', $alocacao->professor_id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->where('dia_semana', $diaSemana)
                            ->where('hora_inicio', $horarioAula['inicio'])
                        ->first();
                    
                        if ($horarioExistente) {
                            continue;
                        }
                        
                        $conflito = $this->verificarConflitoProfessor(
                            $alocacao->professor_id,
                            $diaSemana,
                            $horarioAula['inicio'],
                            $horarioAula['fim'],
                            $anoLetivo->id,
                            null
                        );
                        
                        if (!$conflito) {
                            Horario::create([
                                'turma_id' => $turma->id,
                                'disciplina_id' => $alocacao->disciplina_id,
                                'professor_id' => $alocacao->professor_id,
                                'ano_letivo_id' => $anoLetivo->id,
                                'dia_semana' => $diaSemana,
                                'hora_inicio' => $horarioAula['inicio'],
                                'hora_fim' => $horarioAula['fim'],
                                'sala' => $request->sala_padrao,
                            ]);
                            
                            $alocacoesComCarga[$indice]['minutos_alocados'] += $duracaoAulaMinutos;
                            $horariosCriadosTurma++;
                            $horarioCriado = true;
                            $disciplinasUsadasDia[$alocacao->disciplina_id]++;
                        }
                    }
                }
            }
            
            $totalHorariosCriados += $horariosCriadosTurma;
            $turmasProcessadas++;
        }
        
        $mensagem = "Horários criados para {$turmasProcessadas} turma(s). Total: {$totalHorariosCriados} horários criados.";
        if (!empty($turmasSemDisciplinas)) {
            $mensagem .= " Turmas sem disciplinas: " . implode(', ', $turmasSemDisciplinas) . ".";
        }
        if (!empty($turmasSemProfessores)) {
            $mensagem .= " Turmas sem professores: " . implode(', ', $turmasSemProfessores) . ".";
        }
        
        return redirect()->route('horarios.index')->with('success', $mensagem);
    }

    private function calcularHorarios($horaInicioManha, $horaFimManha, $horaInicioTarde, $horaFimTarde, $duracaoAula, $diasSemana)
    {
        $horarios = [];
        $duracaoMinutos = $duracaoAula;

        // Período da manhã
        $inicioManha = \Carbon\Carbon::parse($horaInicioManha);
        $fimManha = \Carbon\Carbon::parse($horaFimManha);
        
        $atual = $inicioManha->copy();
        while ($atual->copy()->addMinutes($duracaoMinutos)->lte($fimManha)) {
            $horarios[] = [
                'inicio' => $atual->format('H:i'),
                'fim' => $atual->copy()->addMinutes($duracaoMinutos)->format('H:i'),
            ];
            $atual->addMinutes($duracaoMinutos);
        }

        // Período da tarde (se fornecido)
        if ($horaInicioTarde && $horaFimTarde) {
            $inicioTarde = \Carbon\Carbon::parse($horaInicioTarde);
            $fimTarde = \Carbon\Carbon::parse($horaFimTarde);
            
            $atual = $inicioTarde->copy();
            while ($atual->copy()->addMinutes($duracaoMinutos)->lte($fimTarde)) {
                $horarios[] = [
                    'inicio' => $atual->format('H:i'),
                    'fim' => $atual->copy()->addMinutes($duracaoMinutos)->format('H:i'),
                ];
                $atual->addMinutes($duracaoMinutos);
            }
        }

        return $horarios;
    }

    /**
     * Verifica se há conflito de horário para o professor
     * Um professor não pode ter dois horários no mesmo dia e hora em turmas diferentes
     */
    private function verificarConflitoProfessor($professorId, $diaSemana, $horaInicio, $horaFim, $anoLetivoId, $horarioIdExcluir = null)
    {
        $horaInicioCarbon = \Carbon\Carbon::parse($horaInicio);
        $horaFimCarbon = \Carbon\Carbon::parse($horaFim);
        
        $query = Horario::where('professor_id', $professorId)
            ->where('ano_letivo_id', $anoLetivoId)
            ->where('dia_semana', $diaSemana);
        
        // Excluir o próprio horário na atualização
        if ($horarioIdExcluir) {
            $query->where('id', '!=', $horarioIdExcluir);
        }
        
        $horariosExistentes = $query->get();
        
        foreach ($horariosExistentes as $horarioExistente) {
            $horaInicioExistente = \Carbon\Carbon::parse($horarioExistente->hora_inicio);
            $horaFimExistente = \Carbon\Carbon::parse($horarioExistente->hora_fim);
            
            // Verificar sobreposição de horários
            // Conflito se: início novo está entre início e fim existente
            // OU fim novo está entre início e fim existente
            // OU novo horário engloba o existente
            if (
                ($horaInicioCarbon->gte($horaInicioExistente) && $horaInicioCarbon->lt($horaFimExistente)) ||
                ($horaFimCarbon->gt($horaInicioExistente) && $horaFimCarbon->lte($horaFimExistente)) ||
                ($horaInicioCarbon->lte($horaInicioExistente) && $horaFimCarbon->gte($horaFimExistente))
            ) {
                return $horarioExistente;
            }
        }
        
        return null;
    }

    private function validarDuplaConsecutiva($turmaId, $diaSemana, $anoLetivoId, $disciplinaId, $horaInicio, $horaFim, $horarioIdExcluir = null): ?string
    {
        $horarios = Horario::where('turma_id', $turmaId)
            ->where('ano_letivo_id', $anoLetivoId)
            ->where('dia_semana', $diaSemana)
            ->where('disciplina_id', $disciplinaId)
            ->when($horarioIdExcluir, function ($query) use ($horarioIdExcluir) {
                $query->where('id', '!=', $horarioIdExcluir);
            })
            ->orderBy('hora_inicio')
            ->get(['hora_inicio', 'hora_fim']);

        if ($horarios->isEmpty()) {
            return null;
        }

        if ($horarios->count() >= 2) {
            return 'Não é permitido ter mais de duas aulas da mesma disciplina no mesmo dia.';
        }

        return null;
    }

    private function mesmoHorarioEmDiasSeguidos($turmaId, $disciplinaId, $anoLetivoId, $diaSemana, $horaInicio, $horaFim, $horarioIdExcluir = null): bool
    {
        $ordemDias = ['Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta'];
        $indice = array_search($diaSemana, $ordemDias, true);
        if ($indice === false) {
            return false;
        }

        $diasConsecutivos = [];
        if ($indice > 0) {
            $diasConsecutivos[] = $ordemDias[$indice - 1];
        }
        if ($indice < count($ordemDias) - 1) {
            $diasConsecutivos[] = $ordemDias[$indice + 1];
        }

        if (empty($diasConsecutivos)) {
            return false;
        }

        return Horario::where('turma_id', $turmaId)
            ->where('disciplina_id', $disciplinaId)
            ->where('ano_letivo_id', $anoLetivoId)
            ->whereIn('dia_semana', $diasConsecutivos)
            ->where('hora_inicio', $horaInicio)
            ->where('hora_fim', $horaFim)
            ->when($horarioIdExcluir, function ($query) use ($horarioIdExcluir) {
                $query->where('id', '!=', $horarioIdExcluir);
            })
            ->exists();
    }

    private function formatarCargaHoraria(?int $minutos): string
    {
        if (!$minutos) {
            return '-';
        }

        $horas = floor($minutos / 60);
        $min = $minutos % 60;

        if ($horas > 0 && $min > 0) {
            return "{$horas}h {$min}min";
        }

        if ($horas > 0) {
            return "{$horas}h";
        }

        return "{$min}min";
    }

    private function calcularHorariosComIntervalo($horaInicio, $horaFim, $duracaoAula, $duracaoIntervalo = 15)
    {
        $horarios = [];
        $duracaoMinutos = $duracaoAula;
        $intervaloEntreAulas = 5; // 5 minutos entre aulas normais
        $inicio = \Carbon\Carbon::parse($horaInicio);
        $fim = \Carbon\Carbon::parse($horaFim);
        
        $atual = $inicio->copy();
        $aulaNumero = 0;
        
        // Calcular exatamente 6 aulas
        // 1ª aula termina às 08:15, 2ª começa às 08:20 (5 min de intervalo)
        // Intervalo de lanche de 15 minutos após a 3ª aula
        while ($aulaNumero < 6) {
            $fimAula = $atual->copy()->addMinutes($duracaoMinutos);
            
            // Verificar se ainda cabe no horário final
            if ($fimAula->gt($fim)) {
                break; // Não cabe mais aulas
            }
            
            $horarios[] = [
                'inicio' => $atual->format('H:i'),
                'fim' => $fimAula->format('H:i'),
            ];
            
            $atual = $fimAula;
            $aulaNumero++;
            
            // Adicionar intervalo após cada aula
            if ($aulaNumero < 6) {
                if ($aulaNumero == 3) {
                    // Intervalo de lanche de 15 minutos após a 3ª aula
                    $atual->addMinutes($duracaoIntervalo);
                } else {
                    // 5 minutos entre outras aulas
                    $atual->addMinutes($intervaloEntreAulas);
                }
            }
        }

        return $horarios;
    }

    public function downloadPdf(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turmaFiltro = $request->get('turma_id');
        
        $horarios = Horario::where('ano_letivo_id', $anoLetivo?->id)
            ->with(['turma', 'disciplina', 'professor.user'])
            ->when($turmaFiltro, function($query) use ($turmaFiltro) {
                return $query->where('turma_id', $turmaFiltro);
            })
            ->orderBy('hora_inicio')
            ->orderBy('dia_semana')
            ->get();

        // Organizar por turma
        $horariosPorTurma = $horarios->groupBy('turma_id');
        
        // Para cada turma, organizar em grade (dia x hora)
        $grades = [];
        $diasSemana = ['Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta'];
        
        foreach ($horariosPorTurma as $turmaId => $horariosTurma) {
            $turma = Turma::find($turmaId);
            $grade = [];
            
            // Obter todos os horários únicos para esta turma (com início e fim)
            $horariosUnicos = $horariosTurma->map(function($h) {
                return [
                    'inicio' => \Carbon\Carbon::parse($h->hora_inicio)->format('H:i'),
                    'fim' => \Carbon\Carbon::parse($h->hora_fim)->format('H:i'),
                    'hora_inicio' => $h->hora_inicio,
                ];
            })->unique(function($item) {
                return $item['inicio'];
            })->sortBy('hora_inicio')->values();
            
            // Criar estrutura de grade e identificar intervalo
            $horariosComIntervalo = [];
            
            foreach ($horariosUnicos as $index => $horarioUnico) {
                $horaFormatada = $horarioUnico['inicio'];
                $horaCarbon = \Carbon\Carbon::parse($horarioUnico['hora_inicio']);
                $grade[$horaFormatada] = [];
                
                foreach ($diasSemana as $dia) {
                    $horario = $horariosTurma->first(function($h) use ($dia, $horaCarbon) {
                        $horaInicio = \Carbon\Carbon::parse($h->hora_inicio);
                        return $h->dia_semana === $dia && 
                               $horaInicio->format('H:i') === $horaCarbon->format('H:i');
                    });
                    
                    $grade[$horaFormatada][$dia] = $horario;
                }
                
                // Adicionar horário à lista
                $horariosComIntervalo[] = [
                    'inicio' => $horarioUnico['inicio'],
                    'fim' => $horarioUnico['fim'],
                    'display' => $horarioUnico['inicio'] . ' - ' . $horarioUnico['fim'],
                    'tipo' => 'aula',
                ];
                
                // Verificar se há intervalo após a 3ª aula (índice 2, após criar a 3ª)
                if ($index == 2 && isset($horariosUnicos[$index + 1])) {
                    $fimTerceira = \Carbon\Carbon::parse($horarioUnico['fim']);
                    $inicioQuarta = \Carbon\Carbon::parse($horariosUnicos[$index + 1]['inicio']);
                    $diferencaMinutos = $fimTerceira->diffInMinutes($inicioQuarta);
                    
                    // Se a diferença for 15 minutos, é o intervalo de lanche
                    if ($diferencaMinutos == 15) {
                        $horariosComIntervalo[] = [
                            'inicio' => $fimTerceira->format('H:i'),
                            'fim' => $inicioQuarta->format('H:i'),
                            'display' => 'Intervalo: ' . $fimTerceira->format('H:i') . ' - ' . $inicioQuarta->format('H:i'),
                            'tipo' => 'intervalo',
                        ];
                    }
                }
            }
            
            $grades[$turmaId] = [
                'turma' => $turma,
                'grade' => $grade,
                'horarios' => $horariosComIntervalo,
            ];
        }
        
        $configuracao = [
            'republica' => \App\Models\Configuracao::get('republica', 'REPÚBLICA DE MOÇAMBIQUE'),
            'distrito' => \App\Models\Configuracao::get('distrito', 'GOVERNO DO DISTRITO DE MOCUBA'),
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
            'logotipo' => \App\Models\Configuracao::get('logotipo_escola', null),
        ];

        $logotipoBase64 = null;
        if (!empty($configuracao['logotipo'])) {
            $path = storage_path('app/public/' . $configuracao['logotipo']);
            if (is_file($path)) {
                $mime = mime_content_type($path);
                $data = base64_encode(file_get_contents($path));
                $logotipoBase64 = "data:{$mime};base64,{$data}";
            }
        }
        
        $diasAbreviados = ['Segunda' => 'Seg', 'Terça' => 'Ter', 'Quarta' => 'Qua', 'Quinta' => 'Qui', 'Sexta' => 'Sex'];
        
        $pdf = Pdf::loadView('horarios.pdf', compact('grades', 'anoLetivo', 'diasSemana', 'diasAbreviados', 'configuracao', 'logotipoBase64'))
            ->setPaper('a4', 'landscape');
        
        if ($turmaFiltro && isset($grades[$turmaFiltro])) {
            $turmaNome = $grades[$turmaFiltro]['turma']->codigo ?? 'Turma';
            $filename = 'Horarios_' . $turmaNome . '_' . date('Y-m-d') . '.pdf';
        } else {
            $filename = 'Horarios_Todas_Turmas_' . date('Y-m-d') . '.pdf';
        }
        
        return $pdf->download($filename);
    }

    public function destroyAll(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_horarios')) {
            abort(403);
        }
        
        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ]);

        $turma = Turma::findOrFail($request->turma_id);
        $anoLetivo = AnoLetivo::findOrFail($request->ano_letivo_id);

        $deleted = Horario::where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->delete();

        return redirect()->route('horarios.index')
            ->with('success', "Todos os horários da turma {$turma->codigo} - {$turma->nome} foram eliminados com sucesso!");
    }
}



