<?php

namespace App\Http\Controllers;

use App\Models\Financeiro;
use App\Models\Aluno;
use App\Models\AnoLetivo;
use App\Models\Turma;
use App\Models\Payment;
use App\Models\Invoice;
use App\Models\User;
use App\Models\Notification;
use App\Services\FinanceiroService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;

class FinanceiroController extends Controller
{
    protected $financeiroService;

    public function __construct(FinanceiroService $financeiroService)
    {
        $this->middleware('auth');
        $this->financeiroService = $financeiroService;
    }

    public function index(Request $request)
    {
        $user = auth()->user();
        
        // Sempre usar 2026 como padrão
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivo?->id);

        if ($user->isAluno()) {
            $aluno = $user->aluno;
            $financeiro = $aluno ? $aluno->financeiro()->where('ano_letivo_id', $anoLetivoId)->get() : collect();
            $situacao = $aluno ? $this->financeiroService->getSituacaoFinanceira($aluno, $anoLetivoId) : 'REGULAR';
            
            // Para alunos, mostrar apenas 2026 e anos anteriores (não mostrar 2027)
            $anosLetivos = AnoLetivo::where(function($q) {
                $q->where('ano', 'like', '%2026%')
                  ->orWhere('ano', '2026')
                  ->orWhere('ano', 'like', '%2025%')
                  ->orWhere('ano', '2025')
                  ->orWhere('ano', 'like', '%2024%')
                  ->orWhere('ano', '2024')
                  ->orWhere('ano', 'like', '%2023%')
                  ->orWhere('ano', '2023');
            })->orderBy('ano', 'desc')->get();
            
            return view('financeiro.index', compact('financeiro', 'situacao', 'anoLetivo', 'anosLetivos', 'anoLetivoId'));
        } elseif ($user->isAdmin() || $user->isSuperadmin() || $user->podeGerir('gerir_mensalidades')) {
            // Filtros
            $turmaId = $request->get('turma_id');
            $classe = $request->get('classe');
            $pesquisa = $request->get('pesquisa');
            
            // Buscar turmas do ano letivo para o filtro
            $turmas = collect();
            $classes = collect();
            
            if ($anoLetivoId) {
                $turmas = Turma::where('ano_letivo_id', $anoLetivoId)
                    ->where('ativa', true)
                    ->orderBy('codigo')
                    ->orderBy('nome')
                    ->get();
                
                // Extrair classes únicas das turmas
                $classes = $turmas->map(function($turma) {
                    // Extrair classe do nome (ex: "10ª A" -> "10ª")
                    if (preg_match('/^(\d+ª)/', $turma->nome, $matches)) {
                        return $matches[1];
                    }
                    // Ou do código (ex: "10A" -> "10")
                    if (preg_match('/^(\d+)/', $turma->codigo, $matches)) {
                        return $matches[1] . 'ª';
                    }
                    return null;
                })->filter()->unique()->sort()->values();
            }
            
            // Query base para alunos
            if ($anoLetivoId) {
                $query = Aluno::whereExists(function($q) use ($anoLetivoId) {
                    $q->select(DB::raw(1))
                      ->from('turma_aluno')
                      ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                      ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
                });
                
                // Filtro por turma
                if ($turmaId) {
                    $query->whereHas('turmas', function($q) use ($turmaId, $anoLetivoId) {
                        $q->where('turmas.id', $turmaId)
                          ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
                    });
                }
                
                // Filtro por classe
                if ($classe) {
                    $query->whereHas('turmas', function($q) use ($classe, $anoLetivoId) {
                        $q->where(function($subQ) use ($classe) {
                            $subQ->where('turmas.nome', 'like', $classe . '%')
                                 ->orWhere('turmas.codigo', 'like', $classe . '%');
                        })
                        ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
                    });
                }
                
                // Filtro por pesquisa (nome)
                if ($pesquisa) {
                    $query->whereHas('user', function($q) use ($pesquisa) {
                        $q->where('name', 'like', '%' . $pesquisa . '%');
                    });
                }
                
                // Otimização: Paginação e select específico
                $alunos = $query->select('alunos.id', 'alunos.codigo_estudante', 'alunos.user_id', 'alunos.modalidade_pagamento', 'alunos.valor_mensalidade_personalizado')
                    ->with('user:id,name,is_active')
                    ->simplePaginate(25)
                    ->appends(request()->query());
            } else {
                $alunos = collect();
            }
            
            // Gerar mensalidades automaticamente para alunos que não têm
            if ($anoLetivoId && $alunos->isNotEmpty()) {
                // Para paginação, gerar apenas para os alunos da página atual
                $this->gerarMensalidadesAutomaticas($anoLetivoId, $alunos);
            }
            
            // Otimização: Buscar mensalidades apenas dos alunos da página atual
            $alunoIds = $alunos->pluck('id')->toArray();
            $mensalidades = Financeiro::where('ano_letivo_id', $anoLetivoId)
                ->where('tipo', 'Mensalidade')
                ->whereIn('aluno_id', $alunoIds)
                ->select('id', 'aluno_id', 'descricao', 'valor', 'data_vencimento', 'status', 'data_pagamento', 'valor_pago', 'tipo')
                ->get()
                ->groupBy('aluno_id');
            
            // Criar estrutura de meses para cada aluno
            $meses = ['Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
                     'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
            
            $situacaoMensalidades = [];
            foreach ($alunos as $aluno) {
                $situacaoMensalidades[$aluno->id] = [];
                for ($mes = 1; $mes <= 12; $mes++) {
                    // Buscar mensalidade por mês de vencimento também
                    $mensalidade = $mensalidades->get($aluno->id)?->first(function($m) use ($mes) {
                        // Verificar por descrição
                        if (strpos($m->descricao, "/{$mes}/") !== false || 
                            strpos($m->descricao, "{$mes}/") !== false ||
                            strpos($m->descricao, "-{$mes}-") !== false) {
                            return true;
                        }
                        // Verificar por mês de vencimento
                        $mesVencimento = (int) $m->data_vencimento->format('m');
                        return $mesVencimento === $mes;
                    });
                    
                    // Obter valor correto da mensalidade (usar valor personalizado se existir)
                    $valorMensalidadeCorreto = $this->financeiroService->getValorMensalidadePorClasse($aluno, $anoLetivoId);
                    
                    // Se há mensalidade, usar o valor dela, mas se o aluno tem valor personalizado diferente e a mensalidade não está paga, usar o valor correto
                    $valorParaUsar = $valorMensalidadeCorreto;
                    if ($mensalidade) {
                        // Se a mensalidade já está paga, usar o valor original
                        if ($mensalidade->status === 'PAGO') {
                            $valorParaUsar = $mensalidade->valor;
                        } else {
                            // Se não está paga, usar o valor correto (pode ser o personalizado)
                            $valorParaUsar = $valorMensalidadeCorreto;
                            // Atualizar a mensalidade se o valor mudou
                            if (abs($mensalidade->valor - $valorMensalidadeCorreto) > 0.01) {
                                $mensalidade->valor = $valorMensalidadeCorreto;
                                $mensalidade->save();
                            }
                        }
                    }
                    
                    $multa = 0;
                    $valorComMulta = $valorParaUsar;
                    if ($mensalidade && $mensalidade->tipo === 'Mensalidade') {
                        // Salvar valor original temporariamente
                        $valorOriginal = $mensalidade->valor;
                        // Atualizar temporariamente para calcular multa
                        $mensalidade->valor = $valorParaUsar;
                        $multa = $this->financeiroService->calcularMulta($mensalidade);
                        $valorComMulta = $this->financeiroService->getValorComMulta($mensalidade);
                        // Restaurar valor original se não foi atualizado acima
                        if ($mensalidade->status === 'PAGO' || abs($mensalidade->valor - $valorOriginal) < 0.01) {
                            $mensalidade->valor = $valorOriginal;
                        }
                    }
                    
                    $situacaoMensalidades[$aluno->id][$mes] = [
                        'mes' => $meses[$mes - 1],
                        'status' => $mensalidade ? $mensalidade->status : 'PENDENTE',
                        'valor' => $valorParaUsar,
                        'multa' => $multa,
                        'valor_com_multa' => $valorComMulta ?: $valorParaUsar,
                        'data_vencimento' => $mensalidade ? $mensalidade->data_vencimento : null,
                        'data_pagamento' => $mensalidade ? $mensalidade->data_pagamento : null,
                        'financeiro_id' => $mensalidade ? $mensalidade->id : null,
                        'tem_gateway' => $mensalidade ? $this->temPagamentoGateway($mensalidade->id) : false,
                        'descricao' => $mensalidade ? $mensalidade->descricao : null,
                    ];
                }
            }
            
            $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
            
            return view('financeiro.index', compact(
                'alunos', 
                'situacaoMensalidades', 
                'anoLetivo', 
                'anoLetivoId', 
                'anosLetivos', 
                'meses',
                'turmas',
                'classes',
                'turmaId',
                'classe',
                'pesquisa'
            ));
        }

        abort(403);
    }

    public function create()
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $alunos = Aluno::with('user')->get();

        return view('financeiro.create', compact('anoLetivo', 'alunos'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $request->validate([
            'aluno_id' => 'required|exists:alunos,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'tipo' => 'required|in:Mensalidade,Taxa,Outro',
            'descricao' => 'required|string',
            'valor' => 'required|numeric|min:0',
            'data_vencimento' => 'required|date',
        ]);

        Financeiro::create($request->all());

        return redirect()->route('financeiro.index')->with('success', 'Item financeiro criado com sucesso!');
    }

    public function pagar(Request $request, Financeiro $financeiro)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        // Verificar se já existe pagamento via gateway (Emola/M-Pesa)
        $pagamentoGateway = Payment::where('financeiro_id', $financeiro->id)
            ->whereIn('metodo', ['EMOLA', 'MPESA'])
            ->where('status', 'APROVADO')
            ->first();

        if ($pagamentoGateway) {
            return redirect()->back()
                ->withErrors(['error' => 'Este pagamento foi realizado via gateway (Emola/M-Pesa) e não pode ser alterado manualmente.']);
        }

        // Verificar se já existe factura paga via gateway
        $facturaGateway = Invoice::whereHas('payment', function($q) use ($financeiro) {
            $q->where('financeiro_id', $financeiro->id)
              ->whereIn('metodo', ['EMOLA', 'MPESA'])
              ->where('status', 'APROVADO');
        })->where('status', 'PAGA')->first();

        if ($facturaGateway) {
            return redirect()->back()
                ->withErrors(['error' => 'Esta factura foi paga via gateway (Emola/M-Pesa) e não pode ser alterada manualmente.']);
        }

        // Calcular valor com multa se aplicável
        $valorComMulta = $financeiro->valor;
        if ($financeiro->tipo === 'Mensalidade') {
            $multa = $this->financeiroService->calcularMulta($financeiro);
            $valorComMulta = $this->financeiroService->getValorComMulta($financeiro);
        }

        $request->validate([
            'valor_pago' => 'required|numeric|min:0|max:' . $valorComMulta,
            'gerar_factura' => 'nullable|boolean',
        ]);

        // Validar pagamento sequencial: só pode pagar meses futuros se os anteriores estiverem pagos
        if ($financeiro->tipo === 'Mensalidade') {
            try {
                $this->validarPagamentoSequencial($financeiro);
            } catch (\Exception $e) {
                return redirect()->back()
                    ->withErrors(['error' => $e->getMessage()]);
            }
        }

        DB::beginTransaction();
        try {
            // Registrar pagamento (incluindo multa se houver)
            $this->financeiroService->registrarPagamento(
                $financeiro,
                $request->valor_pago,
                now()
            );

            // Criar pagamento presencial
            $payment = Payment::create([
                'aluno_id' => $financeiro->aluno_id,
                'financeiro_id' => $financeiro->id,
                'reference' => 'PRES-' . time() . '-' . $financeiro->id,
                'metodo' => 'OUTRO',
                'valor' => $request->valor_pago,
                'status' => 'APROVADO',
                'data_pagamento' => now(),
                'observacoes' => 'Pagamento presencial registrado por ' . $user->name,
            ]);

            // Gerar factura se solicitado ou se não existir factura para este financeiro
            $invoice = null;
            if ($request->filled('gerar_factura') && $request->gerar_factura || !Invoice::where('payment_id', $payment->id)->exists()) {
                $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
                
                $invoice = Invoice::create([
                    'aluno_id' => $financeiro->aluno_id,
                    'payment_id' => $payment->id,
                    'ano_letivo_id' => $financeiro->ano_letivo_id ?? $anoLetivoAtivo->id,
                    'numero_factura' => Invoice::gerarNumeroFactura(),
                    'codigo_verificacao' => Invoice::gerarCodigoVerificacao(),
                    'valor' => $request->valor_pago,
                    'status' => 'PAGA',
                    'data_emissao' => now(),
                    'data_vencimento' => $financeiro->data_vencimento,
                    'descricao' => $financeiro->descricao,
                    'itens' => [
                        [
                            'descricao' => $financeiro->descricao,
                            'valor' => $request->valor_pago,
                        ]
                    ],
                ]);

                // Atualizar payment com invoice_id
                $payment->invoice_id = $invoice->id;
                $payment->save();
            }

            DB::commit();
            
            // Criar notificações para pagamento presencial
            $this->criarNotificacoesPagamentoPresencial($payment);

            return redirect()->route('financeiro.index', ['ano_letivo_id' => $financeiro->ano_letivo_id])
                ->with('success', 'Pagamento registrado com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();
        return redirect()->back()
            ->withErrors(['error' => 'Erro ao registrar pagamento: ' . $e->getMessage()]);
        }
    }

    /**
     * Pagar mensalidade criando automaticamente se não existir
     */
    public function pagarAlunoMes(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $request->validate([
            'aluno_id' => 'required|exists:alunos,id',
            'mes' => 'required|integer|min:1|max:12',
            'valor_pago' => 'required|numeric|min:0',
            'gerar_factura' => 'nullable|boolean',
        ]);

        $aluno = Aluno::findOrFail($request->aluno_id);
        // Recarregar o aluno para garantir que temos os dados mais recentes
        $aluno->refresh();
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $mes = (int) $request->mes;
        $ano = (int) date('Y');
        
        // Verificar se já existe mensalidade para este mês
        $financeiro = Financeiro::where('aluno_id', $aluno->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->where('tipo', 'Mensalidade')
            ->whereMonth('data_vencimento', $mes)
            ->whereYear('data_vencimento', $ano)
            ->first();
        
        // Se não existe, criar
        if (!$financeiro) {
            $modalidade = $aluno->modalidade_pagamento ?? 'MENSAL';
            // Usar o valor personalizado se existir, senão usar o valor padrão da classe
            $valorMensal = $this->financeiroService->getValorMensalidadePorClasse($aluno, $anoLetivo->id);
            
            if ($valorMensal <= 0) {
                return redirect()->back()
                    ->withErrors(['error' => 'Valor da mensalidade não configurado para este aluno.']);
            }
            
            $dataVencimento = \Carbon\Carbon::create($ano, $mes, 5);
            
            $financeiro = Financeiro::create([
                'aluno_id' => $aluno->id,
                'ano_letivo_id' => $anoLetivo->id,
                'tipo' => 'Mensalidade',
                'descricao' => "Mensalidade {$mes}/{$ano}",
                'valor' => $valorMensal,
                'data_vencimento' => $dataVencimento,
                'status' => 'PENDENTE',
            ]);
            
            // Validar pagamento sequencial antes de processar (para mensalidade recém-criada)
            if ($financeiro->tipo === 'Mensalidade') {
                try {
                    $this->validarPagamentoSequencial($financeiro);
                } catch (\Exception $e) {
                    // Se a validação falhar, deletar a mensalidade recém-criada
                    $financeiro->delete();
                    return redirect()->back()
                        ->withErrors(['error' => $e->getMessage()]);
                }
            }
        } else {
            // Se a mensalidade já existe, atualizar o valor se o aluno tiver valor personalizado diferente
            $valorAtualizado = $this->financeiroService->getValorMensalidadePorClasse($aluno, $anoLetivo->id);
            if ($valorAtualizado > 0 && abs($financeiro->valor - $valorAtualizado) > 0.01) {
                // Atualizar o valor da mensalidade existente se ainda não foi paga
                if ($financeiro->status !== 'PAGO') {
                    $financeiro->valor = $valorAtualizado;
                    $financeiro->save();
                }
            }
            
            // Validar pagamento sequencial antes de processar
            if ($financeiro->tipo === 'Mensalidade') {
                try {
                    $this->validarPagamentoSequencial($financeiro);
                } catch (\Exception $e) {
                    return redirect()->back()
                        ->withErrors(['error' => $e->getMessage()]);
                }
            }
        }
        
        // Redirecionar para o método pagar normal
        return $this->pagar($request, $financeiro);
    }

    /**
     * Buscar alunos para autocomplete
     */
    public function buscarAlunos(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $pesquisa = $request->get('q', '');
        $anoLetivoId = $request->get('ano_letivo_id');

        if (empty($pesquisa) || strlen($pesquisa) < 2) {
            return response()->json([]);
        }

        $query = Aluno::whereHas('user', function($q) use ($pesquisa) {
            $q->where('name', 'like', '%' . $pesquisa . '%');
        });

        if ($anoLetivoId) {
            $query->whereExists(function($q) use ($anoLetivoId) {
                $q->select(DB::raw(1))
                  ->from('turma_aluno')
                  ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                  ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
            });
        }

        $alunos = $query->with('user')
            ->limit(10)
            ->get()
            ->map(function($aluno) {
                return [
                    'id' => $aluno->id,
                    'nome' => $aluno->user->name ?? 'Aluno',
                    'codigo' => $aluno->codigo_estudante,
                ];
            });

        return response()->json($alunos);
    }

    // Gerar mensalidades automaticamente
    public function gerarMensalidades(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $request->validate([
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'valor_mensalidade' => 'required|numeric|min:0',
            'mes_inicio' => 'required|integer|min:1|max:12',
            'mes_fim' => 'required|integer|min:1|max:12',
            'dia_vencimento' => 'required|integer|min:1|max:31',
            'turma_id' => 'nullable|exists:turmas,id',
        ]);

        $anoLetivo = AnoLetivo::findOrFail($request->ano_letivo_id);
        $alunos = Aluno::whereHas('turmas', function($q) use ($request) {
            $q->where('turma_aluno.ano_letivo_id', $request->ano_letivo_id);
            if ($request->filled('turma_id')) {
                $q->where('turmas.id', $request->turma_id);
            }
        })->with('user')->get();

        $ano = date('Y', strtotime($anoLetivo->data_inicio));
        $geradas = 0;
        $erros = [];

        foreach ($alunos as $aluno) {
            // Usar valor personalizado do aluno se existir, senão usar o valor padrão
            $valorMensalidade = $aluno->valor_mensalidade_personalizado ?? $request->valor_mensalidade;
            $modalidade = $aluno->modalidade_pagamento ?? 'MENSAL';
            
            // Determinar quais meses gerar baseado na modalidade
            $mesesParaGerar = $this->determinarMesesPorModalidade($modalidade, $request->mes_inicio, $request->mes_fim);
            
            foreach ($mesesParaGerar as $mes) {
                try {
                    $dataVencimento = date('Y-m-d', mktime(0, 0, 0, $mes, $request->dia_vencimento, $ano));
                    
                    // Verificar se já existe mensalidade para este aluno, mês e ano
                    $existe = Financeiro::where('aluno_id', $aluno->id)
                        ->where('ano_letivo_id', $request->ano_letivo_id)
                        ->where('tipo', 'Mensalidade')
                        ->where('descricao', 'like', "%{$mes}/{$ano}%")
                        ->exists();
                    
                    if (!$existe) {
                        // Calcular valor baseado na modalidade
                        $valorFinal = $this->calcularValorPorModalidade($valorMensalidade, $modalidade, $mes, $request->mes_inicio);
                        
                        Financeiro::create([
                            'aluno_id' => $aluno->id,
                            'ano_letivo_id' => $request->ano_letivo_id,
                            'tipo' => 'Mensalidade',
                            'descricao' => $this->gerarDescricaoMensalidade($modalidade, $mes, $ano),
                            'valor' => $valorFinal,
                            'data_vencimento' => $dataVencimento,
                            'status' => 'PENDENTE',
                        ]);
                        $geradas++;
                    }
                } catch (\Exception $e) {
                    $erros[] = "Erro ao gerar mensalidade para " . ($aluno->user->name ?? 'Aluno') . " - {$mes}/{$ano}: " . $e->getMessage();
                }
            }
        }

        $mensagem = "Mensalidades geradas com sucesso! {$geradas} mensalidade(s) criada(s).";
        if (!empty($erros)) {
            $mensagem .= " Erros: " . implode('; ', array_slice($erros, 0, 5));
        }

        return redirect()->route('financeiro.index')
            ->with('success', $mensagem)
            ->with('erros', $erros);
    }

    public function criarMensalidades()
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->orderBy('codigo')
            ->orderBy('nome')
            ->get();

        return view('financeiro.gerar-mensalidades', compact('anoLetivo', 'anosLetivos', 'turmas'));
    }

    // Bloquear alunos com pagamento pendente
    public function bloquearAlunosPendentes(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $request->validate([
            'aluno_ids' => 'nullable|array',
            'aluno_ids.*' => 'exists:alunos,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'bloquear_todos' => 'nullable|boolean',
        ]);

        $anoLetivoId = $request->ano_letivo_id;
        $bloqueados = 0;

        if ($request->filled('bloquear_todos') && $request->bloquear_todos) {
            // Bloquear todos os alunos com mensalidades pendentes
            $alunosComPendentes = Aluno::whereHas('financeiro', function($q) use ($anoLetivoId) {
                $q->where('ano_letivo_id', $anoLetivoId)
                  ->where('tipo', 'Mensalidade')
                  ->where('status', '!=', 'PAGO');
            })->with('user')->get();

            foreach ($alunosComPendentes as $aluno) {
                if ($aluno->user) {
                    $aluno->user->update(['is_active' => false]);
                    $bloqueados++;
                }
            }
        } else {
            // Bloquear apenas alunos selecionados
            foreach ($request->aluno_ids ?? [] as $alunoId) {
                $aluno = Aluno::find($alunoId);
                if ($aluno && $aluno->user) {
                    $aluno->user->update(['is_active' => false]);
                    $bloqueados++;
                }
            }
        }

        return redirect()->route('financeiro.index', ['ano_letivo_id' => $anoLetivoId])
            ->with('success', "{$bloqueados} aluno(s) bloqueado(s) com sucesso!");
    }

    /**
     * Verificar se há pagamento via gateway
     */
    public function verificarGateway(Financeiro $financeiro)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $pagamentoGateway = Payment::where('financeiro_id', $financeiro->id)
            ->whereIn('metodo', ['EMOLA', 'MPESA'])
            ->where('status', 'APROVADO')
            ->exists();

        $facturaGateway = Invoice::whereHas('payment', function($q) use ($financeiro) {
            $q->where('financeiro_id', $financeiro->id)
              ->whereIn('metodo', ['EMOLA', 'MPESA'])
              ->where('status', 'APROVADO');
        })->where('status', 'PAGA')->exists();

        return response()->json([
            'tem_gateway' => $pagamentoGateway || $facturaGateway
        ]);
    }

    /**
     * Exibir página de configuração de valores e modalidades
     */
    public function configurarMensalidadesIndex(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        // Buscar ano letivo
        $anoLetivoId = $request->get('ano_letivo_id');
        if (!$anoLetivoId) {
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();
            $anoLetivoId = $anoLetivo ? $anoLetivo->id : null;
        } else {
            $anoLetivo = AnoLetivo::find($anoLetivoId);
        }

        // Filtros
        $turmaId = $request->get('turma_id');
        $classe = $request->get('classe');
        $pesquisa = $request->get('pesquisa');
        
        // Buscar turmas do ano letivo para o filtro
        $turmas = collect();
        $classes = collect();
        
        if ($anoLetivoId) {
            $turmas = Turma::where('ano_letivo_id', $anoLetivoId)
                ->where('ativa', true)
                ->orderBy('codigo')
                ->orderBy('nome')
                ->get();
            
            // Extrair classes únicas das turmas
            $classes = $turmas->map(function($turma) {
                if (preg_match('/^(\d+ª)/', $turma->nome, $matches)) {
                    return $matches[1];
                }
                if (preg_match('/^(\d+)/', $turma->codigo, $matches)) {
                    return $matches[1] . 'ª';
                }
                return null;
            })->filter()->unique()->sort()->values();
        }
        
        // Query base para alunos
        $alunos = collect();
        if ($anoLetivoId) {
            $query = Aluno::whereExists(function($q) use ($anoLetivoId) {
                $q->select(DB::raw(1))
                  ->from('turma_aluno')
                  ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                  ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
            });
            
            // Filtro por turma
            if ($turmaId) {
                $query->whereHas('turmas', function($q) use ($turmaId, $anoLetivoId) {
                    $q->where('turmas.id', $turmaId)
                      ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
                });
            }
            
            // Filtro por classe
            if ($classe) {
                $query->whereHas('turmas', function($q) use ($classe, $anoLetivoId) {
                    $q->where(function($subQ) use ($classe) {
                        $subQ->where('turmas.nome', 'like', $classe . '%')
                             ->orWhere('turmas.codigo', 'like', $classe . '%');
                    })
                    ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
                });
            }
            
            // Filtro por pesquisa (nome)
            if ($pesquisa) {
                $query->whereHas('user', function($q) use ($pesquisa) {
                    $q->where('name', 'like', '%' . $pesquisa . '%');
                });
            }
            
            $alunos = $query->with(['user', 'turmas' => function($q) use ($anoLetivoId) {
                $q->wherePivot('ano_letivo_id', $anoLetivoId);
            }])
            ->orderBy('codigo_estudante')
            ->paginate(25)
            ->appends(request()->query());
        }
        
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        return view('financeiro.configurar-mensalidades', compact(
            'alunos',
            'anoLetivoId',
            'anoLetivo',
            'anosLetivos',
            'turmas',
            'classes',
            'turmaId',
            'classe',
            'pesquisa'
        ));
    }

    /**
     * Salvar configurações de valores e modalidades de pagamento
     */
    public function configurarMensalidades(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $request->validate([
            'alunos' => 'nullable|array',
            'alunos.*' => 'exists:alunos,id',
            'valores' => 'nullable|array',
            'valores.*' => 'nullable|numeric|min:0',
            'modalidades' => 'nullable|array',
            'modalidades.*' => 'nullable|in:MENSAL,TRIMESTRAL,SEMESTRAL,ANUAL',
        ]);

        // Verificar se pelo menos um aluno foi selecionado
        if (empty($request->alunos) || !is_array($request->alunos)) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['alunos' => 'Selecione pelo menos um aluno para atualizar.']);
        }

        $atualizados = 0;
        
        foreach ($request->alunos as $alunoId) {
            $aluno = Aluno::find($alunoId);
            if ($aluno) {
                if (isset($request->valores[$alunoId]) && $request->valores[$alunoId] !== '') {
                    $aluno->valor_mensalidade_personalizado = $request->valores[$alunoId];
                } else {
                    $aluno->valor_mensalidade_personalizado = null;
                }
                
                if (isset($request->modalidades[$alunoId])) {
                    $aluno->modalidade_pagamento = $request->modalidades[$alunoId];
                }
                
                $aluno->save();
                $atualizados++;
            }
        }

        $anoLetivoId = $request->get('ano_letivo_id');
        
        return redirect()->route('financeiro.configurar-mensalidades', ['ano_letivo_id' => $anoLetivoId])
            ->with('success', "Configurações atualizadas para {$atualizados} aluno(s).");
    }

    /**
     * Visualizar facturas de um aluno específico
     */
    public function facturasAluno(Aluno $aluno)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $aluno->load('user');
        $invoices = $aluno->invoices()
            ->with('payment')
            ->orderBy('created_at', 'desc')
            ->simplePaginate(25)
            ->appends(request()->query());

        $totalInvoices = $aluno->invoices()->sum('valor');

        // Preparar logotipo para as views
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoUrl = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoUrl = asset('storage/' . $logotipo);
        }

        return view('financeiro.facturas-aluno', compact('aluno', 'invoices', 'totalInvoices', 'logotipoUrl'));
    }

    /**
     * Download PDF da factura (admin)
     */
    public function downloadFacturaAluno(Invoice $invoice)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $invoice->load(['aluno.user', 'payment', 'anoLetivo']);

        // Preparar logotipo em base64 para o PDF
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $logotipoBase64 = 'data:' . mime_content_type($logotipoPath) . ';base64,' . base64_encode($logotipoData);
        }

        $pdf = Pdf::loadView('aluno.invoice.pdf', compact('invoice', 'logotipoBase64'));
        
        return $pdf->download('factura-' . $invoice->numero_factura . '.pdf');
    }

    /**
     * Visualizar factura (retorna HTML para modal)
     */
    public function visualizarFactura(Invoice $invoice)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $invoice->load(['aluno.user', 'payment', 'anoLetivo']);

        // Preparar logotipo para a view
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoUrl = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoUrl = asset('storage/' . $logotipo);
        }

        // Se for requisição AJAX (modal), retornar apenas o conteúdo
        if (request()->ajax() || request()->wantsJson() || request()->header('X-Requested-With') === 'XMLHttpRequest') {
            return view('aluno.invoice._content', compact('invoice', 'logotipoUrl'));
        }

        return view('aluno.invoice.show', compact('invoice', 'logotipoUrl'));
    }

    /**
     * Relatório Financeiro
     */
    public function relatorioFinanceiro(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        // Filtros
        $anoLetivoId = $request->get('ano_letivo_id');
        $dataInicio = $request->get('data_inicio');
        $dataFim = $request->get('data_fim');
        $tipoPeriodo = $request->get('tipo_periodo', 'mensal'); // mensal, trimestral, semestral, anual
        // Sempre usar mensal para a view simplificada
        $tipoPeriodo = 'mensal';
        $turmaId = $request->get('turma_id');
        $nivelEnsino = $request->get('nivel_ensino');
        $agruparPor = $request->get('agrupar_por', 'geral'); // geral, turma, classe, nivel_ensino
        $section = $request->get('section');
        $dadosCarregados = false;

        // Buscar anos letivos
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        // Buscar turmas para filtro
        $turmas = collect();
        if ($anoLetivoId) {
            $turmas = Turma::where('ano_letivo_id', $anoLetivoId)
                ->where('ativa', true)
                ->orderBy('nome')
                ->get();
        }
        
        // Se não houver filtro de ano letivo, usar 2026 como padrão
        if (!$anoLetivoId) {
            $anoLetivoPadrao = AnoLetivo::getAnoLetivoAtual();
            if ($anoLetivoPadrao) {
            $anoLetivoId = $anoLetivoPadrao->id;
            }
        }

        $alunoIdsFiltro = null;

        // Funções de filtro para reaproveitar queries sem colidir selects/agregações
        $applyFinanceiroFilters = function($query) use ($anoLetivoId, $alunoIdsFiltro, $dataInicio, $dataFim) {
            if ($anoLetivoId) {
                $query->where('ano_letivo_id', $anoLetivoId);
            }
            if ($alunoIdsFiltro) {
                $query->whereIn('aluno_id', $alunoIdsFiltro);
            }
            if ($dataInicio) {
                $query->where('data_vencimento', '>=', $dataInicio);
            }
            if ($dataFim) {
                $query->where('data_vencimento', '<=', $dataFim);
            }
            return $query;
        };

        $applyTaxaFilters = function($query) use ($anoLetivoId, $alunoIdsFiltro, $dataInicio, $dataFim) {
            $query->where('tipo', 'Taxa');
            if ($anoLetivoId) {
                $query->where('ano_letivo_id', $anoLetivoId);
            }
            if ($alunoIdsFiltro) {
                $query->whereIn('aluno_id', $alunoIdsFiltro);
            }
            if ($dataInicio) {
                $query->where('data_vencimento', '>=', $dataInicio);
            }
            if ($dataFim) {
                $query->where('data_vencimento', '<=', $dataFim);
            }
            return $query;
        };
        $queryPagamentos = Payment::select('id', 'aluno_id', 'financeiro_id', 'valor', 'status', 'data_pagamento')
            ->with(['financeiro:id,aluno_id', 'financeiro.aluno:id,modalidade_pagamento']);
        $queryPagamentosBase = Payment::query();
        $queryFacturasBase = Invoice::query();

        $alunoIdsFiltro = null;

        if ($anoLetivoId) {
            $queryFacturasBase->where('ano_letivo_id', $anoLetivoId);
        }

        // Filtro por turma (tem prioridade sobre nível de ensino)
        if ($turmaId) {
            $alunoIdsFiltro = DB::table('turma_aluno')
                ->where('turma_id', $turmaId)
                ->where('ano_letivo_id', $anoLetivoId)
                ->pluck('aluno_id');
            
            // Quando uma turma é selecionada, definir automaticamente o nível de ensino
            if (!$nivelEnsino) {
                $turma = Turma::find($turmaId);
                if ($turma && $turma->nivel_ensino) {
                    $nivelEnsino = $turma->nivel_ensino;
                }
            }
        }
        // Filtro por nível de ensino (apenas se não houver turma específica selecionada)
        elseif ($nivelEnsino) {
            $turmasNivel = Turma::where('nivel_ensino', $nivelEnsino)
                ->where('ano_letivo_id', $anoLetivoId)
                ->pluck('id');
            
            $alunoIdsFiltro = DB::table('turma_aluno')
                ->whereIn('turma_id', $turmasNivel)
                ->where('ano_letivo_id', $anoLetivoId)
                ->pluck('aluno_id');
        }

        if ($alunoIdsFiltro) {
            $queryPagamentos->whereIn('aluno_id', $alunoIdsFiltro);
            $queryPagamentosBase->whereIn('aluno_id', $alunoIdsFiltro);
            $queryFacturasBase->whereIn('aluno_id', $alunoIdsFiltro);
        }

        if ($dataInicio) {
            $queryPagamentos->where('data_pagamento', '>=', $dataInicio);
            $queryPagamentosBase->where('data_pagamento', '>=', $dataInicio);
            $queryFacturasBase->where('data_emissao', '>=', $dataInicio);
        }

        if ($dataFim) {
            $queryPagamentos->where('data_pagamento', '<=', $dataFim);
            $queryPagamentosBase->where('data_pagamento', '<=', $dataFim);
            $queryFacturasBase->where('data_emissao', '<=', $dataFim);
        }

        $buildFinanceiroMensal = function() use ($applyFinanceiroFilters) {
            return $applyFinanceiroFilters(
                Financeiro::query()
            )
                ->selectRaw('MONTH(data_vencimento) as mes')
                ->selectRaw('SUM(valor) as esperado')
                ->selectRaw('SUM(CASE WHEN status = "PAGO" THEN COALESCE(valor_pago, valor) ELSE 0 END) as pago')
                ->selectRaw('SUM(CASE WHEN status = "VENCIDO" THEN valor ELSE 0 END) as vencido')
                ->selectRaw('SUM(CASE WHEN status NOT IN ("PAGO","VENCIDO") THEN valor ELSE 0 END) as pendente')
                ->groupBy('mes')
                ->get()
                ->keyBy('mes');
        };

        $buildTaxaMensal = function() use ($applyTaxaFilters) {
            return $applyTaxaFilters(
                Financeiro::query()
            )
                ->selectRaw('MONTH(data_vencimento) as mes')
                ->selectRaw('SUM(valor) as esperado')
                ->selectRaw('SUM(CASE WHEN status = "PAGO" THEN COALESCE(valor_pago, valor) ELSE 0 END) as pago')
                ->selectRaw('SUM(CASE WHEN status = "VENCIDO" THEN valor ELSE 0 END) as vencido')
                ->selectRaw('SUM(CASE WHEN status NOT IN ("PAGO","VENCIDO") THEN valor ELSE 0 END) as pendente')
                ->groupBy('mes')
                ->get()
                ->keyBy('mes');
        };

        $buildFacturasMensal = function() use ($queryFacturasBase) {
            return (clone $queryFacturasBase)
                ->selectRaw('MONTH(data_emissao) as mes')
                ->selectRaw('SUM(valor) as total')
                ->selectRaw('SUM(CASE WHEN status = "PAGA" THEN valor ELSE 0 END) as pagas')
                ->groupBy('mes')
                ->get()
                ->keyBy('mes');
        };

        $buildPagamentosMensal = function() use ($queryPagamentosBase) {
            return (clone $queryPagamentosBase)
                ->where('status', 'APROVADO')
                ->whereNotNull('data_pagamento')
                ->selectRaw('MONTH(data_pagamento) as mes')
                ->selectRaw('SUM(valor) as total')
                ->groupBy('mes')
                ->get()
                ->keyBy('mes');
        };

        if (($request->ajax() || $request->get('ajax')) && $section) {
            // Última data sem multa: dia 5. A partir do dia 6 começa a multa.
            $dataLimite = now()->subDays(4);

            if ($section === 'resumo') {
                $totalEsperado = $applyFinanceiroFilters(
                    Financeiro::query()
                )->selectRaw('SUM(valor) as total')->value('total') ?? 0;
                $totalEsperado += $applyTaxaFilters(
                    Financeiro::query()
                )->selectRaw('SUM(valor) as total')->value('total') ?? 0;

                $totalPago = $applyFinanceiroFilters(
                    Financeiro::query()
                )->where('status', 'PAGO')
                 ->selectRaw('SUM(COALESCE(valor_pago, valor)) as total')
                 ->value('total') ?? 0;
                $totalPago += $applyTaxaFilters(
                    Financeiro::query()
                )->where('status', 'PAGO')
                 ->selectRaw('SUM(COALESCE(valor_pago, valor)) as total')
                 ->value('total') ?? 0;

                $totalPendente = $applyFinanceiroFilters(
                    Financeiro::query()
                )->whereIn('status', ['PENDENTE', 'VENCIDO'])
                 ->where('data_vencimento', '<=', $dataLimite)
                 ->selectRaw('SUM(valor) as total')
                 ->value('total') ?? 0;
                $totalPendente += $applyTaxaFilters(
                    Financeiro::query()
                )->whereIn('status', ['PENDENTE', 'VENCIDO'])
                 ->where('data_vencimento', '<=', $dataLimite)
                 ->selectRaw('SUM(valor) as total')
                 ->value('total') ?? 0;

                $totalVencido = $applyFinanceiroFilters(
                    Financeiro::query()
                )->where('status', 'VENCIDO')
                 ->where('data_vencimento', '<=', $dataLimite)
                 ->selectRaw('SUM(valor) as total')
                 ->value('total') ?? 0;
                $totalVencido += $applyTaxaFilters(
                    Financeiro::query()
                )->where('status', 'VENCIDO')
                 ->where('data_vencimento', '<=', $dataLimite)
                 ->selectRaw('SUM(valor) as total')
                 ->value('total') ?? 0;

                return response()->json([
                    'totalEsperado' => $totalEsperado,
                    'totalPago' => $totalPago,
                    'totalPendente' => $totalPendente,
                    'totalVencido' => $totalVencido,
                ]);
            }

            if ($section === 'graficos' || $section === 'periodo' || $section === 'saldo') {
                $financeiroMensal = $buildFinanceiroMensal();
                $taxaMensal = $buildTaxaMensal();
                $facturasMensal = $buildFacturasMensal();
                $pagamentosMensal = $buildPagamentosMensal();

                $dadosPorPeriodo = $this->agruparPorPeriodoSql($financeiroMensal, $taxaMensal, $facturasMensal, $tipoPeriodo);
                $saldoPorMes = $this->calcularSaldoPorMesSql($financeiroMensal, $taxaMensal, $facturasMensal, $pagamentosMensal);

                if ($section === 'periodo') {
                    return response()->json(['dadosPorPeriodo' => $dadosPorPeriodo]);
                }
                if ($section === 'saldo') {
                    return response()->json(['saldoPorMes' => $saldoPorMes]);
                }

                $pagamentosPorModalidade = $this->agruparPagamentosPorModalidadeSql($alunoIdsFiltro, $dataInicio, $dataFim);
                return response()->json([
                    'dadosPorPeriodo' => $dadosPorPeriodo,
                    'saldoPorMes' => $saldoPorMes,
                    'pagamentosPorModalidade' => $pagamentosPorModalidade,
                ]);
            }

            if ($section === 'modalidade') {
                $pagamentosPorModalidade = $this->agruparPagamentosPorModalidadeSql($alunoIdsFiltro, $dataInicio, $dataFim);
                return response()->json(['pagamentosPorModalidade' => $pagamentosPorModalidade]);
            }

            if ($section === 'dividas') {
                $queryDividas = Financeiro::selectRaw('aluno_id')
                    ->selectRaw('users.name as aluno_nome')
                    ->selectRaw('SUM(valor) as total')
                    ->selectRaw('SUM(CASE WHEN status = "PENDENTE" THEN valor ELSE 0 END) as pendente')
                    ->selectRaw('SUM(CASE WHEN status = "VENCIDO" THEN valor ELSE 0 END) as vencido')
                    ->join('alunos', 'alunos.id', '=', 'financeiro.aluno_id')
                    ->join('users', 'users.id', '=', 'alunos.user_id')
                    ->whereIn('financeiro.status', ['PENDENTE', 'VENCIDO']);

                if ($anoLetivoId) {
                    $queryDividas->where('financeiro.ano_letivo_id', $anoLetivoId);
                }
                if ($alunoIdsFiltro) {
                    $queryDividas->whereIn('financeiro.aluno_id', $alunoIdsFiltro);
                }
                if ($dataInicio) {
                    $queryDividas->where('financeiro.data_vencimento', '>=', $dataInicio);
                }
                if ($dataFim) {
                    $queryDividas->where('financeiro.data_vencimento', '<=', $dataFim);
                }

                $dividasPorAluno = $queryDividas
                    ->groupBy('aluno_id', 'users.name')
                    ->orderByDesc('total')
                    ->get()
                    ->map(function($item) {
                        return [
                            'aluno_nome' => $item->aluno_nome,
                            'total' => (float) $item->total,
                            'pendente' => (float) $item->pendente,
                            'vencido' => (float) $item->vencido,
                        ];
                    })
                    ->toArray();

                return response()->json([
                    'dividasPorAluno' => array_slice($dividasPorAluno, 0, 100),
                    'totalDividas' => count($dividasPorAluno),
                ]);
            }

            if ($section === 'facturas') {
                $totalFacturas = (clone $queryFacturasBase)->selectRaw('SUM(valor) as total')->value('total') ?? 0;
                $totalFacturasPagas = (clone $queryFacturasBase)
                    ->where('status', 'PAGA')
                    ->selectRaw('SUM(valor) as total')
                    ->value('total') ?? 0;
                $qtdFacturas = (clone $queryFacturasBase)->count();
                $qtdFacturasPagas = (clone $queryFacturasBase)->where('status', 'PAGA')->count();

                return response()->json([
                    'totalFacturas' => $totalFacturas,
                    'totalFacturasPagas' => $totalFacturasPagas,
                    'qtdFacturas' => $qtdFacturas,
                    'qtdFacturasPagas' => $qtdFacturasPagas,
                ]);
            }

            if ($section === 'agrupado') {
                $dadosAgrupados = $this->agruparDadosFinanceirosSql(
                    $agruparPor,
                    $anoLetivoId,
                    $alunoIdsFiltro,
                    $dataInicio,
                    $dataFim
                );
                return response()->json([
                    'dadosAgrupados' => $dadosAgrupados,
                    'agruparPor' => $agruparPor,
                ]);
            }

            return response()->json([]);
        }

        // Sempre carregar dados básicos (sem precisar de parâmetro carregar)
        $carregarTudo = $request->boolean('carregar') || $request->get('imprimir') || $request->get('pdf') || true;

        // Evitar carregar todas as linhas (usar agregações no BD)
        $financeiros = collect();
        $pagamentos = collect();
        $facturas = collect();

        // Buscar taxas de inscrição
        $taxasInscricao = $applyTaxaFilters(
            Financeiro::query()
        )->select('id', 'aluno_id', 'valor', 'valor_pago', 'status', 'data_vencimento', 'ano_letivo_id')
         ->get();

        // Resumo Geral
        // Considerar apenas mensalidades vencidas há mais de 4 dias para cálculos de dívida (a partir do dia 6)
        // Última data sem multa: dia 5. A partir do dia 6 começa a multa.
        $dataLimite = now()->subDays(4);
        
        $totalEsperado = $applyFinanceiroFilters(
            Financeiro::query()
        )->selectRaw('SUM(valor) as total')->value('total') ?? 0;
        $totalEsperado += $applyTaxaFilters(
            Financeiro::query()
        )->selectRaw('SUM(valor) as total')->value('total') ?? 0;

        $totalPago = $applyFinanceiroFilters(
            Financeiro::query()
        )->where('status', 'PAGO')
         ->selectRaw('SUM(COALESCE(valor_pago, valor)) as total')
         ->value('total') ?? 0;
        $totalPago += $applyTaxaFilters(
            Financeiro::query()
        )->where('status', 'PAGO')
         ->selectRaw('SUM(COALESCE(valor_pago, valor)) as total')
         ->value('total') ?? 0;
        
        // Total pendente: apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
        $totalPendente = $applyFinanceiroFilters(
            Financeiro::query()
        )->whereIn('status', ['PENDENTE', 'VENCIDO'])
         ->where('data_vencimento', '<=', $dataLimite)
         ->selectRaw('SUM(valor) as total')
         ->value('total') ?? 0;
        $totalPendente += $applyTaxaFilters(
            Financeiro::query()
        )->whereIn('status', ['PENDENTE', 'VENCIDO'])
         ->where('data_vencimento', '<=', $dataLimite)
         ->selectRaw('SUM(valor) as total')
         ->value('total') ?? 0;
        
        // Total vencido: apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
        $totalVencido = $applyFinanceiroFilters(
            Financeiro::query()
        )->where('status', 'VENCIDO')
         ->where('data_vencimento', '<=', $dataLimite)
         ->selectRaw('SUM(valor) as total')
         ->value('total') ?? 0;
        $totalVencido += $applyTaxaFilters(
            Financeiro::query()
        )->where('status', 'VENCIDO')
         ->where('data_vencimento', '<=', $dataLimite)
         ->selectRaw('SUM(valor) as total')
         ->value('total') ?? 0;
        
        $totalFacturas = (clone $queryFacturasBase)->selectRaw('SUM(valor) as total')->value('total') ?? 0;
        $totalFacturasPagas = (clone $queryFacturasBase)
            ->where('status', 'PAGA')
            ->selectRaw('SUM(valor) as total')
            ->value('total') ?? 0;
        
        // Total de taxas de inscrição
        $totalTaxaInscricao = $taxasInscricao->sum('valor');
        $totalTaxaInscricaoPaga = $taxasInscricao->where('status', 'PAGO')->sum('valor_pago') ?? $taxasInscricao->where('status', 'PAGO')->sum('valor');

        // Agrupar por período
        // Agregações mensais (financeiro, taxas, facturas, pagamentos)
        $financeiroMensal = $applyFinanceiroFilters(
            Financeiro::query()
        )
            ->selectRaw('MONTH(data_vencimento) as mes')
            ->selectRaw('SUM(valor) as esperado')
            ->selectRaw('SUM(CASE WHEN status = "PAGO" THEN COALESCE(valor_pago, valor) ELSE 0 END) as pago')
            ->selectRaw('SUM(CASE WHEN status = "VENCIDO" THEN valor ELSE 0 END) as vencido')
            ->selectRaw('SUM(CASE WHEN status NOT IN ("PAGO","VENCIDO") THEN valor ELSE 0 END) as pendente')
            ->groupBy('mes')
            ->get()
            ->keyBy('mes');

        $taxaMensal = $applyTaxaFilters(
            Financeiro::query()
        )
            ->selectRaw('MONTH(data_vencimento) as mes')
            ->selectRaw('SUM(valor) as esperado')
            ->selectRaw('SUM(CASE WHEN status = "PAGO" THEN COALESCE(valor_pago, valor) ELSE 0 END) as pago')
            ->selectRaw('SUM(CASE WHEN status = "VENCIDO" THEN valor ELSE 0 END) as vencido')
            ->selectRaw('SUM(CASE WHEN status NOT IN ("PAGO","VENCIDO") THEN valor ELSE 0 END) as pendente')
            ->groupBy('mes')
            ->get()
            ->keyBy('mes');

        $facturasMensal = (clone $queryFacturasBase)
            ->selectRaw('MONTH(data_emissao) as mes')
            ->selectRaw('SUM(valor) as total')
            ->selectRaw('SUM(CASE WHEN status = "PAGA" THEN valor ELSE 0 END) as pagas')
            ->groupBy('mes')
            ->get()
            ->keyBy('mes');

        $pagamentosMensal = (clone $queryPagamentosBase)
            ->where('status', 'APROVADO')
            ->whereNotNull('data_pagamento')
            ->selectRaw('MONTH(data_pagamento) as mes')
            ->selectRaw('SUM(valor) as total')
            ->groupBy('mes')
            ->get()
            ->keyBy('mes');

        // Agrupar por período (mensal/trimestral/semestral/anual)
        $dadosPorPeriodo = $this->agruparPorPeriodoSql($financeiroMensal, $taxaMensal, $facturasMensal, $tipoPeriodo);

        // Saldo por mês
        $saldoPorMes = $this->calcularSaldoPorMesSql($financeiroMensal, $taxaMensal, $facturasMensal, $pagamentosMensal);

        // Pagamentos por modalidade
        $pagamentosPorModalidade = $this->agruparPagamentosPorModalidadeSql($alunoIdsFiltro, $dataInicio, $dataFim);

        // Dívidas por aluno (usando SQL agregado)
        $queryDividas = Financeiro::selectRaw('aluno_id')
            ->selectRaw('users.name as aluno_nome')
            ->selectRaw('SUM(valor) as total')
            ->selectRaw('SUM(CASE WHEN status = "PENDENTE" THEN valor ELSE 0 END) as pendente')
            ->selectRaw('SUM(CASE WHEN status = "VENCIDO" THEN valor ELSE 0 END) as vencido')
            ->join('alunos', 'alunos.id', '=', 'financeiro.aluno_id')
            ->join('users', 'users.id', '=', 'alunos.user_id')
            ->whereIn('financeiro.status', ['PENDENTE', 'VENCIDO']);

        if ($anoLetivoId) {
            $queryDividas->where('financeiro.ano_letivo_id', $anoLetivoId);
        }
        if ($alunoIdsFiltro) {
            $queryDividas->whereIn('financeiro.aluno_id', $alunoIdsFiltro);
        }
        if ($dataInicio) {
            $queryDividas->where('financeiro.data_vencimento', '>=', $dataInicio);
        }
        if ($dataFim) {
            $queryDividas->where('financeiro.data_vencimento', '<=', $dataFim);
        }

        $dividasPorAluno = $queryDividas
            ->groupBy('aluno_id', 'users.name')
            ->orderByDesc('total')
            ->get()
            ->map(function($item) {
                return [
                    'aluno_nome' => $item->aluno_nome,
                    'total' => (float) $item->total,
                    'pendente' => (float) $item->pendente,
                    'vencido' => (float) $item->vencido,
                ];
            })
            ->toArray();

        // Agrupar dados por turma/classe/nível de ensino (SQL)
        $dadosAgrupados = $this->agruparDadosFinanceirosSql(
            $agruparPor,
            $anoLetivoId,
            $alunoIdsFiltro,
            $dataInicio,
            $dataFim
        );

        // Se for requisição AJAX, retornar JSON
        if ($request->ajax() || $request->get('ajax')) {
            return response()->json([
                'dadosPorPeriodo' => $dadosPorPeriodo,
                'saldoPorMes' => $saldoPorMes,
                'pagamentosPorModalidade' => $pagamentosPorModalidade,
                'totalEsperado' => $totalEsperado,
                'totalPago' => $totalPago,
                'totalPendente' => $totalPendente,
                'totalVencido' => $totalVencido,
            ]);
        }

        // Se for requisição de impressão/PDF
        if ($request->get('imprimir') || $request->get('pdf')) {
            return $this->imprimirRelatorioFinanceiro(
                $financeiros, $pagamentos, $facturas, $totalEsperado, $totalPago, 
                $totalPendente, $totalVencido, $totalFacturas, $totalFacturasPagas,
                $dadosPorPeriodo, $saldoPorMes, $pagamentosPorModalidade, 
                $dividasPorAluno, $dadosAgrupados, $agruparPor,
                $anoLetivoId, $turmaId, $nivelEnsino, $tipoPeriodo
            );
        }

        $dadosCarregados = true;
        return view('financeiro.relatorio', compact(
            'anosLetivos',
            'turmas',
            'anoLetivoId',
            'turmaId',
            'nivelEnsino',
            'agruparPor',
            'dataInicio',
            'dataFim',
            'tipoPeriodo',
            'totalEsperado',
            'totalTaxaInscricao',
            'totalTaxaInscricaoPaga',
            'taxasInscricao',
            'totalPago',
            'totalPendente',
            'totalVencido',
            'totalFacturas',
            'totalFacturasPagas',
            'dadosPorPeriodo',
            'saldoPorMes',
            'pagamentosPorModalidade',
            'dividasPorAluno',
            'dadosAgrupados',
            'financeiros',
            'pagamentos',
            'facturas',
            'dadosCarregados'
        ));
    }

    /**
     * Agrupar dados por período
     */
    private function agruparPorPeriodo($financeiros, $pagamentos, $facturas, $tipoPeriodo)
    {
        $dados = [];

        if ($tipoPeriodo === 'mensal') {
            for ($mes = 1; $mes <= 12; $mes++) {
                $dados[$mes] = [
                    'periodo' => $this->nomeMes($mes),
                    'esperado' => 0,
                    'pago' => 0,
                    'pendente' => 0,
                    'vencido' => 0,
                    'facturas' => 0,
                ];
            }

            foreach ($financeiros as $fin) {
                $mes = (int) $fin->data_vencimento->format('m');
                $dados[$mes]['esperado'] += $fin->valor;
                
                if ($fin->status === 'PAGO') {
                    $dados[$mes]['pago'] += $fin->valor_pago ?? $fin->valor;
                } elseif ($fin->status === 'VENCIDO') {
                    $dados[$mes]['vencido'] += $fin->valor;
                } else {
                    $dados[$mes]['pendente'] += $fin->valor;
                }
            }

            foreach ($facturas as $factura) {
                $mes = (int) $factura->data_emissao->format('m');
                $dados[$mes]['facturas'] += $factura->valor;
            }
        } elseif ($tipoPeriodo === 'trimestral') {
            $trimestres = [
                1 => ['nome' => '1º Trimestre', 'meses' => [1, 2, 3]],
                2 => ['nome' => '2º Trimestre', 'meses' => [4, 5, 6]],
                3 => ['nome' => '3º Trimestre', 'meses' => [7, 8, 9]],
                4 => ['nome' => '4º Trimestre', 'meses' => [10, 11, 12]],
            ];

            foreach ($trimestres as $num => $trim) {
                $dados[$num] = [
                    'periodo' => $trim['nome'],
                    'esperado' => 0,
                    'pago' => 0,
                    'pendente' => 0,
                    'vencido' => 0,
                    'facturas' => 0,
                    'taxa_inscricao' => 0,
                ];

                foreach ($financeiros as $fin) {
                    $mes = (int) $fin->data_vencimento->format('m');
                    if (in_array($mes, $trim['meses'])) {
                        $dados[$num]['esperado'] += $fin->valor;
                        
                        if ($fin->status === 'PAGO') {
                            $dados[$num]['pago'] += $fin->valor_pago ?? $fin->valor;
                        } elseif ($fin->status === 'VENCIDO') {
                            $dados[$num]['vencido'] += $fin->valor;
                        } else {
                            $dados[$num]['pendente'] += $fin->valor;
                        }
                    }
                }
                
                // Incluir taxas de inscrição
                foreach ($taxasInscricao as $taxa) {
                    $mes = (int) $taxa->data_vencimento->format('m');
                    if (in_array($mes, $trim['meses'])) {
                        $dados[$num]['esperado'] += $taxa->valor;
                        $dados[$num]['taxa_inscricao'] += $taxa->valor;
                        
                        if ($taxa->status === 'PAGO') {
                            $dados[$num]['pago'] += $taxa->valor_pago ?? $taxa->valor;
                        } elseif ($taxa->status === 'VENCIDO') {
                            $dados[$num]['vencido'] += $taxa->valor;
                        } else {
                            $dados[$num]['pendente'] += $taxa->valor;
                        }
                    }
                }

                foreach ($facturas as $factura) {
                    $mes = (int) $factura->data_emissao->format('m');
                    if (in_array($mes, $trim['meses'])) {
                        $dados[$num]['facturas'] += $factura->valor;
                    }
                }
            }
        } elseif ($tipoPeriodo === 'semestral') {
            $semestres = [
                1 => ['nome' => '1º Semestre', 'meses' => [1, 2, 3, 4, 5, 6]],
                2 => ['nome' => '2º Semestre', 'meses' => [7, 8, 9, 10, 11, 12]],
            ];

            foreach ($semestres as $num => $sem) {
                $dados[$num] = [
                    'periodo' => $sem['nome'],
                    'esperado' => 0,
                    'pago' => 0,
                    'pendente' => 0,
                    'vencido' => 0,
                    'facturas' => 0,
                    'taxa_inscricao' => 0,
                ];

                foreach ($financeiros as $fin) {
                    $mes = (int) $fin->data_vencimento->format('m');
                    if (in_array($mes, $sem['meses'])) {
                        $dados[$num]['esperado'] += $fin->valor;
                        
                        if ($fin->status === 'PAGO') {
                            $dados[$num]['pago'] += $fin->valor_pago ?? $fin->valor;
                        } elseif ($fin->status === 'VENCIDO') {
                            $dados[$num]['vencido'] += $fin->valor;
                        } else {
                            $dados[$num]['pendente'] += $fin->valor;
                        }
                    }
                }
                
                // Incluir taxas de inscrição
                foreach ($taxasInscricao as $taxa) {
                    $mes = (int) $taxa->data_vencimento->format('m');
                    if (in_array($mes, $sem['meses'])) {
                        $dados[$num]['esperado'] += $taxa->valor;
                        $dados[$num]['taxa_inscricao'] += $taxa->valor;
                        
                        if ($taxa->status === 'PAGO') {
                            $dados[$num]['pago'] += $taxa->valor_pago ?? $taxa->valor;
                        } elseif ($taxa->status === 'VENCIDO') {
                            $dados[$num]['vencido'] += $taxa->valor;
                        } else {
                            $dados[$num]['pendente'] += $taxa->valor;
                        }
                    }
                }

                foreach ($facturas as $factura) {
                    $mes = (int) $factura->data_emissao->format('m');
                    if (in_array($mes, $sem['meses'])) {
                        $dados[$num]['facturas'] += $factura->valor;
                    }
                }
            }
        } else { // anual
            $dados[1] = [
                'periodo' => 'Anual',
                'esperado' => $financeiros->sum('valor') + $taxasInscricao->sum('valor'),
                'pago' => ($financeiros->where('status', 'PAGO')->sum('valor_pago') ?? $financeiros->where('status', 'PAGO')->sum('valor')) + 
                          ($taxasInscricao->where('status', 'PAGO')->sum('valor_pago') ?? $taxasInscricao->where('status', 'PAGO')->sum('valor')),
                'pendente' => $financeiros->whereIn('status', ['PENDENTE', 'VENCIDO'])->sum('valor') + 
                              $taxasInscricao->whereIn('status', ['PENDENTE', 'VENCIDO'])->sum('valor'),
                'vencido' => $financeiros->where('status', 'VENCIDO')->sum('valor') + 
                             $taxasInscricao->where('status', 'VENCIDO')->sum('valor'),
                'facturas' => $facturas->sum('valor'),
                'taxa_inscricao' => $taxasInscricao->sum('valor'),
            ];
        }

        return $dados;
    }

    /**
     * Calcular saldo por mês
     */
    private function calcularSaldoPorMes($financeiros, $pagamentos, $facturas)
    {
        $saldo = [];
        
        for ($mes = 1; $mes <= 12; $mes++) {
            $mesFinanceiros = $financeiros->filter(function($fin) use ($mes) {
                return (int) $fin->data_vencimento->format('m') === $mes;
            });
            
            $mesPagamentos = $pagamentos->filter(function($pay) use ($mes) {
                return $pay->data_pagamento && (int) $pay->data_pagamento->format('m') === $mes;
            });
            
            $mesFacturas = $facturas->filter(function($fact) use ($mes) {
                return (int) $fact->data_emissao->format('m') === $mes;
            });

            $entradas = $mesPagamentos->sum('valor') + $mesFacturas->where('status', 'PAGA')->sum('valor');
            $saidas = 0; // Se houver despesas, adicionar aqui
            $esperado = $mesFinanceiros->sum('valor');
            $recebido = $mesFinanceiros->where('status', 'PAGO')->sum('valor_pago') ?? $mesFinanceiros->where('status', 'PAGO')->sum('valor');

            $saldo[$mes] = [
                'mes' => $this->nomeMes($mes),
                'esperado' => $esperado,
                'recebido' => $recebido,
                'entradas' => $entradas,
                'saidas' => $saidas,
                'saldo' => $entradas - $saidas,
                'pendente' => $esperado - $recebido,
            ];
        }

        return $saldo;
    }

    /**
     * Agrupar pagamentos por modalidade
     */
    private function agruparPagamentosPorModalidade($pagamentos, $financeiros)
    {
        $dados = [
            'MENSAL' => ['total' => 0, 'quantidade' => 0],
            'TRIMESTRAL' => ['total' => 0, 'quantidade' => 0],
            'SEMESTRAL' => ['total' => 0, 'quantidade' => 0],
            'ANUAL' => ['total' => 0, 'quantidade' => 0],
        ];

        foreach ($pagamentos as $pagamento) {
            if ($pagamento->financeiro && $pagamento->financeiro->aluno) {
                $modalidade = $pagamento->financeiro->aluno->modalidade_pagamento ?? 'MENSAL';
                if (isset($dados[$modalidade])) {
                    $dados[$modalidade]['total'] += $pagamento->valor;
                    $dados[$modalidade]['quantidade']++;
                }
            }
        }

        return $dados;
    }

    private function agruparPorPeriodoSql($financeiroMensal, $taxaMensal, $facturasMensal, $tipoPeriodo)
    {
        $dados = [];

        $buildMensal = function() use ($financeiroMensal, $taxaMensal, $facturasMensal) {
            $items = [];
            for ($mes = 1; $mes <= 12; $mes++) {
                $fin = $financeiroMensal->get($mes);
                $taxa = $taxaMensal->get($mes);
                $fact = $facturasMensal->get($mes);

                $esperado = ($fin->esperado ?? 0) + ($taxa->esperado ?? 0);
                $pago = ($fin->pago ?? 0) + ($taxa->pago ?? 0);
                $pendente = ($fin->pendente ?? 0) + ($taxa->pendente ?? 0);
                $vencido = ($fin->vencido ?? 0) + ($taxa->vencido ?? 0);

                $items[$mes] = [
                    'periodo' => $this->nomeMes($mes),
                    'esperado' => $esperado,
                    'pago' => $pago,
                    'pendente' => $pendente,
                    'vencido' => $vencido,
                    'facturas' => $fact->total ?? 0,
                    'taxa_inscricao' => $taxa->esperado ?? 0,
                ];
            }
            return $items;
        };

        if ($tipoPeriodo === 'mensal') {
            return $buildMensal();
        }

        if ($tipoPeriodo === 'trimestral') {
            $trimestres = [
                1 => ['nome' => '1º Trimestre', 'meses' => [1, 2, 3]],
                2 => ['nome' => '2º Trimestre', 'meses' => [4, 5, 6]],
                3 => ['nome' => '3º Trimestre', 'meses' => [7, 8, 9]],
                4 => ['nome' => '4º Trimestre', 'meses' => [10, 11, 12]],
            ];
            foreach ($trimestres as $num => $trim) {
                $dados[$num] = [
                    'periodo' => $trim['nome'],
                    'esperado' => 0,
                    'pago' => 0,
                    'pendente' => 0,
                    'vencido' => 0,
                    'facturas' => 0,
                    'taxa_inscricao' => 0,
                ];
                foreach ($trim['meses'] as $mes) {
                    $fin = $financeiroMensal->get($mes);
                    $taxa = $taxaMensal->get($mes);
                    $fact = $facturasMensal->get($mes);
                    $dados[$num]['esperado'] += ($fin->esperado ?? 0) + ($taxa->esperado ?? 0);
                    $dados[$num]['pago'] += ($fin->pago ?? 0) + ($taxa->pago ?? 0);
                    $dados[$num]['pendente'] += ($fin->pendente ?? 0) + ($taxa->pendente ?? 0);
                    $dados[$num]['vencido'] += ($fin->vencido ?? 0) + ($taxa->vencido ?? 0);
                    $dados[$num]['facturas'] += $fact->total ?? 0;
                    $dados[$num]['taxa_inscricao'] += $taxa->esperado ?? 0;
                }
            }
            return $dados;
        }

        if ($tipoPeriodo === 'semestral') {
            $semestres = [
                1 => ['nome' => '1º Semestre', 'meses' => [1, 2, 3, 4, 5, 6]],
                2 => ['nome' => '2º Semestre', 'meses' => [7, 8, 9, 10, 11, 12]],
            ];
            foreach ($semestres as $num => $sem) {
                $dados[$num] = [
                    'periodo' => $sem['nome'],
                    'esperado' => 0,
                    'pago' => 0,
                    'pendente' => 0,
                    'vencido' => 0,
                    'facturas' => 0,
                    'taxa_inscricao' => 0,
                ];
                foreach ($sem['meses'] as $mes) {
                    $fin = $financeiroMensal->get($mes);
                    $taxa = $taxaMensal->get($mes);
                    $fact = $facturasMensal->get($mes);
                    $dados[$num]['esperado'] += ($fin->esperado ?? 0) + ($taxa->esperado ?? 0);
                    $dados[$num]['pago'] += ($fin->pago ?? 0) + ($taxa->pago ?? 0);
                    $dados[$num]['pendente'] += ($fin->pendente ?? 0) + ($taxa->pendente ?? 0);
                    $dados[$num]['vencido'] += ($fin->vencido ?? 0) + ($taxa->vencido ?? 0);
                    $dados[$num]['facturas'] += $fact->total ?? 0;
                    $dados[$num]['taxa_inscricao'] += $taxa->esperado ?? 0;
                }
            }
            return $dados;
        }

        // anual
        $mensal = $buildMensal();
        $esperado = 0;
        $pago = 0;
        $pendente = 0;
        $vencido = 0;
        $facturas = 0;
        $taxa = 0;
        foreach ($mensal as $item) {
            $esperado += $item['esperado'];
            $pago += $item['pago'];
            $pendente += $item['pendente'];
            $vencido += $item['vencido'];
            $facturas += $item['facturas'];
            $taxa += $item['taxa_inscricao'];
        }

        return [
            1 => [
                'periodo' => 'Anual',
                'esperado' => $esperado,
                'pago' => $pago,
                'pendente' => $pendente,
                'vencido' => $vencido,
                'facturas' => $facturas,
                'taxa_inscricao' => $taxa,
            ]
        ];
    }

    private function calcularSaldoPorMesSql($financeiroMensal, $taxaMensal, $facturasMensal, $pagamentosMensal)
    {
        $saldo = [];

        for ($mes = 1; $mes <= 12; $mes++) {
            $fin = $financeiroMensal->get($mes);
            $taxa = $taxaMensal->get($mes);
            $fact = $facturasMensal->get($mes);
            $pay = $pagamentosMensal->get($mes);

            $esperado = ($fin->esperado ?? 0) + ($taxa->esperado ?? 0);
            $recebido = ($fin->pago ?? 0) + ($taxa->pago ?? 0);
            $entradas = ($pay->total ?? 0) + ($fact->pagas ?? 0);
            $saidas = 0;

            $saldo[$mes] = [
                'mes' => $this->nomeMes($mes),
                'esperado' => $esperado,
                'recebido' => $recebido,
                'entradas' => $entradas,
                'saidas' => $saidas,
                'saldo' => $entradas - $saidas,
                'pendente' => $esperado - $recebido,
            ];
        }

        return $saldo;
    }

    private function agruparPagamentosPorModalidadeSql($alunoIdsFiltro, $dataInicio, $dataFim)
    {
        $query = Payment::selectRaw('COALESCE(alunos.modalidade_pagamento, "MENSAL") as modalidade')
            ->selectRaw('COUNT(*) as quantidade')
            ->selectRaw('SUM(payments.valor) as total')
            ->join('alunos', 'alunos.id', '=', 'payments.aluno_id')
            ->where('payments.status', 'APROVADO');

        if ($alunoIdsFiltro) {
            $query->whereIn('payments.aluno_id', $alunoIdsFiltro);
        }
        if ($dataInicio) {
            $query->where('payments.data_pagamento', '>=', $dataInicio);
        }
        if ($dataFim) {
            $query->where('payments.data_pagamento', '<=', $dataFim);
        }

        $result = $query->groupBy('modalidade')->get();

        $dados = [
            'MENSAL' => ['total' => 0, 'quantidade' => 0],
            'TRIMESTRAL' => ['total' => 0, 'quantidade' => 0],
            'SEMESTRAL' => ['total' => 0, 'quantidade' => 0],
            'ANUAL' => ['total' => 0, 'quantidade' => 0],
        ];

        foreach ($result as $row) {
            $mod = $row->modalidade ?? 'MENSAL';
            if (!isset($dados[$mod])) {
                $dados[$mod] = ['total' => 0, 'quantidade' => 0];
            }
            $dados[$mod]['total'] = (float) $row->total;
            $dados[$mod]['quantidade'] = (int) $row->quantidade;
        }

        return $dados;
    }

    private function calcularDividasPorAlunoSql($financeiros)
    {
        $dividas = [];
        foreach ($financeiros as $fin) {
            $dividas[] = [
                'aluno_nome' => $fin->aluno_nome,
                'total' => (float) $fin->total,
                'pendente' => (float) $fin->pendente,
                'vencido' => (float) $fin->vencido,
            ];
        }
        return $dividas;
    }

    private function agruparDadosFinanceirosSql($agruparPor, $anoLetivoId, $alunoIdsFiltro, $dataInicio, $dataFim)
    {
        if ($agruparPor === 'geral') {
            return [];
        }

        $groupSelect = '';
        $groupLabel = '';
        $groupByColumns = [];

        if ($agruparPor === 'turma') {
            $groupSelect = 't.id as chave';
            $groupLabel = 'CONCAT(t.codigo, " - ", t.nome) as label';
            $groupByColumns = ['t.id', 't.codigo', 't.nome'];
        } elseif ($agruparPor === 'classe') {
            $groupSelect = 't.codigo as chave';
            $groupLabel = 'CONCAT("Classe ", t.codigo) as label';
            $groupByColumns = ['t.codigo'];
        } else {
            $groupSelect = 't.nivel_ensino as chave';
            $groupLabel = 'CASE WHEN t.nivel_ensino = "PRIMARIO" THEN "Ensino Primário" ELSE "Ensino Secundário" END as label';
            $groupByColumns = ['t.nivel_ensino'];
        }

        $financeiroQuery = Financeiro::selectRaw($groupSelect)
            ->selectRaw($groupLabel)
            ->selectRaw('SUM(financeiro.valor) as esperado')
            ->selectRaw('SUM(CASE WHEN financeiro.status = "PAGO" THEN COALESCE(financeiro.valor_pago, financeiro.valor) ELSE 0 END) as pago')
            ->selectRaw('SUM(CASE WHEN financeiro.status = "VENCIDO" THEN financeiro.valor ELSE 0 END) as vencido')
            ->selectRaw('SUM(CASE WHEN financeiro.status NOT IN ("PAGO","VENCIDO") THEN financeiro.valor ELSE 0 END) as pendente')
            ->join('turma_aluno as ta', function($join) use ($anoLetivoId) {
                $join->on('ta.aluno_id', '=', 'financeiro.aluno_id');
                if ($anoLetivoId) {
                    $join->where('ta.ano_letivo_id', '=', $anoLetivoId);
                }
            })
            ->join('turmas as t', 't.id', '=', 'ta.turma_id');

        if ($anoLetivoId) {
            $financeiroQuery->where('financeiro.ano_letivo_id', $anoLetivoId);
        }
        if ($alunoIdsFiltro) {
            $financeiroQuery->whereIn('financeiro.aluno_id', $alunoIdsFiltro);
        }
        if ($dataInicio) {
            $financeiroQuery->where('financeiro.data_vencimento', '>=', $dataInicio);
        }
        if ($dataFim) {
            $financeiroQuery->where('financeiro.data_vencimento', '<=', $dataFim);
        }

        $financeiroGrouped = $financeiroQuery->groupBy($groupByColumns)->get();

        $facturasQuery = Invoice::selectRaw($groupSelect)
            ->selectRaw('SUM(invoices.valor) as facturas')
            ->selectRaw('SUM(CASE WHEN invoices.status = "PAGA" THEN invoices.valor ELSE 0 END) as facturas_pagas')
            ->join('turma_aluno as ta', function($join) use ($anoLetivoId) {
                $join->on('ta.aluno_id', '=', 'invoices.aluno_id');
                if ($anoLetivoId) {
                    $join->where('ta.ano_letivo_id', '=', $anoLetivoId);
                }
            })
            ->join('turmas as t', 't.id', '=', 'ta.turma_id');

        if ($anoLetivoId) {
            $facturasQuery->where('invoices.ano_letivo_id', $anoLetivoId);
        }
        if ($alunoIdsFiltro) {
            $facturasQuery->whereIn('invoices.aluno_id', $alunoIdsFiltro);
        }
        if ($dataInicio) {
            $facturasQuery->where('invoices.data_emissao', '>=', $dataInicio);
        }
        if ($dataFim) {
            $facturasQuery->where('invoices.data_emissao', '<=', $dataFim);
        }

        $facturasGrouped = $facturasQuery->groupBy($groupByColumns)->get()->keyBy('chave');

        $dados = [];
        foreach ($financeiroGrouped as $row) {
            $fact = $facturasGrouped->get($row->chave);
            $dados[$row->chave] = [
                'label' => $row->label,
                'esperado' => (float) $row->esperado,
                'pago' => (float) $row->pago,
                'pendente' => (float) $row->pendente,
                'vencido' => (float) $row->vencido,
                'facturas' => (float) ($fact->facturas ?? 0),
                'facturas_pagas' => (float) ($fact->facturas_pagas ?? 0),
            ];
        }

        return $dados;
    }

    /**
     * Calcular dívidas por aluno
     */
    private function calcularDividasPorAluno($financeiros)
    {
        $dividas = [];

        foreach ($financeiros->whereIn('status', ['PENDENTE', 'VENCIDO']) as $fin) {
            $alunoId = $fin->aluno_id;
            if (!isset($dividas[$alunoId])) {
                $dividas[$alunoId] = [
                    'aluno' => $fin->aluno,
                    'total' => 0,
                    'pendente' => 0,
                    'vencido' => 0,
                    'itens' => [],
                ];
            }

            $dividas[$alunoId]['total'] += $fin->valor;
            if ($fin->status === 'VENCIDO') {
                $dividas[$alunoId]['vencido'] += $fin->valor;
            } else {
                $dividas[$alunoId]['pendente'] += $fin->valor;
            }
            $dividas[$alunoId]['itens'][] = $fin;
        }

        // Ordenar por total de dívida (maior primeiro)
        usort($dividas, function($a, $b) {
            return $b['total'] <=> $a['total'];
        });

        return $dividas;
    }

    /**
     * Nome do mês
     */
    private function nomeMes($mes)
    {
        $meses = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];
        return $meses[$mes] ?? '';
    }

    /**
     * Página de sucesso após pagamento com opção de imprimir/baixar factura
     */
    public function pagamentoSucesso(Invoice $invoice)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $invoice->load(['aluno.user', 'payment', 'anoLetivo']);

        // Preparar logotipo para a view
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoUrl = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoUrl = asset('storage/' . $logotipo);
        }

        return view('financeiro.pagamento-sucesso', compact('invoice', 'logotipoUrl'));
    }

    /**
     * Download PDF da factura após pagamento
     */
    public function downloadFactura(Invoice $invoice)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_mensalidades')) {
            abort(403);
        }

        $invoice->load(['aluno.user', 'payment', 'anoLetivo']);

        // Preparar logotipo em base64 para o PDF
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $logotipoBase64 = 'data:' . mime_content_type($logotipoPath) . ';base64,' . base64_encode($logotipoData);
        }

        $pdf = Pdf::loadView('aluno.invoice.pdf', compact('invoice', 'logotipoBase64'));
        
        return $pdf->download('factura-' . $invoice->numero_factura . '.pdf');
    }

    /**
     * Determina quais meses gerar baseado na modalidade de pagamento
     */
    private function determinarMesesPorModalidade(string $modalidade, int $mesInicio, int $mesFim): array
    {
        $meses = [];
        
        switch ($modalidade) {
            case 'MENSAL':
                // Gerar todos os meses
                for ($i = $mesInicio; $i <= $mesFim; $i++) {
                    $meses[] = $i;
                }
                break;
                
            case 'TRIMESTRAL':
                // Gerar apenas no início de cada trimestre (Janeiro, Abril, Julho, Outubro)
                $trimestres = [1, 4, 7, 10];
                foreach ($trimestres as $trimestre) {
                    if ($trimestre >= $mesInicio && $trimestre <= $mesFim) {
                        $meses[] = $trimestre;
                    }
                }
                break;
                
            case 'SEMESTRAL':
                // Gerar apenas no início de cada semestre (Janeiro, Julho)
                $semestres = [1, 7];
                foreach ($semestres as $semestre) {
                    if ($semestre >= $mesInicio && $semestre <= $mesFim) {
                        $meses[] = $semestre;
                    }
                }
                break;
                
            case 'ANUAL':
                // Gerar apenas em Janeiro
                if ($mesInicio <= 1 && $mesFim >= 1) {
                    $meses[] = 1;
                }
                break;
        }
        
        return $meses;
    }
    
    /**
     * Calcula o valor baseado na modalidade de pagamento
     */
    private function calcularValorPorModalidade(float $valorMensal, string $modalidade, int $mes, int $mesInicio): float
    {
        switch ($modalidade) {
            case 'MENSAL':
                return $valorMensal;
                
            case 'TRIMESTRAL':
                // Valor de 3 meses
                return $valorMensal * 3;
                
            case 'SEMESTRAL':
                // Valor de 6 meses
                return $valorMensal * 6;
                
            case 'ANUAL':
                // Valor de 12 meses
                return $valorMensal * 12;
                
            default:
                return $valorMensal;
        }
    }
    
    /**
     * Gera descrição da mensalidade baseada na modalidade
     */
    private function gerarDescricaoMensalidade(string $modalidade, int $mes, int $ano): string
    {
        $meses = ['Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
                 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
        
        switch ($modalidade) {
            case 'MENSAL':
                return "Mensalidade {$mes}/{$ano}";
                
            case 'TRIMESTRAL':
                $trimestre = ceil($mes / 3);
                return "Mensalidade Trimestral - {$trimestre}º Trimestre {$ano}";
                
            case 'SEMESTRAL':
                $semestre = $mes <= 6 ? 1 : 2;
                return "Mensalidade Semestral - {$semestre}º Semestre {$ano}";
                
            case 'ANUAL':
                return "Mensalidade Anual {$ano}";
                
            default:
                return "Mensalidade {$mes}/{$ano}";
        }
    }

    /**
     * Verifica se há pagamento via gateway para um financeiro
     */
    private function temPagamentoGateway($financeiroId): bool
    {
        $pagamentoGateway = Payment::where('financeiro_id', $financeiroId)
            ->whereIn('metodo', ['EMOLA', 'MPESA'])
            ->where('status', 'APROVADO')
            ->exists();

        $facturaGateway = Invoice::whereHas('payment', function($q) use ($financeiroId) {
            $q->where('financeiro_id', $financeiroId)
              ->whereIn('metodo', ['EMOLA', 'MPESA'])
              ->where('status', 'APROVADO');
        })->where('status', 'PAGA')->exists();

        return $pagamentoGateway || $facturaGateway;
    }

    /**
     * Agrupa dados financeiros por turma, classe ou nível de ensino
     */
    private function agruparDadosFinanceiros($financeiros, $pagamentos, $facturas, $agruparPor, $anoLetivoId)
    {
        $dados = [];

        if ($agruparPor === 'geral') {
            return [];
        }

        foreach ($financeiros as $fin) {
            $aluno = $fin->aluno;
            if (!$aluno) continue;

            $chave = null;
            $label = null;

            if ($agruparPor === 'turma') {
                $turma = $aluno->turmas()
                    ->wherePivot('ano_letivo_id', $anoLetivoId)
                    ->first();
                if ($turma) {
                    $chave = $turma->id;
                    $label = $turma->codigo . ' - ' . $turma->nome;
                }
            } elseif ($agruparPor === 'classe') {
                $turma = $aluno->turmas()
                    ->wherePivot('ano_letivo_id', $anoLetivoId)
                    ->first();
                if ($turma && $turma->codigo) {
                    $chave = $turma->codigo;
                    $label = 'Classe ' . $turma->codigo;
                }
            } elseif ($agruparPor === 'nivel_ensino') {
                $turma = $aluno->turmas()
                    ->wherePivot('ano_letivo_id', $anoLetivoId)
                    ->first();
                if ($turma && $turma->nivel_ensino) {
                    $chave = $turma->nivel_ensino;
                    $label = $turma->nivel_ensino === 'PRIMARIO' ? 'Ensino Primário' : 'Ensino Secundário';
                }
            }

            if (!$chave) continue;

            if (!isset($dados[$chave])) {
                $dados[$chave] = [
                    'label' => $label,
                    'esperado' => 0,
                    'pago' => 0,
                    'pendente' => 0,
                    'vencido' => 0,
                    'facturas' => 0,
                    'facturas_pagas' => 0,
                ];
            }

            $dados[$chave]['esperado'] += $fin->valor;
            
            if ($fin->status === 'PAGO') {
                $dados[$chave]['pago'] += $fin->valor_pago ?? $fin->valor;
            } elseif ($fin->status === 'VENCIDO') {
                $dados[$chave]['vencido'] += $fin->valor;
            } else {
                $dados[$chave]['pendente'] += $fin->valor;
            }
        }

        foreach ($facturas as $factura) {
            $aluno = $factura->aluno;
            if (!$aluno) continue;

            $chave = null;
            $label = null;

            if ($agruparPor === 'turma') {
                $turma = $aluno->turmas()
                    ->wherePivot('ano_letivo_id', $anoLetivoId)
                    ->first();
                if ($turma) {
                    $chave = $turma->id;
                }
            } elseif ($agruparPor === 'classe') {
                $turma = $aluno->turmas()
                    ->wherePivot('ano_letivo_id', $anoLetivoId)
                    ->first();
                if ($turma && $turma->codigo) {
                    $chave = $turma->codigo;
                }
            } elseif ($agruparPor === 'nivel_ensino') {
                $turma = $aluno->turmas()
                    ->wherePivot('ano_letivo_id', $anoLetivoId)
                    ->first();
                if ($turma && $turma->nivel_ensino) {
                    $chave = $turma->nivel_ensino;
                }
            }

            if ($chave && isset($dados[$chave])) {
                $dados[$chave]['facturas'] += $factura->valor;
                if ($factura->status === 'PAGA') {
                    $dados[$chave]['facturas_pagas'] += $factura->valor;
                }
            }
        }

        return $dados;
    }

    /**
     * Gera PDF do relatório financeiro
     */
    private function imprimirRelatorioFinanceiro(
        $financeiros, $pagamentos, $facturas, $totalEsperado, $totalPago,
        $totalPendente, $totalVencido, $totalFacturas, $totalFacturasPagas,
        $dadosPorPeriodo, $saldoPorMes, $pagamentosPorModalidade,
        $dividasPorAluno, $dadosAgrupados, $agruparPor,
        $anoLetivoId, $turmaId, $nivelEnsino, $tipoPeriodo
    ) {
        $anoLetivo = AnoLetivo::find($anoLetivoId);
        $turma = $turmaId ? Turma::find($turmaId) : null;
        
        $configuracao = [
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        // Preparar logotipo em base64
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $mime = mime_content_type($logotipoPath);
            $logotipoBase64 = "data:{$mime};base64," . base64_encode($logotipoData);
        }

        $pdf = Pdf::loadView('financeiro.relatorio-pdf', compact(
            'financeiros', 'pagamentos', 'facturas', 'totalEsperado', 'totalPago',
            'totalPendente', 'totalVencido', 'totalFacturas', 'totalFacturasPagas',
            'dadosPorPeriodo', 'saldoPorMes', 'pagamentosPorModalidade',
            'dividasPorAluno', 'dadosAgrupados', 'agruparPor',
            'anoLetivo', 'turma', 'nivelEnsino', 'tipoPeriodo',
            'configuracao', 'logotipoBase64'
        ))->setPaper('a4', 'landscape');

        $filename = 'Relatorio_Financeiro_' . date('Y-m-d') . '.pdf';
        return $pdf->download($filename);
    }

    /**
     * Gera mensalidades automaticamente para alunos que não têm
     * Otimizado para usar queries em lote e bulk insert
     */
    private function gerarMensalidadesAutomaticas($anoLetivoId, $alunos)
    {
        $anoLetivo = AnoLetivo::find($anoLetivoId);
        if (!$anoLetivo || $alunos->isEmpty()) {
            return;
        }

        $ano = (int) date('Y');
        $alunoIds = $alunos->pluck('id')->toArray();
        
        // Buscar todas as mensalidades existentes de uma vez
        $mensalidadesExistentes = Financeiro::whereIn('aluno_id', $alunoIds)
            ->where('ano_letivo_id', $anoLetivoId)
            ->where('tipo', 'Mensalidade')
            ->get()
            ->groupBy('aluno_id');
        
        // Identificar alunos que não têm mensalidades
        $alunosSemMensalidades = $alunos->filter(function($aluno) use ($mensalidadesExistentes) {
            return !$mensalidadesExistentes->has($aluno->id) || 
                   $mensalidadesExistentes->get($aluno->id)->isEmpty();
        });
        
        if ($alunosSemMensalidades->isEmpty()) {
            return;
        }
        
        // Preparar dados para bulk insert
        $mensalidadesParaInserir = [];
        $now = now();
        
        foreach ($alunosSemMensalidades as $aluno) {
            $modalidade = $aluno->modalidade_pagamento ?? 'MENSAL';
            $valorMensal = $this->financeiroService->getValorMensalidadePorClasse($aluno, $anoLetivoId);

            if ($valorMensal > 0) {
                // Determinar meses a gerar baseado na modalidade
                $meses = $this->determinarMesesPorModalidade($modalidade, 1, 12);
                
                foreach ($meses as $mes) {
                    $valor = $this->calcularValorPorModalidade($valorMensal, $modalidade, $mes, 1);
                    $dataVencimento = \Carbon\Carbon::create($ano, $mes, 5);

                    $mensalidadesParaInserir[] = [
                        'aluno_id' => $aluno->id,
                        'ano_letivo_id' => $anoLetivoId,
                        'tipo' => 'Mensalidade',
                        'descricao' => $this->gerarDescricaoMensalidade($modalidade, $mes, $ano),
                        'valor' => $valor,
                        'data_vencimento' => $dataVencimento,
                        'status' => 'PENDENTE',
                        'created_at' => $now,
                        'updated_at' => $now,
                    ];
                }
            }
        }
        
        // Inserir em lote (chunks de 100 para evitar problemas de memória)
        if (!empty($mensalidadesParaInserir)) {
            foreach (array_chunk($mensalidadesParaInserir, 100) as $chunk) {
                Financeiro::insert($chunk);
            }
        }
    }

    /**
     * Valida pagamento sequencial: só pode pagar meses futuros se os anteriores estiverem pagos
     * Permite pagamento adiantado se os meses anteriores estiverem pagos ou ainda não vencidos
     */
    private function validarPagamentoSequencial(Financeiro $financeiro)
    {
        // Extrair mês da descrição ou usar mês de vencimento
        $mes = null;
        $ano = null;
        
        // Tentar extrair da descrição
        if (preg_match('/(\d+)\/(\d+)/', $financeiro->descricao, $matches)) {
            $mes = (int) $matches[1];
            $ano = (int) $matches[2];
        } else {
            // Se não encontrar na descrição, usar mês de vencimento
            $mes = (int) $financeiro->data_vencimento->format('m');
            $ano = (int) $financeiro->data_vencimento->format('Y');
        }
        
        if (!$mes || !$ano) {
            return; // Se não conseguir determinar mês/ano, não validar
        }
        
        // Verificar se há meses anteriores não pagos
        // NÃO permitir pagar mês seguinte sem pagar os anteriores, independentemente de estarem vencidos
        for ($m = 1; $m < $mes; $m++) {
            $mensalidadeAnterior = Financeiro::where('aluno_id', $financeiro->aluno_id)
                ->where('ano_letivo_id', $financeiro->ano_letivo_id)
                ->where('tipo', 'Mensalidade')
                ->where(function($query) use ($m, $ano) {
                    $query->where('descricao', 'like', "%{$m}/{$ano}%")
                          ->orWhere(function($q) use ($m, $ano) {
                              // Também verificar por mês de vencimento
                              $q->whereMonth('data_vencimento', $m)
                                ->whereYear('data_vencimento', $ano);
                          });
                })
                ->first();
            
            // Se a mensalidade anterior existe e não está paga, bloquear pagamento
            if ($mensalidadeAnterior && $mensalidadeAnterior->status !== 'PAGO') {
                $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
                         'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
                $mesNomeAnterior = $meses[$m] ?? $m;
                $mesNomeAtual = $meses[$mes] ?? $mes;
                throw new \Exception("Não é possível pagar a mensalidade de {$mesNomeAtual}/{$ano} sem antes pagar a mensalidade de {$mesNomeAnterior}/{$ano}.");
            }
        }
    }

    /**
     * Criar notificações de pagamento presencial para admin e funcionários
     */
    private function criarNotificacoesPagamentoPresencial(Payment $payment)
    {
        $payment->load(['aluno.user', 'financeiro']);
        $alunoNome = $payment->aluno->user->name ?? 'Aluno';
        $valor = $payment->valor;
        $metodo = 'PRESENCIAL';
        
        // Link para visualizar o pagamento
        $link = route('financeiro.index', ['ano_letivo_id' => $payment->financeiro->ano_letivo_id ?? null]);
        
        // Notificar todos os admins e superadmins (todos os tipos de pagamento)
        $admins = User::whereIn('tipo', ['admin', 'superadmin'])->get();
        foreach ($admins as $admin) {
            Notification::criarNotificacaoPagamento(
                $admin->id,
                'payment',
                $valor,
                $alunoNome,
                $metodo,
                $link
            );
        }
    }
}



