<?php

namespace App\Http\Controllers;

use App\Models\Evento;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\AnoLetivo;
use Illuminate\Http\Request;

class EventoController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        $user = auth()->user();
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();

        if ($user->isAluno()) {
            $aluno = $user->aluno;
            $turmas = $aluno ? $aluno->turmas()->wherePivot('ano_letivo_id', $anoLetivo?->id)->get() : collect();
            $turmaIds = $turmas->pluck('id');
            
            $eventos = Evento::where('ano_letivo_id', $anoLetivo?->id)
                ->where(function($query) use ($turmaIds) {
                    $query->where('tipo', 'Geral')
                        ->orWhereIn('turma_id', $turmaIds);
                })
                ->orderBy('data', 'asc')
                ->orderBy('hora_inicio', 'asc')
                ->paginate(25)
                ->onEachSide(2)
                ->appends(request()->query());
        } else {
            $eventos = Evento::where('ano_letivo_id', $anoLetivo?->id)
                ->select('eventos.id', 'eventos.titulo', 'eventos.data', 'eventos.hora_inicio', 'eventos.hora_fim', 'eventos.turma_id', 'eventos.disciplina_id', 'eventos.tipo')
                ->with([
                    'turma:id,nome,codigo',
                    'disciplina:id,nome,codigo'
                ])
                ->orderBy('data', 'asc')
                ->orderBy('hora_inicio', 'asc')
                ->paginate(25)
                ->onEachSide(2)
                ->appends(request()->query());
        }

        return view('eventos.index', compact('eventos', 'anoLetivo'));
    }

    public function create()
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_eventos')) {
            abort(403, 'Acesso negado. Você não tem permissão para criar eventos.');
        }

        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)->where('ativa', true)->get();
        $disciplinas = Disciplina::where('ativa', true)->get();

        return view('eventos.create', compact('turmas', 'disciplinas', 'anoLetivo'));
    }

    public function store(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_eventos')) {
            abort(403, 'Acesso negado. Você não tem permissão para criar eventos.');
        }

        $rules = [
            'titulo' => 'required|string',
            'descricao' => 'nullable|string',
            'data' => 'required|date',
            'hora_inicio' => 'nullable|date_format:H:i',
            'tipo' => 'required|in:Geral,Turma,Disciplina',
            'turma_id' => 'nullable|required_if:tipo,Turma|exists:turmas,id',
            'disciplina_id' => 'nullable|required_if:tipo,Disciplina|exists:disciplinas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ];
        
        if ($request->filled('hora_fim')) {
            $rules['hora_fim'] = 'nullable|date_format:H:i|after:hora_inicio';
        }
        
        $request->validate($rules);

        Evento::create($request->all());

        return redirect()->route('eventos.index')->with('success', 'Evento criado com sucesso!');
    }

    public function edit(Evento $evento)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_eventos')) {
            abort(403, 'Acesso negado. Você não tem permissão para editar eventos.');
        }

        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)->where('ativa', true)->get();
        $disciplinas = Disciplina::where('ativa', true)->get();

        return view('eventos.edit', compact('evento', 'turmas', 'disciplinas', 'anoLetivo'));
    }

    public function update(Request $request, Evento $evento)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_eventos')) {
            abort(403, 'Acesso negado. Você não tem permissão para editar eventos.');
        }

        $rules = [
            'titulo' => 'required|string',
            'descricao' => 'nullable|string',
            'data' => 'required|date',
            'hora_inicio' => 'nullable|date_format:H:i',
            'tipo' => 'required|in:Geral,Turma,Disciplina',
            'turma_id' => 'nullable|required_if:tipo,Turma|exists:turmas,id',
            'disciplina_id' => 'nullable|required_if:tipo,Disciplina|exists:disciplinas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ];
        
        if ($request->filled('hora_fim')) {
            $rules['hora_fim'] = 'nullable|date_format:H:i|after:hora_inicio';
        }
        
        $request->validate($rules);

        $evento->update($request->all());

        return redirect()->route('eventos.index')->with('success', 'Evento atualizado com sucesso!');
    }

    public function destroy(Evento $evento)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_eventos')) {
            abort(403, 'Acesso negado. Você não tem permissão para eliminar eventos.');
        }

        // Hard delete (remove do banco de dados)
        $evento->forceDelete();

        return redirect()->route('eventos.index')->with('success', 'Evento eliminado com sucesso!');
    }

    public function cancelar(Evento $evento)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_eventos')) {
            abort(403, 'Acesso negado. Você não tem permissão para cancelar eventos.');
        }

        $evento->cancelado = !$evento->cancelado;
        $evento->save();

        $status = $evento->cancelado ? 'cancelado' : 'reativado';
        return redirect()->route('eventos.index')->with('success', "Evento {$status} com sucesso!");
    }
}



