<?php

namespace App\Http\Controllers;

use App\Models\Avaliacao;
use App\Models\Aluno;
use App\Models\Disciplina;
use App\Models\Turma;
use App\Models\Trimestre;
use App\Services\AvaliacaoService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Barryvdh\DomPDF\Facade\Pdf;

class AvaliacaoController extends Controller
{
    protected $avaliacaoService;

    public function __construct(AvaliacaoService $avaliacaoService)
    {
        $this->middleware('auth');
        // Permite acesso para admin, superadmin, professor e aluno
        $this->avaliacaoService = $avaliacaoService;
    }

    public function index(Request $request)
    {
        $user = auth()->user();
        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();

        // Admin, Superadmin e funcionários com permissão podem ver todas as avaliações
        if ($user->isAdmin() || $user->isSuperadmin() || $user->podeGerir('gerir_avaliacoes')) {
            $avaliacoes = \App\Models\Avaliacao::where('ano_letivo_id', $anoLetivo?->id)
                ->with(['disciplina', 'trimestre', 'turma', 'aluno.user'])
                ->orderBy('created_at', 'desc')
                ->get();
            
            // Buscar todas as turmas do ano letivo ativo para permitir visualização completa
            $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
                ->where('ativa', true)
                ->orderBy('nome')
                ->get();
            
            return view('avaliacoes.index', compact('avaliacoes', 'anoLetivo', 'turmas'));
        }

        if ($user->isProfessor()) {
            $professor = $user->professor;
            $turmasComDisciplinas = [];
            
            if (!$professor) {
                // Se o usuário é professor mas não tem registro na tabela professores
                return view('avaliacoes.index', compact('turmasComDisciplinas', 'anoLetivo'))
                    ->withErrors(['error' => 'Seu perfil de professor não está completo. Entre em contato com o administrador.']);
            }
            
            // Verificar se há filtros de turma e disciplina
            $turmaFiltro = $request->get('turma_id');
            $disciplinaFiltro = $request->get('disciplina_id');
            
            if ($turmaFiltro && $disciplinaFiltro) {
                // Verificar se o professor leciona esta disciplina nesta turma
                $alocacao = DB::table('turma_disciplina_professor')
                    ->where('turma_id', $turmaFiltro)
                    ->where('disciplina_id', $disciplinaFiltro)
                    ->where('professor_id', $professor->id)
                    ->where('ano_letivo_id', $anoLetivo?->id)
                    ->first();
                
                if (!$alocacao) {
                    abort(403, 'Você não tem permissão para ver avaliações desta disciplina.');
                }
                
                $turma = Turma::find($turmaFiltro);
                $disciplina = Disciplina::find($disciplinaFiltro);
                
                // Buscar trimestres do ano letivo
                $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo?->id)
                    ->orderBy('numero')
                    ->get();
                
                // Filtro de trimestre ou anual
                $filtroTipo = $request->get('filtro_tipo', 'anual'); // 'trimestre' ou 'anual'
                $trimestreFiltro = $request->get('trimestre_id');
                
                // Buscar avaliações da turma e disciplina especificadas
                $query = Avaliacao::where('turma_id', $turmaFiltro)
                    ->where('disciplina_id', $disciplinaFiltro)
                    ->where('ano_letivo_id', $anoLetivo?->id)
                    ->with(['disciplina', 'trimestre', 'turma', 'aluno.user']);
                
                if ($filtroTipo === 'trimestre' && $trimestreFiltro) {
                    $query->where('trimestre_id', $trimestreFiltro);
                }
                
                $avaliacoes = $query->orderBy('aluno_id')
                    ->orderBy('trimestre_id')
                    ->get();
                
                // Se for anual, organizar por aluno e buscar MFRD
                $avaliacoesPorAluno = [];
                $mediasAnuais = [];
                $classificacoes = [];
                
                if ($filtroTipo === 'anual') {
                    foreach ($avaliacoes as $avaliacao) {
                        $alunoId = $avaliacao->aluno_id;
                        if (!isset($avaliacoesPorAluno[$alunoId])) {
                            $avaliacoesPorAluno[$alunoId] = [];
                        }
                        $avaliacoesPorAluno[$alunoId][$avaliacao->trimestre_id] = $avaliacao;
                        
                        // Buscar MFRD da primeira avaliação (todas têm o mesmo valor)
                        if (!isset($mediasAnuais[$alunoId]) && $avaliacao->mfrd !== null) {
                            $mediasAnuais[$alunoId] = round($avaliacao->mfrd, 0);
                            $classificacoes[$alunoId] = \App\Models\Avaliacao::isAprovado($avaliacao->mfrd) ? 'APROVADO' : 'REPROVADO';
                        }
                    }
                }
                
                return view('avaliacoes.index', compact(
                    'avaliacoes', 'anoLetivo', 'turma', 'disciplina', 'turmasComDisciplinas',
                    'trimestres', 'filtroTipo', 'trimestreFiltro', 'avaliacoesPorAluno', 'mediasAnuais', 'classificacoes'
                ));
            }
            
            // Se não há filtros, mostrar lista de turmas e disciplinas
            if ($anoLetivo) {
                $alocacoes = \DB::table('turma_disciplina_professor')
                    ->where('professor_id', $professor->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->get();
                
                foreach ($alocacoes as $alocacao) {
                    $turma = Turma::find($alocacao->turma_id);
                    $disciplina = Disciplina::find($alocacao->disciplina_id);
                    
                    if ($turma && $disciplina) {
                        $turmasComDisciplinas[] = (object)[
                            'turma' => $turma,
                            'disciplina' => $disciplina,
                            'turma_id' => $turma->id,
                            'disciplina_id' => $disciplina->id,
                        ];
                    }
                }
            }
            
            return view('avaliacoes.index', compact('turmasComDisciplinas', 'anoLetivo'));
        } elseif ($user->isAluno()) {
            $aluno = $user->aluno;
            $avaliacoes = $aluno ? $aluno->avaliacoes()->where('ano_letivo_id', $anoLetivo?->id)->with(['disciplina', 'trimestre', 'turma'])->get() : collect();
            return view('avaliacoes.index', compact('avaliacoes', 'anoLetivo'));
        }

        abort(403, 'Acesso negado. Você não tem permissão para acessar avaliações.');
    }

    public function create(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403);
        }

        // Validar parâmetros obrigatórios
        if (!$request->has('turma_id') || !$request->has('disciplina_id')) {
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->with('error', 'Parâmetros obrigatórios não fornecidos (turma_id e disciplina_id).');
        }

        try {
            $request->validate([
                'turma_id' => 'required|exists:turmas,id',
                'disciplina_id' => 'required|exists:disciplinas,id',
            ], [
                'turma_id.required' => 'O ID da turma é obrigatório.',
                'turma_id.exists' => 'A turma selecionada não existe.',
                'disciplina_id.required' => 'O ID da disciplina é obrigatório.',
                'disciplina_id.exists' => 'A disciplina selecionada não existe.',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->withErrors($e->errors())
                ->withInput();
        }

        $professor = $user->professor;
        
        if (!$professor) {
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->with('error', 'Professor não encontrado.');
        }

        $turma = Turma::find($request->turma_id);
        if (!$turma) {
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->with('error', 'Turma não encontrada.');
        }

        $disciplina = Disciplina::find($request->disciplina_id);
        if (!$disciplina) {
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->with('error', 'Disciplina não encontrada.');
        }
        
        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();
        
        if (!$anoLetivo) {
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->with('error', 'Nenhum ano letivo ativo encontrado.');
        }
        
        // Verificar se existem trimestres para o ano letivo ativo, se não, criar automaticamente
        $trimestresExistentes = Trimestre::where('ano_letivo_id', $anoLetivo->id)->count();
        
        // Debug: verificar se os trimestres existem
        if ($trimestresExistentes === 0) {
            \Log::info('Nenhum trimestre encontrado para o ano letivo', [
                'ano_letivo_id' => $anoLetivo->id,
                'ano_letivo_ano' => $anoLetivo->ano,
            ]);
            // Criar trimestres automaticamente
            // Extrair ano inicial do formato "2025/2026" ou usar o ano diretamente
            $anoString = $anoLetivo->ano;
            if (strpos($anoString, '/') !== false) {
                $anoInicial = (int) explode('/', $anoString)[0];
            } else {
                $anoInicial = (int) $anoString;
            }
            $anoProximo = $anoInicial + 1;
            
            $trimestres = [
                [
                    'nome' => '1º Trimestre',
                    'numero' => 1,
                    'data_inicio' => "{$anoInicial}-09-01",
                    'data_fim' => "{$anoInicial}-12-15",
                    'ativo' => false,
                    'ano_letivo_id' => $anoLetivo->id,
                ],
                [
                    'nome' => '2º Trimestre',
                    'numero' => 2,
                    'data_inicio' => "{$anoInicial}-12-16",
                    'data_fim' => "{$anoProximo}-03-15",
                    'ativo' => false,
                    'ano_letivo_id' => $anoLetivo->id,
                ],
                [
                    'nome' => '3º Trimestre',
                    'numero' => 3,
                    'data_inicio' => "{$anoProximo}-03-16",
                    'data_fim' => "{$anoProximo}-07-31",
                    'ativo' => false,
                    'ano_letivo_id' => $anoLetivo->id,
                ],
            ];
            
            foreach ($trimestres as $trimestreData) {
                $trimestreCriado = Trimestre::firstOrCreate(
                    [
                        'ano_letivo_id' => $anoLetivo->id,
                        'numero' => $trimestreData['numero'],
                    ],
                    $trimestreData
                );
            }
            
            // Recarregar contagem após criação
            $trimestresExistentes = Trimestre::where('ano_letivo_id', $anoLetivo->id)->count();
            \Log::info('Trimestres criados', [
                'ano_letivo_id' => $anoLetivo->id,
                'total_criados' => $trimestresExistentes,
            ]);
        }
        
        // Buscar trimestre do ano letivo ativo
        $trimestreId = $request->input('trimestre_id');
        
        // Primeiro, sempre buscar todos os trimestres disponíveis para o ano letivo
        $trimestresDisponiveis = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();
        
        // Se não houver trimestres, algo está errado
        if ($trimestresDisponiveis->isEmpty()) {
            \Log::error('Nenhum trimestre disponível após verificação', [
                'ano_letivo_id' => $anoLetivo->id,
                'ano_letivo_ano' => $anoLetivo->ano,
            ]);
            
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->with('error', 'Nenhum trimestre disponível para o ano letivo ativo. Por favor, contacte o administrador.');
        }
        
        // Tentar encontrar o trimestre específico se foi fornecido
        $trimestre = null;
        if (!empty($trimestreId) && $trimestreId != '0' && is_numeric($trimestreId)) {
            // Buscar o trimestre específico na lista de disponíveis
            $trimestre = $trimestresDisponiveis->firstWhere('id', (int)$trimestreId);
            
            // Se não encontrou na lista, redirecionar para o primeiro trimestre disponível
            if (!$trimestre) {
                \Log::warning('Trimestre ID fornecido não pertence ao ano letivo, redirecionando', [
                    'trimestre_id_request' => $trimestreId,
                    'ano_letivo_id' => $anoLetivo->id,
                    'ids_disponiveis' => $trimestresDisponiveis->pluck('id')->toArray(),
                ]);
                
                // Redirecionar para o primeiro trimestre disponível
                $primeiroTrimestre = $trimestresDisponiveis->first();
                if ($primeiroTrimestre) {
                    return redirect()->route('avaliacoes.create', [
                        'turma_id' => $request->turma_id,
                        'disciplina_id' => $request->disciplina_id,
                        'trimestre_id' => $primeiroTrimestre->id,
                        'ano_letivo_id' => $anoLetivo->id,
                    ])->with('info', 'O trimestre selecionado não existe. Redirecionado para o primeiro trimestre disponível.');
                }
            }
        }
        
        // Se não encontrou o específico ou não foi fornecido, usar o primeiro disponível
        if (!$trimestre) {
            $trimestre = $trimestresDisponiveis->first();
        }
        
        // Verificação final (não deveria acontecer, mas por segurança)
        if (!$trimestre) {
            \Log::error('Erro crítico: trimestre não encontrado após todas as tentativas', [
                'ano_letivo_id' => $anoLetivo->id,
                'trimestre_id_request' => $trimestreId,
                'total_trimestres' => $trimestresDisponiveis->count(),
            ]);
            
            $redirectUrl = $request->headers->get('referer') ?: route('professor.turmas');
            return redirect($redirectUrl)
                ->with('error', 'Erro ao buscar trimestre. Por favor, tente novamente.');
        }

        // Verifica se o professor leciona esta disciplina nesta turma
        $alocacao = DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('disciplina_id', $disciplina->id)
            ->where('professor_id', $professor->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->first();

        if (!$alocacao) {
            // Se for requisição AJAX, retornar JSON em vez de abort
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'error' => true,
                    'message' => 'Você não leciona esta disciplina nesta turma.',
                    'code' => 403
                ], 403);
            }
            abort(403, 'Você não leciona esta disciplina nesta turma.');
        }

        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->with(['user' => function($query) {
                $query->whereNull('deleted_at');
            }])
            ->get()
            ->sortBy(function($aluno) {
                return $aluno->user ? $aluno->user->name : '';
            })
            ->values();

        $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero')
            ->get();

        // Verificar se o lançamento de notas está liberado para este trimestre específico
        $lancamentoPorTrimestre = \App\Models\Configuracao::get('lancamento_notas_trimestres', []);
        if (is_string($lancamentoPorTrimestre)) {
            $lancamentoPorTrimestre = json_decode($lancamentoPorTrimestre, true) ?? [];
        }
        $lancamentoNotasLiberado = $lancamentoPorTrimestre[$trimestre->id] ?? true; // Padrão: liberado

        return view('avaliacoes.create', compact('turma', 'disciplina', 'trimestre', 'trimestres', 'alunos', 'anoLetivo', 'lancamentoNotasLiberado'));
    }

    public function store(Request $request)
    {
        try {
            $user = auth()->user();
            
            if (!$user->isProfessor()) {
                abort(403);
            }

            // Verificar se o lançamento de notas está liberado para este trimestre específico
            $trimestreAtual = Trimestre::findOrFail($request->trimestre_id);
            $lancamentoPorTrimestre = \App\Models\Configuracao::get('lancamento_notas_trimestres', []);
            if (is_string($lancamentoPorTrimestre)) {
                $lancamentoPorTrimestre = json_decode($lancamentoPorTrimestre, true) ?? [];
            }
            $lancamentoNotasLiberado = $lancamentoPorTrimestre[$trimestreAtual->id] ?? true; // Padrão: liberado
            
            if (!$lancamentoNotasLiberado) {
                $isAjax = $request->ajax() 
                    || $request->wantsJson() 
                    || $request->header('X-Requested-With') === 'XMLHttpRequest'
                    || $request->header('Accept') === 'application/json';
                
                $mensagem = "O lançamento de notas do {$trimestreAtual->numero}º Trimestre está trancado. Você pode apenas visualizar as notas, mas não pode alterá-las.";
                
                if ($isAjax) {
                    return response()->json([
                        'success' => false,
                        'error' => true,
                        'message' => $mensagem,
                        'code' => 403
                    ], 403);
                }
                
                return redirect()->back()
                    ->withErrors(['error' => $mensagem])
                    ->withInput();
            }

            // Log dos dados recebidos para debug
            Log::info('Dados recebidos para salvar avaliações', [
                'turma_id' => $request->turma_id,
                'disciplina_id' => $request->disciplina_id,
                'trimestre_id' => $request->trimestre_id,
                'ano_letivo_id' => $request->ano_letivo_id,
                'total_notas' => count($request->notas ?? []),
            ]);
            
            // Validação detalhada das notas
            $rules = [
                'turma_id' => 'required|exists:turmas,id',
                'disciplina_id' => 'required|exists:disciplinas,id',
                'trimestre_id' => 'required|exists:trimestres,id',
                'ano_letivo_id' => 'required|exists:ano_letivo,id',
                'notas' => 'required|array',
            ];
        
        // Adicionar validação para cada campo de nota (0 a 20)
        $notaRules = [
            'nullable', 'numeric', 'min:0', 'max:20'
        ];
        
        if ($request->has('notas') && is_array($request->notas)) {
            foreach ($request->notas as $alunoId => $notas) {
                $rules["notas.{$alunoId}.teste1"] = $notaRules;
                $rules["notas.{$alunoId}.teste2"] = $notaRules;
                $rules["notas.{$alunoId}.avaliacao_final"] = $notaRules;
                $rules["notas.{$alunoId}.a_pratica_1"] = $notaRules;
                $rules["notas.{$alunoId}.a_pratica_2"] = $notaRules;
                $rules["notas.{$alunoId}.acs_1"] = $notaRules;
                $rules["notas.{$alunoId}.acs_2"] = $notaRules;
                $rules["notas.{$alunoId}.at"] = $notaRules;
            }
        }
        
        try {
            $request->validate($rules, [
                'turma_id.required' => 'O ID da turma é obrigatório.',
                'turma_id.exists' => 'A turma selecionada não existe.',
                'disciplina_id.required' => 'O ID da disciplina é obrigatório.',
                'disciplina_id.exists' => 'A disciplina selecionada não existe.',
                'trimestre_id.required' => 'O ID do trimestre é obrigatório.',
                'trimestre_id.exists' => 'O trimestre selecionado não existe.',
                'ano_letivo_id.required' => 'O ID do ano letivo é obrigatório.',
                'ano_letivo_id.exists' => 'O ano letivo selecionado não existe.',
                'notas.required' => 'É necessário fornecer pelo menos uma nota.',
                'notas.array' => 'As notas devem ser fornecidas em formato de array.',
                'notas.*.teste1.min' => 'O campo Teste 1 deve ser no mínimo 0.',
                'notas.*.teste1.max' => 'O campo Teste 1 deve ser no máximo 20.',
                'notas.*.teste2.min' => 'O campo Teste 2 deve ser no mínimo 0.',
                'notas.*.teste2.max' => 'O campo Teste 2 deve ser no máximo 20.',
                'notas.*.avaliacao_final.min' => 'O campo Avaliação Final deve ser no mínimo 0.',
                'notas.*.avaliacao_final.max' => 'O campo Avaliação Final deve ser no máximo 20.',
                'notas.*.a_pratica_1.min' => 'O campo A.Prática 1ª deve ser no mínimo 0.',
                'notas.*.a_pratica_1.max' => 'O campo A.Prática 1ª deve ser no máximo 20.',
                'notas.*.a_pratica_2.min' => 'O campo A.Prática 2ª deve ser no mínimo 0.',
                'notas.*.a_pratica_2.max' => 'O campo A.Prática 2ª deve ser no máximo 20.',
                'notas.*.acs_1.min' => 'O campo ACS 1ª deve ser no mínimo 0.',
                'notas.*.acs_1.max' => 'O campo ACS 1ª deve ser no máximo 20.',
                'notas.*.acs_2.min' => 'O campo ACS 2ª deve ser no mínimo 0.',
                'notas.*.acs_2.max' => 'O campo ACS 2ª deve ser no máximo 20.',
                'notas.*.at.min' => 'O campo AT deve ser no mínimo 0.',
                'notas.*.at.max' => 'O campo AT deve ser no máximo 20.',
            ]);
            
            // Validação adicional: verificar se o trimestre pertence ao ano letivo
            $anoLetivoValidacao = \App\Models\AnoLetivo::where('ativo', true)->first();
            if ($anoLetivoValidacao && $request->trimestre_id) {
                $trimestre = Trimestre::where('id', $request->trimestre_id)
                    ->where('ano_letivo_id', $anoLetivoValidacao->id)
                    ->first();
                
                if (!$trimestre) {
                    $trimestresDisponiveis = Trimestre::where('ano_letivo_id', $anoLetivoValidacao->id)->pluck('id')->toArray();
                    Log::warning('Trimestre não pertence ao ano letivo', [
                        'trimestre_id' => $request->trimestre_id,
                        'ano_letivo_id' => $anoLetivoValidacao->id,
                        'trimestres_disponiveis' => $trimestresDisponiveis
                    ]);
                    
                    if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                        return response()->json([
                            'success' => false,
                            'message' => 'O trimestre selecionado não pertence ao ano letivo ativo. Trimestres disponíveis: ' . implode(', ', $trimestresDisponiveis),
                            'errors' => [
                                'trimestre_id' => ['O trimestre selecionado não pertence ao ano letivo ativo. Por favor, selecione um trimestre válido.']
                            ]
                        ], 422);
                    }
                    
                    return redirect()->back()
                        ->withErrors(['trimestre_id' => 'O trimestre selecionado não pertence ao ano letivo ativo.'])
                        ->withInput();
                }
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Se for requisição AJAX, retornar erros em JSON
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro de validação',
                    'errors' => $e->errors()
                ], 422);
            }
            throw $e;
        }

        $professor = $user->professor;
        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();
        $turma = Turma::findOrFail($request->turma_id);
        $nivelEnsino = $turma->nivel_ensino ?? 'PRIMARIO';
        
        // Verifica permissão usando a mesma lógica do método create()
        $alocacao = DB::table('turma_disciplina_professor')
            ->where('turma_id', $request->turma_id)
            ->where('disciplina_id', $request->disciplina_id)
            ->where('professor_id', $professor->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->first();

        if (!$alocacao) {
            // Se for requisição AJAX, retornar JSON em vez de abort
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                return response()->json([
                    'success' => false,
                    'error' => true,
                    'message' => 'Você não leciona esta disciplina nesta turma.',
                    'code' => 403
                ], 403);
            }
            abort(403, 'Você não leciona esta disciplina nesta turma.');
        }
        
        // Validar ordem dos trimestres - não pode lançar notas de um trimestre sem ter notas dos anteriores
        // IMPORTANTE: Esta validação deve ser executada ANTES de processar as notas
        $trimestreAtual = Trimestre::findOrFail($request->trimestre_id);
        $numeroTrimestreAtual = $trimestreAtual->numero;
        
        // Buscar todos os alunos da turma
        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->get()
            ->pluck('id')
            ->toArray();
        
        // Verificar se há notas nos trimestres anteriores
        // Se for trimestre 2 ou 3, verificar se os anteriores têm notas
        if ($numeroTrimestreAtual > 1 && !empty($alunos)) {
            // Buscar trimestres anteriores - IMPORTANTE: verificar TODOS os trimestres anteriores
            $trimestresAnteriores = Trimestre::where('ano_letivo_id', $anoLetivo->id)
                ->where('numero', '<', $numeroTrimestreAtual)
                ->orderBy('numero', 'asc')
                ->get();
            
            // Se não houver trimestres anteriores, pode prosseguir (caso especial)
            if ($trimestresAnteriores->isEmpty()) {
                // Pode prosseguir
            } else {
                // Verificar CADA trimestre anterior - TODOS devem ter notas para TODOS os alunos
                foreach ($trimestresAnteriores as $trimestreAnterior) {
                    // Verificar se TODOS os alunos têm pelo menos uma nota REALMENTE PREENCHIDA no trimestre anterior
                    $alunosComNotas = [];
                    
                    foreach ($alunos as $alunoId) {
                        $avaliacaoAnterior = Avaliacao::where('turma_id', $request->turma_id)
                            ->where('disciplina_id', $request->disciplina_id)
                            ->where('trimestre_id', $trimestreAnterior->id)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->where('aluno_id', $alunoId)
                            ->first();
                        
                        // Verificar se a avaliação existe E tem pelo menos uma nota preenchida
                        if ($avaliacaoAnterior) {
                            // Verificar se tem pelo menos um campo de nota preenchido
                            // Considerar que 0 é um valor válido (nota zero), então apenas verificar se não é null
                            // Também verificar campos calculados (MAP, MACS, MT, media_nacional) como indicadores
                            $temNota = false;
                            
                            // Para primário: verificar a_pratica_1, a_pratica_2, acs_1, acs_2, at
                            // Ou campos calculados: map, macs, mt
                            if ($nivelEnsino === 'PRIMARIO') {
                                $temNota = $avaliacaoAnterior->a_pratica_1 !== null
                                         || $avaliacaoAnterior->a_pratica_2 !== null
                                         || $avaliacaoAnterior->acs_1 !== null
                                         || $avaliacaoAnterior->acs_2 !== null
                                         || $avaliacaoAnterior->at !== null
                                         || $avaliacaoAnterior->map !== null
                                         || $avaliacaoAnterior->macs !== null
                                         || $avaliacaoAnterior->mt !== null;
                            } 
                            // Para secundário: verificar teste1, teste2, avaliacao_final ou media_nacional
                            else {
                                $temNota = $avaliacaoAnterior->teste1 !== null
                                         || $avaliacaoAnterior->teste2 !== null
                                         || $avaliacaoAnterior->avaliacao_final !== null
                                         || $avaliacaoAnterior->media_nacional !== null;
                            }
                            
                            if ($temNota) {
                                $alunosComNotas[] = $alunoId;
                            }
                        }
                    }
                    
                    // Se nem todos os alunos têm notas no trimestre anterior, bloquear
                    if (count($alunosComNotas) < count($alunos)) {
                        $faltam = count($alunos) - count($alunosComNotas);
                        $mensagemErro = "Não é possível lançar notas do {$numeroTrimestreAtual}º trimestre. " .
                                       "Primeiro deve lançar as notas do {$trimestreAnterior->numero}º trimestre para todos os alunos. " .
                                       "Faltam notas para {$faltam} aluno(s).";
                        
                        // Verificar se é requisição AJAX de múltiplas formas
                        $isAjax = $request->ajax() 
                                || $request->wantsJson() 
                                || $request->header('X-Requested-With') === 'XMLHttpRequest'
                                || $request->header('Accept') === 'application/json';
                        
                        if ($isAjax) {
                            return response()->json([
                                'success' => false,
                                'error' => true,
                                'message' => $mensagemErro,
                                'code' => 422
                            ], 422);
                        }
                        
                        return redirect()->back()
                            ->withErrors(['trimestre' => $mensagemErro])
                            ->withInput();
                    }
                }
            }
        }

        // Processa notas de cada aluno
        foreach ($request->notas as $alunoId => $notas) {
            // Função auxiliar para converter string vazia em null
            $toNull = function($value) {
                return ($value === '' || $value === null) ? null : $value;
            };
            
            $dados = [
                'aluno_id' => $alunoId,
                'disciplina_id' => $request->disciplina_id,
                'turma_id' => $request->turma_id,
                'trimestre_id' => $request->trimestre_id,
                'ano_letivo_id' => $request->ano_letivo_id,
                'modelo' => 'NACIONAL',
                'a_pratica_1' => $toNull($notas['a_pratica_1'] ?? null),
                'a_pratica_2' => $toNull($notas['a_pratica_2'] ?? null),
                'acs_1' => $toNull($notas['acs_1'] ?? null),
                'acs_2' => $toNull($notas['acs_2'] ?? null),
                'at' => $toNull($notas['at'] ?? null),
                'teste1' => $toNull($notas['teste1'] ?? null),
                'teste2' => $toNull($notas['teste2'] ?? null),
                'avaliacao_final' => $toNull($notas['avaliacao_final'] ?? null),
            ];

            try {
                $this->avaliacaoService->salvarAvaliacao($dados, $nivelEnsino);
            } catch (\Exception $e) {
                Log::error('Erro ao salvar avaliação individual', [
                    'aluno_id' => $alunoId,
                    'dados' => $dados,
                    'erro' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                throw $e;
            }
        }

        // Sempre retornar JSON para requisições AJAX (não recarregar página)
        // Verificar múltiplas formas de identificar requisição AJAX
        $isAjax = $request->ajax() 
            || $request->wantsJson() 
            || $request->header('X-Requested-With') === 'XMLHttpRequest'
            || $request->header('Accept') === 'application/json'
            || str_contains($request->header('Accept', ''), 'application/json');
            
        if ($isAjax) {
            return response()->json([
                'success' => true,
                'message' => 'Avaliações salvas com sucesso!'
            ], 200, ['Content-Type' => 'application/json']);
        }

        // Caso contrário, redireciona de volta para a mesma página (fallback)
        return redirect()->route('avaliacoes.create', [
            'turma_id' => $request->turma_id,
            'disciplina_id' => $request->disciplina_id,
            'trimestre_id' => $request->trimestre_id
        ])->with('success', 'Avaliações registradas com sucesso!');
        
        } catch (ModelNotFoundException $e) {
            // Tratar erro de modelo não encontrado
            $isAjax = $request->ajax() 
                || $request->wantsJson() 
                || $request->header('X-Requested-With') === 'XMLHttpRequest'
                || $request->header('Accept') === 'application/json';
            
            if ($isAjax) {
                return response()->json([
                    'success' => false,
                    'error' => true,
                    'message' => 'Registro não encontrado. Por favor, recarregue a página.',
                    'code' => 404
                ], 404);
            }
            
            return redirect()->back()
                ->withErrors(['error' => 'Registro não encontrado.'])
                ->withInput();
                
        } catch (\Exception $e) {
            // Log do erro para debug
            Log::error('Erro ao salvar avaliações', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Verificar se é requisição AJAX
            $isAjax = $request->ajax() 
                || $request->wantsJson() 
                || $request->header('X-Requested-With') === 'XMLHttpRequest'
                || $request->header('Accept') === 'application/json';
            
            if ($isAjax) {
                return response()->json([
                    'success' => false,
                    'error' => true,
                    'message' => 'Erro ao salvar avaliações: ' . $e->getMessage(),
                    'code' => 500
                ], 500);
            }
            
            return redirect()->back()
                ->withErrors(['error' => 'Erro ao salvar avaliações: ' . $e->getMessage()])
                ->withInput();
        }
    }

    public function edit(Avaliacao $avaliacao)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403);
        }

        $professor = $user->professor;
        
        // Verifica permissão
        $alocacao = $professor->turmas()
            ->wherePivot('turma_id', $avaliacao->turma_id)
            ->wherePivot('disciplina_id', $avaliacao->disciplina_id)
            ->first();

        if (!$alocacao) {
            abort(403);
        }

        return view('avaliacoes.edit', compact('avaliacao'));
    }

    public function update(Request $request, Avaliacao $avaliacao)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403);
        }

        $request->validate([
            'teste1' => 'nullable|numeric|min:0|max:20',
            'teste2' => 'nullable|numeric|min:0|max:20',
            'avaliacao_final' => 'nullable|numeric|min:0|max:20',
        ]);

        $professor = $user->professor;
        
        // Verifica permissão
        $alocacao = $professor->turmas()
            ->wherePivot('turma_id', $avaliacao->turma_id)
            ->wherePivot('disciplina_id', $avaliacao->disciplina_id)
            ->first();

        if (!$alocacao) {
            abort(403);
        }

        $dados = $avaliacao->toArray();
        $dados = array_merge($dados, $request->all());
        $turma = Turma::find($avaliacao->turma_id);
        $nivelEnsino = $turma?->nivel_ensino ?? 'PRIMARIO';
        $avaliacao = $this->avaliacaoService->salvarAvaliacao($dados, $nivelEnsino);

        return redirect()->route('avaliacoes.index')->with('success', 'Avaliação atualizada com sucesso!');
    }

    public function imprimirFicha(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403);
        }

        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'disciplina_id' => 'required|exists:disciplinas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ]);

        $professor = $user->professor;
        $turma = Turma::findOrFail($request->turma_id);
        $disciplina = Disciplina::findOrFail($request->disciplina_id);
        $anoLetivo = \App\Models\AnoLetivo::findOrFail($request->ano_letivo_id);

        // Verifica permissão
        $alocacao = $professor->turmas()
            ->wherePivot('turma_id', $turma->id)
            ->wherePivot('disciplina_id', $disciplina->id)
            ->first();

        if (!$alocacao) {
            abort(403, 'Você não leciona esta disciplina nesta turma.');
        }

        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->with('user')
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*')
            ->get();

        // Buscar TODOS os trimestres do ano letivo, ordenados por número
        $trimestres = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        if ($trimestres->isEmpty()) {
            abort(404, 'Nenhum trimestre encontrado para este ano letivo.');
        }

        // Log para debug (apenas em desenvolvimento)
        if (config('app.debug')) {
            \Log::info('Trimestres carregados para ficha de avaliações', [
                'ano_letivo_id' => $anoLetivo->id,
                'total' => $trimestres->count(),
                'trimestres' => $trimestres->pluck('numero', 'nome')->toArray()
            ]);
        }

        // Buscar todas as avaliações
        $avaliacoes = [];
        foreach ($alunos as $aluno) {
            $avaliacoes[$aluno->id] = [];
            foreach ($trimestres as $trimestre) {
                $avaliacoes[$aluno->id][$trimestre->id] = \App\Models\Avaliacao::where('aluno_id', $aluno->id)
                    ->where('disciplina_id', $disciplina->id)
                    ->where('turma_id', $turma->id)
                    ->where('trimestre_id', $trimestre->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->first();
            }
        }

        $configuracao = [
            'republica' => \App\Models\Configuracao::get('republica', 'REPÚBLICA DE MOÇAMBIQUE'),
            'distrito' => \App\Models\Configuracao::get('distrito', 'GOVERNO DO DISTRITO DE MOCUBA'),
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        return view('avaliacoes.imprimir-ficha', compact('turma', 'disciplina', 'anoLetivo', 'alunos', 'trimestres', 'avaliacoes', 'configuracao'));
    }

    public function todasDisciplinas(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor() && !$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        // Buscar todos os anos letivos e turmas para os filtros
        $anosLetivos = \App\Models\AnoLetivo::orderBy('ano', 'desc')->get();
        $todasTurmas = Turma::where('ativa', true)->orderBy('nome')->get();
        $todasDisciplinas = Disciplina::where('ativa', true)->orderBy('nome')->get();

        // Se não houver filtros, usar valores padrão
        $anoLetivoId = $request->get('ano_letivo_id') ?? \App\Models\AnoLetivo::where('ativo', true)->first()?->id;
        $turmaId = $request->get('turma_id');
        $disciplinaId = $request->get('disciplina_id');

        // Se não tiver turma selecionada, retornar apenas o formulário
        if (!$turmaId || !$anoLetivoId) {
            return view('avaliacoes.todas-disciplinas', compact(
                'anosLetivos', 'todasTurmas', 'todasDisciplinas', 
                'anoLetivoId', 'turmaId', 'disciplinaId'
            ));
        }

        $request->validate([
            'turma_id' => 'exists:turmas,id',
            'ano_letivo_id' => 'exists:ano_letivo,id',
            'disciplina_id' => 'nullable|exists:disciplinas,id',
        ]);

        $turma = Turma::findOrFail($turmaId);
        $anoLetivo = \App\Models\AnoLetivo::findOrFail($anoLetivoId);

        // Verificar permissão apenas para professores
        if ($user->isProfessor()) {
            $professor = $user->professor;
            if (!$professor) {
                abort(403, 'Seu perfil de professor não está completo.');
            }
            
            $alocacao = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('professor_id', $professor->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->first();

            if (!$alocacao) {
                abort(403, 'Você não leciona nesta turma.');
            }
        }

        // Buscar todas as disciplinas da turma
        $disciplinasIds = DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        if (empty($disciplinasIds)) {
            return view('avaliacoes.todas-disciplinas', compact(
                'anosLetivos', 'todasTurmas', 'todasDisciplinas', 
                'anoLetivoId', 'turmaId', 'disciplinaId'
            ))->with('error', 'Nenhuma disciplina encontrada para esta turma.');
        }

        // Aplicar filtro de disciplina se selecionado
        $queryDisciplinas = Disciplina::whereIn('id', $disciplinasIds);
        if ($disciplinaId) {
            $queryDisciplinas->where('id', $disciplinaId);
        }
        $disciplinas = $queryDisciplinas->orderBy('nome')->get();

        $trimestres = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->with('user')
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*')
            ->get();

        // Buscar todas as avaliações organizadas por disciplina
        $avaliacoesPorDisciplina = [];
        foreach ($disciplinas as $disciplina) {
            $avaliacoesPorDisciplina[$disciplina->id] = [];
            foreach ($alunos as $aluno) {
                $avaliacoesPorDisciplina[$disciplina->id][$aluno->id] = [];
                foreach ($trimestres as $trimestre) {
                    $avaliacoesPorDisciplina[$disciplina->id][$aluno->id][$trimestre->id] = 
                        \App\Models\Avaliacao::where('aluno_id', $aluno->id)
                            ->where('disciplina_id', $disciplina->id)
                            ->where('turma_id', $turma->id)
                            ->where('trimestre_id', $trimestre->id)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->first();
                }
            }
        }

        $configuracao = [
            'republica' => \App\Models\Configuracao::get('republica', 'REPÚBLICA DE MOÇAMBIQUE'),
            'distrito' => \App\Models\Configuracao::get('distrito', 'GOVERNO DO DISTRITO DE MOCUBA'),
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        return view('avaliacoes.todas-disciplinas', compact(
            'turma', 'disciplinas', 'anoLetivo', 'alunos', 'trimestres', 
            'avaliacoesPorDisciplina', 'configuracao', 'anosLetivos', 
            'todasTurmas', 'todasDisciplinas', 'anoLetivoId', 'turmaId', 'disciplinaId'
        ));
    }

    public function exportarExcel(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor() && !$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ]);

        $turma = Turma::findOrFail($request->turma_id);
        $anoLetivo = \App\Models\AnoLetivo::findOrFail($request->ano_letivo_id);

        // Buscar todas as disciplinas da turma
        $disciplinasIds = DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        $disciplinas = Disciplina::whereIn('id', $disciplinasIds)
            ->orderBy('nome')
            ->get();

        $trimestres = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->with('user')
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*')
            ->get();

        // Buscar todas as avaliações organizadas por disciplina
        $avaliacoesPorDisciplina = [];
        foreach ($disciplinas as $disciplina) {
            $avaliacoesPorDisciplina[$disciplina->id] = [];
            foreach ($alunos as $aluno) {
                $avaliacoesPorDisciplina[$disciplina->id][$aluno->id] = [];
                foreach ($trimestres as $trimestre) {
                    $avaliacoesPorDisciplina[$disciplina->id][$aluno->id][$trimestre->id] = 
                        \App\Models\Avaliacao::where('aluno_id', $aluno->id)
                            ->where('disciplina_id', $disciplina->id)
                            ->where('turma_id', $turma->id)
                            ->where('trimestre_id', $trimestre->id)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->first();
                }
            }
        }

        // Gerar conteúdo Excel (formato XML Excel)
        $excelContent = $this->gerarExcel($turma, $disciplinas, $trimestres, $alunos, $anoLetivo, $avaliacoesPorDisciplina);

        $filename = 'Avaliacoes_' . $turma->codigo . '_' . $turma->nome . '_' . date('Y-m-d') . '.xls';

        return response($excelContent, 200)
            ->header('Content-Type', 'application/vnd.ms-excel')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->header('Cache-Control', 'max-age=0');
    }

    public function downloadPdfTodasDisciplinas(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isProfessor() && !$user->isAdmin() && !$user->isSuperadmin() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ]);

        $turma = Turma::findOrFail($request->turma_id);
        $anoLetivo = \App\Models\AnoLetivo::findOrFail($request->ano_letivo_id);

        // Buscar todas as disciplinas da turma
        $disciplinasIds = DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        $disciplinas = Disciplina::whereIn('id', $disciplinasIds)
            ->orderBy('nome')
            ->get();

        $trimestres = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->with('user')
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*')
            ->get();

        // Buscar todas as avaliações organizadas por disciplina
        $avaliacoesPorDisciplina = [];
        foreach ($disciplinas as $disciplina) {
            $avaliacoesPorDisciplina[$disciplina->id] = [];
            foreach ($alunos as $aluno) {
                $avaliacoesPorDisciplina[$disciplina->id][$aluno->id] = [];
                foreach ($trimestres as $trimestre) {
                    $avaliacoesPorDisciplina[$disciplina->id][$aluno->id][$trimestre->id] = 
                        \App\Models\Avaliacao::where('aluno_id', $aluno->id)
                            ->where('disciplina_id', $disciplina->id)
                            ->where('turma_id', $turma->id)
                            ->where('trimestre_id', $trimestre->id)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->first();
                }
            }
        }

        $configuracao = [
            'republica' => \App\Models\Configuracao::get('republica', 'REPÚBLICA DE MOÇAMBIQUE'),
            'distrito' => \App\Models\Configuracao::get('distrito', 'GOVERNO DO DISTRITO DE MOCUBA'),
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        $pdf = Pdf::loadView('avaliacoes.pdf-todas-disciplinas', compact(
            'turma', 'disciplinas', 'trimestres', 'alunos', 'anoLetivo', 
            'avaliacoesPorDisciplina', 'configuracao'
        ))->setPaper('a4', 'landscape');

        $filename = 'Avaliacoes_' . $turma->codigo . '_' . $turma->nome . '_' . date('Y-m-d') . '.pdf';

        return $pdf->download($filename);
    }

    private function gerarExcel($turma, $disciplinas, $trimestres, $alunos, $anoLetivo, $avaliacoesPorDisciplina)
    {
        $html = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $html .= '<?mso-application progid="Excel.Sheet"?>' . "\n";
        $html .= '<Workbook xmlns="urn:schemas-microsoft-com:office:spreadsheet"' . "\n";
        $html .= ' xmlns:o="urn:schemas-microsoft-com:office:office"' . "\n";
        $html .= ' xmlns:x="urn:schemas-microsoft-com:office:excel"' . "\n";
        $html .= ' xmlns:ss="urn:schemas-microsoft-com:office:spreadsheet"' . "\n";
        $html .= ' xmlns:html="http://www.w3.org/TR/REC-html40">' . "\n";
        $html .= '<Styles>' . "\n";
        $html .= '<Style ss:ID="Header"><Font ss:Bold="1"/><Interior ss:Color="#D3D3D3" ss:Pattern="Solid"/><Alignment ss:Horizontal="Center" ss:Vertical="Center" ss:WrapText="1"/></Style>' . "\n";
        $html .= '<Style ss:ID="Bold"><Font ss:Bold="1"/></Style>' . "\n";
        $html .= '<Style ss:ID="Center"><Alignment ss:Horizontal="Center" ss:Vertical="Center"/></Style>' . "\n";
        $html .= '</Styles>' . "\n";

        $isSecundario = ($turma->nivel_ensino ?? 'PRIMARIO') === 'SECUNDARIO';

        // Criar uma folha para cada disciplina
        foreach ($disciplinas as $index => $disciplina) {
            $sheetName = $this->sanitizeSheetName($disciplina->nome);
            $html .= '<Worksheet ss:Name="' . htmlspecialchars($sheetName) . '">' . "\n";
            $html .= '<Table>' . "\n";

            // Cabeçalho
            $html .= '<Row>' . "\n";
            $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">N°</Data></Cell>' . "\n";
            $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">Nome completo do aluno</Data></Cell>' . "\n";
            $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">G</Data></Cell>' . "\n";
            
            foreach ($trimestres as $trimestre) {
                $merge = $isSecundario ? '3' : '6';
                $html .= '<Cell ss:StyleID="Header" ss:MergeAcross="' . $merge . '"><Data ss:Type="String">' . $trimestre->numero . '° Trimestre</Data></Cell>' . "\n";
            }
            $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">MFrD</Data></Cell>' . "\n";
            $html .= '</Row>' . "\n";

            // Sub-cabeçalho
            $html .= '<Row>' . "\n";
            $html .= '<Cell></Cell><Cell></Cell><Cell></Cell>' . "\n";
            foreach ($trimestres as $trimestre) {
                if ($isSecundario) {
                    $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">Teste 1</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">Teste 2</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">Av. Final</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">Média</Data></Cell>' . "\n";
                } else {
                $html .= '<Cell ss:StyleID="Header" ss:MergeAcross="2"><Data ss:Type="String">A.Prática</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">MAP</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header" ss:MergeAcross="2"><Data ss:Type="String">ACS</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">MACS</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">AT</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">MT</Data></Cell>' . "\n";
                }
            }
            $html .= '<Cell></Cell>' . "\n";
            $html .= '</Row>' . "\n";

            if (!$isSecundario) {
            // Sub-sub-cabeçalho
            $html .= '<Row>' . "\n";
            $html .= '<Cell></Cell><Cell></Cell><Cell></Cell>' . "\n";
            foreach ($trimestres as $trimestre) {
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">1ª</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">2ª</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">MAP</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">1ª</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">2ª</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">MACS</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">AT</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Header"><Data ss:Type="String">MT</Data></Cell>' . "\n";
            }
            $html .= '<Cell></Cell>' . "\n";
            $html .= '</Row>' . "\n";
            }

            // Dados dos alunos
            foreach ($alunos as $alunoIndex => $aluno) {
                $avaliacoesAluno = $avaliacoesPorDisciplina[$disciplina->id][$aluno->id] ?? [];
                
                $html .= '<Row>' . "\n";
                $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($alunoIndex + 1) . '</Data></Cell>' . "\n";
                $html .= '<Cell><Data ss:Type="String">' . htmlspecialchars($aluno->user->name ?? 'Aluno') . '</Data></Cell>' . "\n";
                $html .= '<Cell ss:StyleID="Center"><Data ss:Type="String">' . ($aluno->genero == 'M' ? 'H' : ($aluno->genero == 'F' ? 'M' : '')) . '</Data></Cell>' . "\n";
                
                foreach ($trimestres as $trimestre) {
                    $avaliacao = $avaliacoesAluno[$trimestre->id] ?? null;
                    
                    if ($isSecundario) {
                        $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->teste1 !== null ? number_format($avaliacao->teste1, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                        $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->teste2 !== null ? number_format($avaliacao->teste2, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                        $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->avaliacao_final !== null ? number_format($avaliacao->avaliacao_final, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                        $html .= '<Cell ss:StyleID="Bold"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->media_nacional !== null ? number_format($avaliacao->media_nacional, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    } else {
                    $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->a_pratica_1 !== null ? number_format($avaliacao->a_pratica_1, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->a_pratica_2 !== null ? number_format($avaliacao->a_pratica_2, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Bold"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->map !== null ? number_format($avaliacao->map, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->acs_1 !== null ? number_format($avaliacao->acs_1, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->acs_2 !== null ? number_format($avaliacao->acs_2, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Bold"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->macs !== null ? number_format($avaliacao->macs, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Center"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->at !== null ? number_format($avaliacao->at, 1, '.', '') : '') . '</Data></Cell>' . "\n";
                    $html .= '<Cell ss:StyleID="Bold"><Data ss:Type="Number">' . ($avaliacao && $avaliacao->mt !== null ? number_format($avaliacao->mt, 0, '.', '') : '') . '</Data></Cell>' . "\n";
                    }
                }
                
                // MFrD
                $primeiraAvaliacao = collect($avaliacoesAluno)->first();
                $mfrd = $primeiraAvaliacao ? $primeiraAvaliacao->mfrd : null;
                $html .= '<Cell ss:StyleID="Bold"><Data ss:Type="Number">' . ($mfrd !== null ? number_format($mfrd, 0, '.', '') : '') . '</Data></Cell>' . "\n";
                
                $html .= '</Row>' . "\n";
            }

            $html .= '</Table>' . "\n";
            $html .= '</Worksheet>' . "\n";
        }

        $html .= '</Workbook>';

        return $html;
    }

    private function sanitizeSheetName($name)
    {
        // Limitar a 31 caracteres (limite do Excel) e remover caracteres inválidos
        $name = preg_replace('/[\\\\\/\?\*\[\]:]/', '', $name);
        return mb_substr($name, 0, 31);
    }

    /**
     * Visualizar notas por estudante
     */
    public function notasPorEstudante(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->isProfessor() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();
        if (!$anoLetivo) {
            return redirect()->route('avaliacoes.index')->with('error', 'Nenhum ano letivo ativo encontrado.');
        }

        $turmaId = $request->get('turma_id');
        $alunoId = $request->get('aluno_id');

        $turmas = Turma::where('ano_letivo_id', $anoLetivo->id)
            ->where('ativa', true)
            ->orderBy('nome')
            ->get();

        $alunos = collect();
        $avaliacoesPorAluno = [];
        $disciplinas = Disciplina::where('ativa', true)->orderBy('nome')->get();
        $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        $turma = null;
        if ($turmaId) {
            $turma = Turma::findOrFail($turmaId);
            $query = $turma->alunos()
                ->wherePivot('ano_letivo_id', $anoLetivo->id)
                ->join('users', 'alunos.user_id', '=', 'users.id')
                ->orderBy('users.name')
                ->select('alunos.*');

            if ($alunoId) {
                $query->where('alunos.id', $alunoId);
            }

            $alunos = $query->get();

            foreach ($alunos as $aluno) {
                foreach ($disciplinas as $disciplina) {
                    foreach ($trimestres as $trimestre) {
                        $avaliacao = Avaliacao::where('aluno_id', $aluno->id)
                            ->where('disciplina_id', $disciplina->id)
                            ->where('turma_id', $turma->id)
                            ->where('trimestre_id', $trimestre->id)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->first();
                        
                        if ($avaliacao) {
                            $avaliacoesPorAluno[$aluno->id][$disciplina->id][$trimestre->id] = $avaliacao;
                        }
                    }
                }
            }
        }

        return view('avaliacoes.notas-por-estudante', compact(
            'turmas', 'alunos', 'avaliacoesPorAluno', 'disciplinas', 
            'trimestres', 'anoLetivo', 'turmaId', 'alunoId', 'turma'
        ));
    }

    /**
     * Imprimir/Download notas por estudante (PDF ou HTML para impressão)
     */
    public function imprimirNotasPorEstudante(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->isProfessor() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();
        if (!$anoLetivo) {
            return redirect()->route('avaliacoes.index')->with('error', 'Nenhum ano letivo ativo encontrado.');
        }

        $turmaId = $request->get('turma_id');
        $alunoId = $request->get('aluno_id');

        if (!$turmaId) {
            return redirect()->route('avaliacoes.notas-por-estudante')->with('error', 'Selecione uma turma.');
        }

        $turma = Turma::findOrFail($turmaId);
        $query = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*');

        if ($alunoId) {
            $query->where('alunos.id', $alunoId);
        }

        $alunos = $query->get();
        $disciplinas = Disciplina::where('ativa', true)->orderBy('nome')->get();
        $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        $avaliacoesPorAluno = [];
        foreach ($alunos as $aluno) {
            foreach ($disciplinas as $disciplina) {
                foreach ($trimestres as $trimestre) {
                    $avaliacao = Avaliacao::where('aluno_id', $aluno->id)
                        ->where('disciplina_id', $disciplina->id)
                        ->where('turma_id', $turma->id)
                        ->where('trimestre_id', $trimestre->id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->first();
                    
                    if ($avaliacao) {
                        $avaliacoesPorAluno[$aluno->id][$disciplina->id][$trimestre->id] = $avaliacao;
                    }
                }
            }
        }

        $configuracao = [
            'republica' => \App\Models\Configuracao::get('republica', 'REPÚBLICA DE MOÇAMBIQUE'),
            'distrito' => \App\Models\Configuracao::get('distrito', 'GOVERNO DO DISTRITO DE MOCUBA'),
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        return view('avaliacoes.imprimir-notas-por-estudante', compact(
            'turma', 'alunos', 'avaliacoesPorAluno', 'disciplinas', 
            'trimestres', 'anoLetivo', 'configuracao'
        ));
    }

    /**
     * Download PDF - Notas por Estudante
     */
    public function downloadPdfNotasPorEstudante(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->isProfessor() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();
        if (!$anoLetivo) {
            return redirect()->route('avaliacoes.index')->with('error', 'Nenhum ano letivo ativo encontrado.');
        }

        $turmaId = $request->get('turma_id');
        $alunoId = $request->get('aluno_id');

        if (!$turmaId) {
            return redirect()->route('avaliacoes.notas-por-estudante')->with('error', 'Selecione uma turma.');
        }

        $turma = Turma::findOrFail($turmaId);
        $query = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*');

        if ($alunoId) {
            $query->where('alunos.id', $alunoId);
        }

        $alunos = $query->get();
        $disciplinas = Disciplina::where('ativa', true)->orderBy('nome')->get();
        $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        $avaliacoesPorAluno = [];
        foreach ($alunos as $aluno) {
            foreach ($disciplinas as $disciplina) {
                foreach ($trimestres as $trimestre) {
                    $avaliacao = Avaliacao::where('aluno_id', $aluno->id)
                        ->where('disciplina_id', $disciplina->id)
                        ->where('turma_id', $turma->id)
                        ->where('trimestre_id', $trimestre->id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->first();
                    
                    if ($avaliacao) {
                        $avaliacoesPorAluno[$aluno->id][$disciplina->id][$trimestre->id] = $avaliacao;
                    }
                }
            }
        }

        $configuracao = [
            'republica' => \App\Models\Configuracao::get('republica', 'REPÚBLICA DE MOÇAMBIQUE'),
            'distrito' => \App\Models\Configuracao::get('distrito', 'GOVERNO DO DISTRITO DE MOCUBA'),
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        $pdf = Pdf::loadView('avaliacoes.imprimir-notas-por-estudante', compact(
            'turma', 'alunos', 'avaliacoesPorAluno', 'disciplinas', 
            'trimestres', 'anoLetivo', 'configuracao'
        ))->setPaper('a4', 'landscape');

        $filename = 'Notas_Estudantes_' . $turma->codigo . '_' . ($alunoId ? $alunos->first()->user->name : 'Todos') . '_' . date('Y-m-d') . '.pdf';

        return $pdf->download($filename);
    }

    /**
     * Estatísticas de aprovação/reprovação
     */
    public function estatisticas(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->isProfessor() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();
        if (!$anoLetivo) {
            return redirect()->route('avaliacoes.index')->with('error', 'Nenhum ano letivo ativo encontrado.');
        }

        $turmaId = $request->get('turma_id');
        $trimestreId = $request->get('trimestre_id');
        $tipo = $request->get('tipo', 'anual'); // 'trimestral' ou 'anual'

        $turmas = Turma::where('ano_letivo_id', $anoLetivo->id)
            ->where('ativa', true)
            ->orderBy('nome')
            ->get();

        $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero', 'asc')
            ->get();

        $estatisticas = [];

        if ($turmaId) {
            $turma = Turma::findOrFail($turmaId);
            $disciplinas = Disciplina::where('ativa', true)->orderBy('nome')->get();
            
            foreach ($disciplinas as $disciplina) {
                if ($tipo === 'trimestral' && $trimestreId) {
                    // Estatísticas por trimestre
                    $estatisticas[$disciplina->id] = $this->calcularEstatisticasTrimestre(
                        $turma->id, 
                        $disciplina->id, 
                        $trimestreId, 
                        $anoLetivo->id
                    );
                } else {
                    // Estatísticas anuais (baseado em MFrD)
                    $estatisticas[$disciplina->id] = $this->calcularEstatisticasAnuais(
                        $turma->id, 
                        $disciplina->id, 
                        $anoLetivo->id
                    );
                }
            }
        }

        return view('avaliacoes.estatisticas', compact(
            'turmas', 'trimestres', 'anoLetivo', 'turmaId', 'trimestreId', 
            'tipo', 'estatisticas'
        ));
    }

    /**
     * Calcular estatísticas por trimestre
     */
    private function calcularEstatisticasTrimestre($turmaId, $disciplinaId, $trimestreId, $anoLetivoId)
    {
        $alunos = Turma::find($turmaId)->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivoId)
            ->get();

        $aprovadosHomens = 0;
        $aprovadosMulheres = 0;
        $reprovadosHomens = 0;
        $reprovadosMulheres = 0;
        $desistiram = 0;

        foreach ($alunos as $aluno) {
            $avaliacao = Avaliacao::where('aluno_id', $aluno->id)
                ->where('disciplina_id', $disciplinaId)
                ->where('turma_id', $turmaId)
                ->where('trimestre_id', $trimestreId)
                ->where('ano_letivo_id', $anoLetivoId)
                ->first();

            if (!$avaliacao || $avaliacao->mt === null) {
                $desistiram++;
                continue;
            }

            $aprovado = $avaliacao->mt >= 10;
            $genero = $aluno->genero;

            if ($aprovado) {
                if ($genero === 'M') {
                    $aprovadosHomens++;
                } else {
                    $aprovadosMulheres++;
                }
            } else {
                if ($genero === 'M') {
                    $reprovadosHomens++;
                } else {
                    $reprovadosMulheres++;
                }
            }
        }

        return [
            'aprovados_homens' => $aprovadosHomens,
            'aprovados_mulheres' => $aprovadosMulheres,
            'reprovados_homens' => $reprovadosHomens,
            'reprovados_mulheres' => $reprovadosMulheres,
            'desistiram' => $desistiram,
            'total_aprovados' => $aprovadosHomens + $aprovadosMulheres,
            'total_reprovados' => $reprovadosHomens + $reprovadosMulheres,
            'total' => $alunos->count(),
        ];
    }

    /**
     * Calcular estatísticas anuais (baseado em MFrD)
     */
    private function calcularEstatisticasAnuais($turmaId, $disciplinaId, $anoLetivoId)
    {
        $alunos = Turma::find($turmaId)->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivoId)
            ->get();

        $aprovadosHomens = 0;
        $aprovadosMulheres = 0;
        $reprovadosHomens = 0;
        $reprovadosMulheres = 0;
        $desistiram = 0;

        foreach ($alunos as $aluno) {
            // Buscar qualquer avaliação da disciplina para pegar o MFrD
            $avaliacao = Avaliacao::where('aluno_id', $aluno->id)
                ->where('disciplina_id', $disciplinaId)
                ->where('turma_id', $turmaId)
                ->where('ano_letivo_id', $anoLetivoId)
                ->whereNotNull('mfrd')
                ->first();

            if (!$avaliacao || $avaliacao->mfrd === null) {
                $desistiram++;
                continue;
            }

            // Aprovado se MFrD >= 10 (com arredondamento por excesso de 9.5-9.99 para 10)
            $aprovado = \App\Models\Avaliacao::isAprovado($avaliacao->mfrd);
            $genero = $aluno->genero;

            if ($aprovado) {
                if ($genero === 'M') {
                    $aprovadosHomens++;
                } else {
                    $aprovadosMulheres++;
                }
            } else {
                if ($genero === 'M') {
                    $reprovadosHomens++;
                } else {
                    $reprovadosMulheres++;
                }
            }
        }

        return [
            'aprovados_homens' => $aprovadosHomens,
            'aprovados_mulheres' => $aprovadosMulheres,
            'reprovados_homens' => $reprovadosHomens,
            'reprovados_mulheres' => $reprovadosMulheres,
            'desistiram' => $desistiram,
            'total_aprovados' => $aprovadosHomens + $aprovadosMulheres,
            'total_reprovados' => $reprovadosHomens + $reprovadosMulheres,
            'total' => $alunos->count(),
        ];
    }

    /**
     * Situação Pedagógica - Aproveitamento Pedagógico do Trimestre
     */
    public function situacaoPedagogica(Request $request)
    {
        $user = auth()->user();
        
        if (!$user->isAdmin() && !$user->isSuperadmin() && !$user->isProfessor() && !$user->podeGerir('gerir_avaliacoes')) {
            abort(403);
        }

        $anoLetivo = \App\Models\AnoLetivo::where('ativo', true)->first();
        if (!$anoLetivo) {
            return redirect()->route('avaliacoes.index')->with('error', 'Nenhum ano letivo ativo encontrado.');
        }

        $turmaId = $request->get('turma_id');
        $trimestreId = $request->get('trimestre_id', 1); // Default primeiro trimestre

        if (!$turmaId) {
            $turmas = Turma::where('ano_letivo_id', $anoLetivo->id)
                ->where('ativa', true)
                ->orderBy('nome')
                ->get();
            
            $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
                ->orderBy('numero', 'asc')
                ->get();
            
            return view('avaliacoes.situacao-pedagogica-select', compact('turmas', 'trimestres', 'anoLetivo'));
        }

        $turma = Turma::findOrFail($turmaId);
        $trimestre = Trimestre::where('id', $trimestreId)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->firstOrFail();

        // Buscar alunos da turma ordenados por nome
        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->with('user')
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*')
            ->get();

        // Buscar disciplinas da turma
        $disciplinasIds = DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        $disciplinas = Disciplina::whereIn('id', $disciplinasIds)
            ->orderBy('nome')
            ->get();

        // Buscar avaliações e calcular médias
        $situacoes = [];
        foreach ($alunos as $aluno) {
            $mediasPorDisciplina = [];
            $totalMedias = 0;
            $disciplinasComNota = 0;

            foreach ($disciplinas as $disciplina) {
                $avaliacao = Avaliacao::where('aluno_id', $aluno->id)
                    ->where('disciplina_id', $disciplina->id)
                    ->where('turma_id', $turma->id)
                    ->where('trimestre_id', $trimestre->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->first();

                $media = null;
                if ($avaliacao && $avaliacao->mt !== null) {
                    $media = round($avaliacao->mt, 0); // Média Trimestral (MT) arredondada
                    $totalMedias += $media;
                    $disciplinasComNota++;
                }

                $mediasPorDisciplina[$disciplina->id] = $media;
            }

            // Calcular média trimestral geral (média das médias das disciplinas)
            $mediaTrimestralGeral = $disciplinasComNota > 0 ? round($totalMedias / $disciplinasComNota, 0) : null;

            // Classificar média trimestral
            $classificacaoMedia = $this->classificarMedia($mediaTrimestralGeral);

            $situacoes[] = [
                'aluno' => $aluno,
                'mediasPorDisciplina' => $mediasPorDisciplina,
                'mediaTrimestralGeral' => $mediaTrimestralGeral,
                'classificacaoMedia' => $classificacaoMedia,
            ];
        }

        $configuracao = [
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        return view('avaliacoes.situacao-pedagogica', compact(
            'turma', 'trimestre', 'anoLetivo', 'alunos', 'disciplinas', 
            'situacoes', 'configuracao'
        ));
    }

    /**
     * Classificar média por faixa
     */
    private function classificarMedia($media)
    {
        if ($media === null) {
            return '-';
        }

        $mediaInt = (int) $media;

        if ($mediaInt >= 0 && $mediaInt <= 9) {
            return 'Não Satisfatório';
        } elseif ($mediaInt >= 10 && $mediaInt <= 13) {
            return 'Satisfatório';
        } elseif ($mediaInt >= 14 && $mediaInt <= 16) {
            return 'Bom';
        } elseif ($mediaInt >= 17 && $mediaInt <= 18) {
            return 'Muito Bom';
        } elseif ($mediaInt >= 19 && $mediaInt <= 20) {
            return 'Excelente';
        }

        return '-';
    }
}

