<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\Funcionario;
use App\Models\Auditoria;
use App\Models\LoginAttempt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function showLoginForm()
    {
        return view('auth.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'login' => 'required|string',
            'password' => 'required|string',
        ]);

        $login = $request->input('login');
        $password = $request->input('password');

        // Determina se é email, código de estudante ou número de funcionário
        $user = null;
        if (filter_var($login, FILTER_VALIDATE_EMAIL)) {
            $user = User::where('email', $login)->first();
        } else {
            // Tenta como código de estudante
            $aluno = Aluno::where('codigo_estudante', $login)->first();
            if ($aluno) {
                $user = $aluno->user;
            } else {
                // Tenta como número de funcionário do professor
                $professor = Professor::where('numero_funcionario', $login)->first();
                if ($professor) {
                    $user = $professor->user;
                } else {
                    // Tenta como número de funcionário administrativo
                    $funcionario = Funcionario::where('numero_funcionario', $login)->first();
                    if ($funcionario) {
                        $user = $funcionario->user;
                    }
                }
            }
        }

        if (!$user || !Hash::check($password, $user->password)) {
            LoginAttempt::create([
                'login' => $login,
                'ip_address' => $request->ip(),
                'success' => false,
                'user_agent' => $request->userAgent(),
                'motivo' => 'credenciais_invalidas',
            ]);
            throw ValidationException::withMessages([
                'login' => ['Credenciais inválidas.'],
            ]);
        }

        if (!$user->is_active) {
            LoginAttempt::create([
                'login' => $login,
                'ip_address' => $request->ip(),
                'success' => false,
                'user_id' => $user->id,
                'user_agent' => $request->userAgent(),
                'motivo' => 'conta_desativada',
            ]);
            throw ValidationException::withMessages([
                'login' => ['Sua conta está desativada.'],
            ]);
        }

        Auth::login($user, $request->boolean('remember'));

        $user->update(['last_login_at' => now()]);

        LoginAttempt::create([
            'login' => $login,
            'ip_address' => $request->ip(),
            'success' => true,
            'user_id' => $user->id,
            'user_agent' => $request->userAgent(),
        ]);

        Auditoria::log('LOGIN', 'User', $user->id, null, ['email' => $user->email]);

        // Redireciona conforme o tipo de usuário
        if ($user->must_change_password) {
            return redirect()->route('password.change');
        }

        return redirect()->intended(route('dashboard'));
    }

    public function logout(Request $request)
    {
        Auditoria::log('LOGOUT', 'User', auth()->id());
        
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login');
    }

    public function showPasswordChangeForm()
    {
        return view('auth.change-password');
    }

    public function changePassword(Request $request)
    {
        $user = auth()->user();

        // Se o usuário precisa alterar a senha (primeiro login), não exige senha atual
        if ($user->must_change_password) {
            $request->validate([
                'password' => 'required|string|min:8|confirmed',
            ]);
        } else {
            // Se não é primeiro login, exige senha atual
            $request->validate([
                'current_password' => 'required',
                'password' => 'required|string|min:8|confirmed',
            ]);

            // Verifica se a senha atual está correta
            if (!Hash::check($request->current_password, $user->password)) {
                return back()->withErrors(['current_password' => 'Senha atual incorreta.']);
            }
        }

        $user->password = Hash::make($request->password);
        $user->must_change_password = false;
        $user->save();

        Auditoria::log('UPDATE', 'User', $user->id, null, ['password_changed' => true]);

        return redirect()->route('dashboard')->with('success', 'Senha alterada com sucesso!');
    }

    public function showForgotPasswordForm()
    {
        return view('auth.forgot-password');
    }

    public function sendResetLink(Request $request)
    {
        $request->validate(['email' => 'required|email']);

        $email = $request->input('email');
        $user = User::where('email', $email)->first();

        if (!$user) {
            return back()->withErrors(['email' => 'Não encontramos um usuário com esse endereço de email.']);
        }

        // Verifica se o usuário está ativo
        if (!$user->is_active) {
            return back()->withErrors(['email' => 'Esta conta está desativada. Entre em contato com o administrador.']);
        }

        // Apenas admin e superadmin podem recuperar senha via email
        // Alunos, professores e funcionários usam códigos para login, não email
        if (!$user->isAdmin() && !$user->isSuperadmin()) {
            return back()->withErrors(['email' => 'Recuperação de senha via email disponível apenas para administradores. Alunos, professores e funcionários devem entrar em contato com a administração da escola para redefinir sua senha.']);
        }

        // Permite recuperação de senha apenas para admin/superadmin
        $status = Password::sendResetLink(
            ['email' => $email]
        );

        return $status === Password::RESET_LINK_SENT
            ? back()->with(['status' => 'Enviamos um link de recuperação de senha para o email cadastrado.'])
            : back()->withErrors(['email' => __($status)]);
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $status = Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->password = Hash::make($password);
                $user->must_change_password = false;
                $user->save();

                Auditoria::log('PASSWORD_RESET', 'User', $user->id, null, [
                    'reset_via' => 'email_link',
                ]);
            }
        );

        return $status == Password::PASSWORD_RESET
            ? redirect()->route('login')->with('status', __($status))
            : back()->withErrors(['email' => [__($status)]]);
    }
}



