<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Horario;
use App\Models\Evento;
use App\Models\AnoLetivo;
use App\Models\Financeiro;
use App\Models\Configuracao;
use App\Models\PresencaProfessor;
use App\Repositories\AlunoRepository;
use App\Repositories\ProfessorRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;
use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;
use chillerlan\QRCode\Output\QROutputInterface;

class AdminController extends Controller
{
    protected $alunoRepository;
    protected $professorRepository;

    public function __construct(AlunoRepository $alunoRepository, ProfessorRepository $professorRepository)
    {
        $this->middleware(['auth', 'role:admin,superadmin,funcionario']);
        $this->alunoRepository = $alunoRepository;
        $this->professorRepository = $professorRepository;
    }

    /**
     * Verifica se o usuário tem permissão para acessar o recurso
     */
    private function verificarPermissao($recurso)
    {
        $user = auth()->user();
        
        // Admin e Superadmin têm todas as permissões
        if ($user->isAdmin() || $user->isSuperadmin()) {
            return true;
        }
        
        // Funcionários precisam ter a permissão específica
        if ($user->isFuncionario()) {
            if (!$user->podeGerir($recurso)) {
                abort(403, 'Acesso negado. Você não tem permissão para acessar este recurso.');
            }
        }
        
        return true;
    }

    /**
     * Classificar média por faixa
     */
    private function classificarMedia($media)
    {
        if ($media === null) {
            return '-';
        }

        $mediaInt = (int) $media;

        if ($mediaInt >= 0 && $mediaInt <= 9) {
            return 'Não Satisfatório';
        } elseif ($mediaInt >= 10 && $mediaInt <= 13) {
            return 'Satisfatório';
        } elseif ($mediaInt >= 14 && $mediaInt <= 16) {
            return 'Bom';
        } elseif ($mediaInt >= 17 && $mediaInt <= 18) {
            return 'Muito Bom';
        } elseif ($mediaInt >= 19 && $mediaInt <= 20) {
            return 'Excelente';
        }

        return '-';
    }

    // Gestão de Alunos
    public function indexAlunos(Request $request)
    {
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        
        // Filtro de ano letivo (usa o selecionado ou o ativo por padrão)
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtivo?->id);
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : $anoLetivoAtivo;
        
        $query = Aluno::select('alunos.*')->with(['user']);
        
        // Pesquisa por código ou nome
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('codigo_estudante', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filtro por classe (codigo da turma)
        if ($request->filled('classe')) {
            if ($anoLetivoSelecionado) {
                $query->whereExists(function($q) use ($request, $anoLetivoSelecionado) {
                    $q->select(DB::raw(1))
                      ->from('turma_aluno')
                      ->join('turmas', 'turma_aluno.turma_id', '=', 'turmas.id')
                      ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                      ->where('turmas.codigo', $request->classe)
                      ->where('turma_aluno.ano_letivo_id', $anoLetivoSelecionado->id);
                });
            } else {
                $query->whereHas('turmas', function($q) use ($request) {
                    $q->where('turmas.codigo', $request->classe);
                });
            }
        }
        
        // Filtro por turma (nome da turma)
        if ($request->filled('turma')) {
            if ($anoLetivoSelecionado) {
                $query->whereExists(function($q) use ($request, $anoLetivoSelecionado) {
                    $q->select(DB::raw(1))
                      ->from('turma_aluno')
                      ->join('turmas', 'turma_aluno.turma_id', '=', 'turmas.id')
                      ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                      ->where('turmas.nome', $request->turma)
                      ->where('turma_aluno.ano_letivo_id', $anoLetivoSelecionado->id);
                });
            } else {
                $query->whereHas('turmas', function($q) use ($request) {
                    $q->where('turmas.nome', $request->turma);
                });
            }
        }
        
        // Otimização: Carregar apenas colunas necessárias e usar eager loading ANTES da paginação
        $alunos = $query->select('alunos.id', 'alunos.codigo_estudante', 'alunos.user_id', 'alunos.telefone')
            ->with([
                'user:id,name,is_active',
                'turmas' => function($q) use ($anoLetivoSelecionado) {
                    $q->select('turmas.id', 'turmas.nome', 'turmas.codigo', 'turmas.nivel_ensino')
                      ->wherePivot('ano_letivo_id', $anoLetivoSelecionado?->id);
                }
            ])
            ->distinct()
            ->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());
        
        // Buscar classes e turmas disponíveis para os filtros (baseado no ano letivo selecionado)
        $classes = Turma::where('ano_letivo_id', $anoLetivoSelecionado?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('codigo')
            ->sort()
            ->values();
            
        $turmas = Turma::where('ano_letivo_id', $anoLetivoSelecionado?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('nome')
            ->sort()
            ->values();
        
        $anosLetivos = AnoLetivo::getAnosLetivosDisponiveis();
        
        return view('admin.alunos.index', compact('alunos', 'classes', 'turmas', 'anosLetivos', 'anoLetivoId', 'anoLetivoSelecionado'));
    }

    public function createAluno()
    {
        // Para novos ingressos, usar anos letivos disponíveis para matrícula
        $anosLetivos = AnoLetivo::getAnosLetivosParaMatricula();
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->orderBy('codigo')
            ->orderBy('nome')
            ->get();
        $classes = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('codigo')
            ->sort()
            ->values();
        
        return view('admin.alunos.create', compact('turmas', 'classes', 'anoLetivo', 'anosLetivos'));
    }

    public function storeAluno(Request $request)
    {
        // Proteção contra submissão duplicada
        if ($request->has('_token') && session()->has('last_aluno_created_' . $request->input('name'))) {
            $lastCreated = session()->get('last_aluno_created_' . $request->input('name'));
            if ($lastCreated && (time() - $lastCreated) < 5) {
                return redirect()->back()->with('error', 'Por favor, aguarde alguns segundos antes de criar novamente.')->withInput();
            }
        }
        
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:users,email',
            'codigo_estudante' => 'nullable|string|unique:alunos,codigo_estudante',
            'telefone' => 'nullable|string',
            'endereco' => 'nullable|string',
            'data_nascimento' => 'nullable|date',
            'local_nascimento' => 'nullable|string',
            'naturalidade' => 'nullable|string',
            'distrito_natural' => 'nullable|string',
            'provincia_natural' => 'nullable|string',
            'num_irmaos_mais_velhos' => 'nullable|integer|min:0',
            'num_irmaos_mais_novos' => 'nullable|integer|min:0',
            'num_irmaos_mesma_idade' => 'nullable|integer|min:0',
            'num_documento' => 'nullable|string',
            'tipo_documento' => 'nullable|in:BI,Cédula,Passaporte',
            'foto' => 'nullable|image|mimes:jpeg,jpg,png|max:2048',
            'foto_base64' => 'nullable|string',
            'local_emissao' => 'nullable|string',
            'data_emissao' => 'nullable|date',
            'nome_pai' => 'nullable|string',
            'local_trabalho_pai' => 'nullable|string',
            'endereco_pai_bairro' => 'nullable|string',
            'endereco_pai_rua' => 'nullable|string',
            'endereco_pai_numero' => 'nullable|string',
            'endereco_pai_quarteirao' => 'nullable|string',
            'endereco_pai_flat' => 'nullable|string',
            'telefone_pai' => 'nullable|string',
            'bi_pai' => 'nullable|string',
            'nome_mae' => 'nullable|string',
            'local_trabalho_mae' => 'nullable|string',
            'endereco_mae_bairro' => 'nullable|string',
            'endereco_mae_rua' => 'nullable|string',
            'endereco_mae_numero' => 'nullable|string',
            'endereco_mae_quarteirao' => 'nullable|string',
            'endereco_mae_flat' => 'nullable|string',
            'telefone_mae' => 'nullable|string',
            'bi_mae' => 'nullable|string',
            'genero' => 'nullable|in:M,F,Outro',
            'plano_saude' => 'nullable|string',
            'problema_saude_cronico' => 'nullable|string',
            'alergias' => 'nullable|string',
            'diagnostico_deficiencia' => 'nullable|string',
            'dificuldade_motora_auditiva_visual_fala_emocional' => 'nullable|string',
            'tratamento_medico' => 'nullable|string',
            'medicacao' => 'nullable|string',
            'vacinas_em_dia' => 'nullable|boolean',
            'contato_emergencia' => 'nullable|string',
            'outras_informacoes_saude' => 'nullable|string',
            'individuo_autorizado_1_nome' => 'nullable|string',
            'individuo_autorizado_1_telefone' => 'nullable|string',
            'individuo_autorizado_2_nome' => 'nullable|string',
            'individuo_autorizado_2_telefone' => 'nullable|string',
            'individuo_autorizado_3_nome' => 'nullable|string',
            'individuo_autorizado_3_telefone' => 'nullable|string',
            'nome_encarregado' => 'nullable|string',
            'telefone_encarregado' => 'nullable|string',
            'endereco_encarregado' => 'nullable|string',
            'profissao_encarregado' => 'nullable|string',
            'bi_encarregado' => 'nullable|string',
            'grau_parentesco' => 'nullable|string',
            'pago' => 'nullable|boolean',
            'modalidade_pagamento' => 'nullable|in:MENSAL,TRIMESTRAL,SEMESTRAL,ANUAL',
            'valor_mensalidade_personalizado' => 'nullable|numeric|min:0',
            'ano_ingresso' => 'nullable|integer|min:2000|max:' . date('Y'),
            'turma_id' => 'nullable|exists:turmas,id',
            'ano_letivo_id' => 'nullable|exists:ano_letivo,id',
        ]);

        $dados = $request->except(['foto', 'foto_base64']);
        $dados['pago'] = $request->boolean('pago', true);
        
        // Processar upload de foto
        if ($request->hasFile('foto')) {
            $arquivo = $request->file('foto');
            $nomeArquivo = time() . '_' . uniqid() . '.' . $arquivo->getClientOriginalExtension();
            $caminho = $arquivo->storeAs('fotos/alunos', $nomeArquivo, 'public');
            $dados['foto'] = $caminho;
        } elseif ($request->filled('foto_base64')) {
            // Processar foto capturada da webcam (base64)
            $imageData = $request->foto_base64;
            if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
                $type = strtolower($type[1]);
                
                if (!in_array($type, ['jpg', 'jpeg', 'png'])) {
                    return redirect()->back()->withErrors(['foto' => 'Formato de imagem inválido. Use JPG ou PNG.'])->withInput();
                }
                
                $imageData = base64_decode($imageData);
                $nomeArquivo = time() . '_' . uniqid() . '.' . $type;
                $caminho = 'fotos/alunos/' . $nomeArquivo;
                
                Storage::disk('public')->put($caminho, $imageData);
                $dados['foto'] = $caminho;
            }
        }
        
        // Construir data_nascimento a partir dos campos individuais se não fornecida
        if (!$request->filled('data_nascimento') && $request->filled('dia_nascimento') && $request->filled('mes_nascimento') && $request->filled('ano_nascimento')) {
            $dia = str_pad($request->dia_nascimento, 2, '0', STR_PAD_LEFT);
            $mes = str_pad($request->mes_nascimento, 2, '0', STR_PAD_LEFT);
            $ano = $request->ano_nascimento;
            $dados['data_nascimento'] = "{$ano}-{$mes}-{$dia}";
        }
        
        $anoIngresso = $request->ano_ingresso ?? date('Y');

        // Verificar se já existe aluno com mesmo nome e email (proteção adicional)
        if ($request->filled('email')) {
            $alunoExistente = \App\Models\User::where('email', $request->email)->first();
            if ($alunoExistente && $alunoExistente->aluno) {
                return redirect()->back()->with('error', 'Já existe um aluno com este email.')->withInput();
            }
        }

        $aluno = $this->alunoRepository->create($dados, $anoIngresso);
        
        // Registrar criação para proteção contra duplicação
        session()->put('last_aluno_created_' . $request->input('name'), time());
        
        // Alocar aluno à turma se selecionada
        if ($request->filled('turma_id')) {
            $anoLetivoId = $request->filled('ano_letivo_id') 
                ? $request->ano_letivo_id 
                : AnoLetivo::getAnoLetivoAtual()?->id;
            
            if ($anoLetivoId) {
                $turma = Turma::find($request->turma_id);
                if ($turma) {
                    // Verificar se já existe alocação para este aluno, turma e ano letivo
                    $existe = DB::table('turma_aluno')
                        ->where('aluno_id', $aluno->id)
                        ->where('turma_id', $turma->id)
                        ->where('ano_letivo_id', $anoLetivoId)
                        ->exists();
                    
                    if (!$existe) {
                        // Verificar capacidade máxima da turma (35 alunos)
                        $capacidadeMaxima = $turma->capacidade_maxima ?? 35;
                        $alunosNaTurma = DB::table('turma_aluno')
                            ->where('turma_id', $turma->id)
                            ->where('ano_letivo_id', $anoLetivoId)
                            ->count();
                        
                        if ($alunosNaTurma >= $capacidadeMaxima) {
                            return redirect()->back()
                                ->withInput()
                                ->with('error', "A turma {$turma->codigo} - {$turma->nome} já atingiu a capacidade máxima de {$capacidadeMaxima} alunos.");
                        }
                        
                        // Alocar à nova turma
                        $aluno->turmas()->attach($turma->id, ['ano_letivo_id' => $anoLetivoId]);
                        
                        // Definir valor da mensalidade baseado na classe da turma (se não tiver valor personalizado)
                        if (!$aluno->valor_mensalidade_personalizado || $aluno->valor_mensalidade_personalizado <= 0) {
                            $financeiroService = app(\App\Services\FinanceiroService::class);
                            $valorMensalidade = $financeiroService->getValorMensalidadePorTurma($turma);
                            $aluno->update(['valor_mensalidade_personalizado' => $valorMensalidade]);
                        }
                    }
                }
            }
        }

        // Redirecionar para a ficha de inscrição em PDF
        return redirect()->route('admin.alunos.ficha-inscricao', $aluno);
    }

    public function editAluno(Aluno $aluno)
    {
        $aluno->load('user');
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        $turmasBase = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->withCount([
                'alunos as alunos_ano_count' => function ($q) use ($anoLetivo) {
                    if ($anoLetivo?->id) {
                        $q->where('turma_aluno.ano_letivo_id', $anoLetivo->id);
                    }
                }
            ])
            ->orderBy('codigo')
            ->orderBy('nome')
            ->get();
        $classes = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('codigo')
            ->sort()
            ->values();
        
        // Buscar turma atual do aluno no ano letivo ativo
        $turmaAtual = null;
        if ($anoLetivo) {
            $turmaAtual = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivo->id)
                ->first();
        }

        $turmas = $turmasBase->filter(function ($turma) use ($turmaAtual) {
            if ($turmaAtual && $turma->id === $turmaAtual->id) {
                return true;
            }
            $capacidadeMaxima = $turma->capacidade_maxima ?? 35;
            return ($turma->alunos_ano_count ?? 0) < $capacidadeMaxima;
        })->values();
        
        return view('admin.alunos.edit', compact('aluno', 'turmas', 'classes', 'anoLetivo', 'anosLetivos', 'turmaAtual'));
    }

    public function updateAluno(Request $request, Aluno $aluno)
    {
        // Remover campos vazios de data antes da validação
        $requestData = $request->all();
        foreach (['dia_nascimento', 'mes_nascimento', 'ano_nascimento'] as $field) {
            if (isset($requestData[$field]) && ($requestData[$field] === '' || $requestData[$field] === null || trim((string)$requestData[$field]) === '')) {
                unset($requestData[$field]);
            } elseif (isset($requestData[$field]) && is_numeric($requestData[$field])) {
                $requestData[$field] = (int)$requestData[$field];
            }
        }
        $request->replace($requestData);
        
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:users,email,' . $aluno->user_id,
            'telefone' => 'nullable|string',
            'endereco' => 'nullable|string',
            'data_nascimento' => 'nullable|date',
            'dia_nascimento' => 'nullable|sometimes|integer|min:1|max:31',
            'mes_nascimento' => 'nullable|sometimes|integer|min:1|max:12',
            'ano_nascimento' => 'nullable|sometimes|integer|min:1900|max:' . date('Y'),
            'local_nascimento' => 'nullable|string',
            'naturalidade' => 'nullable|string',
            'distrito_natural' => 'nullable|string',
            'provincia_natural' => 'nullable|string',
            'num_documento' => 'nullable|string',
            'tipo_documento' => 'nullable|in:BI,Cédula,Passaporte',
            'foto' => 'nullable|image|mimes:jpeg,jpg,png|max:2048',
            'foto_base64' => 'nullable|string',
            'remove_foto' => 'nullable|boolean',
            'local_emissao' => 'nullable|string',
            'data_emissao' => 'nullable|date',
            'num_irmaos_mais_velhos' => 'nullable|integer|min:0',
            'num_irmaos_mais_novos' => 'nullable|integer|min:0',
            'num_irmaos_mesma_idade' => 'nullable|integer|min:0',
            'nome_pai' => 'nullable|string',
            'local_trabalho_pai' => 'nullable|string',
            'endereco_pai_bairro' => 'nullable|string',
            'endereco_pai_rua' => 'nullable|string',
            'endereco_pai_numero' => 'nullable|string',
            'endereco_pai_quarteirao' => 'nullable|string',
            'endereco_pai_flat' => 'nullable|string',
            'telefone_pai' => 'nullable|string',
            'bi_pai' => 'nullable|string',
            'nome_mae' => 'nullable|string',
            'local_trabalho_mae' => 'nullable|string',
            'endereco_mae_bairro' => 'nullable|string',
            'endereco_mae_rua' => 'nullable|string',
            'endereco_mae_numero' => 'nullable|string',
            'endereco_mae_quarteirao' => 'nullable|string',
            'endereco_mae_flat' => 'nullable|string',
            'telefone_mae' => 'nullable|string',
            'bi_mae' => 'nullable|string',
            'genero' => 'nullable|in:M,F,Outro',
            'plano_saude' => 'nullable|string',
            'problema_saude_cronico' => 'nullable|string',
            'alergias' => 'nullable|string',
            'diagnostico_deficiencia' => 'nullable|string',
            'dificuldade_motora_auditiva_visual_fala_emocional' => 'nullable|string',
            'tratamento_medico' => 'nullable|string',
            'medicacao' => 'nullable|string',
            'vacinas_em_dia' => 'nullable|boolean',
            'contato_emergencia' => 'nullable|string',
            'outras_informacoes_saude' => 'nullable|string',
            'individuo_autorizado_1_nome' => 'nullable|string',
            'individuo_autorizado_1_telefone' => 'nullable|string',
            'individuo_autorizado_2_nome' => 'nullable|string',
            'individuo_autorizado_2_telefone' => 'nullable|string',
            'individuo_autorizado_3_nome' => 'nullable|string',
            'individuo_autorizado_3_telefone' => 'nullable|string',
            'nome_encarregado' => 'nullable|string',
            'telefone_encarregado' => 'nullable|string',
            'endereco_encarregado' => 'nullable|string',
            'profissao_encarregado' => 'nullable|string',
            'bi_encarregado' => 'nullable|string',
            'grau_parentesco' => 'nullable|string',
            'pago' => 'nullable|boolean',
            'modalidade_pagamento' => 'nullable|in:MENSAL,TRIMESTRAL,SEMESTRAL,ANUAL',
            'valor_mensalidade_personalizado' => 'nullable|numeric|min:0',
            'turma_id' => 'nullable|exists:turmas,id',
        ]);

        $dados = $request->except(['foto', 'foto_base64', 'remove_foto']);
        $dados['pago'] = $request->boolean('pago', $aluno->pago ?? true);
        
        // Processar upload de foto
        if ($request->hasFile('foto')) {
            // Deletar foto antiga se existir
            if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
                Storage::disk('public')->delete($aluno->foto);
            }
            $arquivo = $request->file('foto');
            $nomeArquivo = time() . '_' . uniqid() . '.' . $arquivo->getClientOriginalExtension();
            $caminho = $arquivo->storeAs('fotos/alunos', $nomeArquivo, 'public');
            $dados['foto'] = $caminho;
        } elseif ($request->filled('foto_base64')) {
            // Processar foto capturada da webcam (base64)
            $imageData = $request->foto_base64;
            if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
                // Deletar foto antiga se existir
                if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
                    Storage::disk('public')->delete($aluno->foto);
                }
                $imageData = substr($imageData, strpos($imageData, ',') + 1);
                $type = strtolower($type[1]);
                
                if (!in_array($type, ['jpg', 'jpeg', 'png'])) {
                    return redirect()->back()->withErrors(['foto' => 'Formato de imagem inválido. Use JPG ou PNG.'])->withInput();
                }
                
                $imageData = base64_decode($imageData);
                $nomeArquivo = time() . '_' . uniqid() . '.' . $type;
                $caminho = 'fotos/alunos/' . $nomeArquivo;
                
                Storage::disk('public')->put($caminho, $imageData);
                $dados['foto'] = $caminho;
            }
        } elseif ($request->boolean('remove_foto')) {
            // Remover foto se checkbox marcada
            if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
                Storage::disk('public')->delete($aluno->foto);
                $dados['foto'] = null;
            }
        }
        
        // Construir data_nascimento a partir dos campos individuais se não fornecida
        if (!$request->filled('data_nascimento') && $request->filled('dia_nascimento') && $request->filled('mes_nascimento') && $request->filled('ano_nascimento')) {
            $dia = str_pad($request->dia_nascimento, 2, '0', STR_PAD_LEFT);
            $mes = str_pad($request->mes_nascimento, 2, '0', STR_PAD_LEFT);
            $ano = $request->ano_nascimento;
            $dados['data_nascimento'] = "{$ano}-{$mes}-{$dia}";
        }
        
        $this->alunoRepository->update($aluno, $dados);
        
        // Atualizar alocação do aluno à turma
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        if ($anoLetivo && $request->filled('turma_id')) {
            $turma = Turma::find($request->turma_id);
            if ($turma) {
                // Verificar se já existe alocação para este aluno, turma e ano letivo
                $existe = DB::table('turma_aluno')
                    ->where('aluno_id', $aluno->id)
                    ->where('turma_id', $turma->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->exists();
                
                if (!$existe) {
                    // Verificar capacidade máxima da turma (35 alunos)
                    $capacidadeMaxima = $turma->capacidade_maxima ?? 35;
                    $alunosNaTurma = DB::table('turma_aluno')
                        ->where('turma_id', $turma->id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->count();
                    
                    if ($alunosNaTurma >= $capacidadeMaxima) {
                        return redirect()->back()
                            ->with('error', "A turma {$turma->codigo} - {$turma->nome} já atingiu a capacidade máxima de {$capacidadeMaxima} alunos.");
                    }
                    
                    // Remover alocação anterior no mesmo ano letivo (se houver)
                    DB::table('turma_aluno')
                        ->where('aluno_id', $aluno->id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->where('turma_id', '!=', $turma->id)
                        ->delete();
                    
                    // Adicionar nova alocação
                    $aluno->turmas()->attach($turma->id, ['ano_letivo_id' => $anoLetivo->id]);
                    
                    // Definir valor da mensalidade baseado na classe da turma (se não tiver valor personalizado)
                    if (!$aluno->valor_mensalidade_personalizado || $aluno->valor_mensalidade_personalizado <= 0) {
                        $financeiroService = app(\App\Services\FinanceiroService::class);
                        $valorMensalidade = $financeiroService->getValorMensalidadePorTurma($turma);
                        $aluno->update(['valor_mensalidade_personalizado' => $valorMensalidade]);
                    }
                }
            }
        }

        return redirect()->route('admin.alunos.edit', $aluno)->with('success', 'Aluno atualizado com sucesso!');
    }

    public function showAluno(Aluno $aluno)
    {
        $aluno->load('user');
        return view('admin.alunos.show', compact('aluno'));
    }

    // Alocar aluno a uma turma
    public function alocarAluno(Aluno $aluno, Request $request)
    {
        $anoLetivoId = $request->get('ano_letivo_id');
        $anoLetivo = $anoLetivoId 
            ? AnoLetivo::find($anoLetivoId) 
            : AnoLetivo::getAnoLetivoAtual();
        
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        $turmas = collect();
        if ($anoLetivo) {
            $turmas = Turma::where('ano_letivo_id', $anoLetivo->id)
                ->where('ativa', true)
                ->orderBy('codigo')
                ->orderBy('nome')
                ->get();
        }
        
        // Buscar turma atual do aluno no ano letivo selecionado
        $turmaAtual = null;
        if ($anoLetivo) {
            $turmaAtual = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivo->id)
                ->first();
        }
        
        return view('admin.alunos.alocar', compact('aluno', 'turmas', 'anoLetivo', 'anosLetivos', 'turmaAtual'));
    }

    public function storeAlocarAluno(Request $request, Aluno $aluno)
    {
        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ]);
        
        $anoLetivo = AnoLetivo::findOrFail($request->ano_letivo_id);
        $turma = Turma::findOrFail($request->turma_id);
        
        // Verificar se a turma pertence ao ano letivo
        if ($turma->ano_letivo_id != $anoLetivo->id) {
            return redirect()->back()
                ->with('error', 'A turma selecionada não pertence ao ano letivo escolhido.');
        }
        
        // Verificar se já existe alocação para este aluno, turma e ano letivo
        $existe = DB::table('turma_aluno')
            ->where('aluno_id', $aluno->id)
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->exists();
        
        if ($existe) {
            return redirect()->back()
                ->with('error', 'Aluno já está alocado nesta turma no ano letivo selecionado.');
        }
        
        // Verificar capacidade máxima da turma (35 alunos)
        $capacidadeMaxima = $turma->capacidade_maxima ?? 35;
        $alunosNaTurma = DB::table('turma_aluno')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->count();
        
        if ($alunosNaTurma >= $capacidadeMaxima) {
            return redirect()->back()
                ->with('error', "A turma {$turma->codigo} - {$turma->nome} já atingiu a capacidade máxima de {$capacidadeMaxima} alunos.");
        }
        
        // Remover alocação anterior no mesmo ano letivo (se houver)
        DB::table('turma_aluno')
            ->where('aluno_id', $aluno->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->where('turma_id', '!=', $turma->id)
            ->delete();
        
        // Alocar aluno à nova turma
        $aluno->turmas()->attach($turma->id, [
            'ano_letivo_id' => $anoLetivo->id
        ]);
        
        // Definir valor da mensalidade baseado na classe da turma (se não tiver valor personalizado)
        if (!$aluno->valor_mensalidade_personalizado || $aluno->valor_mensalidade_personalizado <= 0) {
            $financeiroService = app(\App\Services\FinanceiroService::class);
            $valorMensalidade = $financeiroService->getValorMensalidadePorTurma($turma);
            $aluno->update(['valor_mensalidade_personalizado' => $valorMensalidade]);
        }
        
        return redirect()->route('admin.alunos.index')
            ->with('success', "Aluno alocado com sucesso à turma {$turma->codigo} - {$turma->nome}!");
    }

    public function viewAluno(Aluno $aluno)
    {
        $aluno->load('user');
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->orderBy('codigo')
            ->orderBy('nome')
            ->get();
        $classes = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('codigo')
            ->sort()
            ->values();
        
        // Buscar turma atual do aluno no ano letivo ativo
        $turmaAtual = null;
        if ($anoLetivo) {
            $turmaAtual = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivo->id)
                ->first();
        }
        
        $viewOnly = true; // Modo somente leitura
        return view('admin.alunos.edit', compact('aluno', 'turmas', 'classes', 'anoLetivo', 'anosLetivos', 'turmaAtual', 'viewOnly'));
    }

    public function imprimirMatricula(Aluno $aluno)
    {
        // Usar a mesma lógica da ficha de inscrição
        return $this->fichaInscricao($aluno);
    }

    public function imprimirRecibo(Aluno $aluno)
    {
        $aluno->load('user');
        $configuracao = [
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
            'logotipo' => \App\Models\Configuracao::get('logotipo_escola', null),
        ];
        return view('admin.alunos.imprimir-recibo', compact('aluno', 'configuracao'));
    }

    public function imprimirCartao(Aluno $aluno)
    {
        $aluno->load('user');
        $configuracao = $this->configuracaoCartao();
        return view('admin.alunos.imprimir-cartao', compact('aluno', 'configuracao'));
    }

    public function downloadCartao(Aluno $aluno)
    {
        set_time_limit(120);
        $this->verificarPermissao('gerir_alunos');

        try {
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();
            $aluno->load([
                'user:id,name',
                'turmas' => function ($q) use ($anoLetivo) {
                    $q->select('turmas.id', 'turmas.nome', 'turmas.codigo')
                      ->wherePivot('ano_letivo_id', $anoLetivo?->id);
                }
            ]);
            $configuracao = $this->configuracaoCartao();

            $logotipoBase64 = null;
            if (!empty($configuracao['logotipo']) && Storage::disk('public')->exists($configuracao['logotipo'])) {
                $logotipoPath = Storage::disk('public')->path($configuracao['logotipo']);
                $logotipoData = file_get_contents($logotipoPath);
                $logotipoBase64 = 'data:' . mime_content_type($logotipoPath) . ';base64,' . base64_encode($logotipoData);
            }

            $fotoBase64 = null;
            if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
                $fotoPath = Storage::disk('public')->path($aluno->foto);
                $fotoData = file_get_contents($fotoPath);
                $fotoBase64 = 'data:' . mime_content_type($fotoPath) . ';base64,' . base64_encode($fotoData);
            }
            $aluno->foto_base64 = $fotoBase64;
            $qrCodeBase64 = $this->gerarQrCodeBase64($aluno->codigo_estudante);

            $pdf = Pdf::loadView('admin.alunos.imprimir-cartao', compact('aluno', 'configuracao', 'anoLetivo', 'logotipoBase64', 'qrCodeBase64') + ['download' => true]);
            $pdf->setPaper([0, 0, 242.65, 153], 'portrait');
            $pdf->setOption('enable-local-file-access', true);
            $pdf->setOption('isHtml5ParserEnabled', true);
            $pdf->setOption('isRemoteEnabled', true);
            $pdf->setOption('chroot', public_path());
            $pdf->setOption('dpi', 150);
            $pdf->setOption('defaultFont', 'Arial');

            $nome = $aluno->user->name ?? $aluno->codigo_estudante ?? 'cartao';
            $nomeArquivo = 'cartao-' . preg_replace('/[^a-z0-9\-]/', '-', strtolower($nome)) . '.pdf';
            return $pdf->download($nomeArquivo);
        } catch (\Throwable $e) {
            \Log::error('downloadCartao: ' . $e->getMessage(), ['aluno_id' => $aluno->id, 'trace' => $e->getTraceAsString()]);
            return redirect()->route('admin.cartoes-estudantes.index')
                ->with('error', 'Erro ao gerar PDF do cartão. Tente novamente ou imprima o cartão individual.');
        }
    }

    public function cartoesEstudantesIndex(Request $request)
    {
        $this->verificarPermissao('gerir_alunos');
        
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtivo?->id);
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : $anoLetivoAtivo;
        
        $query = $this->cartoesEstudantesQuery($request, $anoLetivoSelecionado);
        $alunos = $query->with(['user:id,name', 'turmas' => function ($q) use ($anoLetivoSelecionado) {
            $q->select('turmas.id', 'turmas.nome', 'turmas.codigo')
              ->wherePivot('ano_letivo_id', $anoLetivoSelecionado?->id);
        }])->orderBy('codigo_estudante')->paginate(50)->withQueryString();
        
        $classes = Turma::where('ano_letivo_id', $anoLetivoSelecionado?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('codigo')
            ->sort()
            ->values();
        $turmas = Turma::where('ano_letivo_id', $anoLetivoSelecionado?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('nome')
            ->sort()
            ->values();
        $anosLetivos = AnoLetivo::getAnosLetivosDisponiveis();
        
        return view('admin.alunos.cartoes-estudantes', compact(
            'alunos', 'classes', 'turmas', 'anosLetivos', 'anoLetivoId', 'anoLetivoSelecionado'
        ));
    }

    public function cartoesEstudantesFiltros(Request $request)
    {
        $this->verificarPermissao('gerir_alunos');
        $anoLetivoId = $request->get('ano_letivo_id');
        $ano = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : null;
        
        $classes = $ano
            ? Turma::where('ano_letivo_id', $ano->id)->where('ativa', true)->distinct()->pluck('codigo')->sort()->values()
            : collect();
        $turmas = $ano
            ? Turma::where('ano_letivo_id', $ano->id)->where('ativa', true)->distinct()->pluck('nome')->sort()->values()
            : collect();
        
        return response()->json(['classes' => $classes, 'turmas' => $turmas]);
    }

    public function cartoesEstudantesImprimir(Request $request)
    {
        $this->verificarPermissao('gerir_alunos');
        $alunos = $this->cartoesEstudantesAlunos($request);
        if ($alunos->isEmpty()) {
            return redirect()->route('admin.cartoes-estudantes.index')->with('error', 'Nenhum estudante encontrado para os filtros ou seleção indicados.');
        }
        $configuracao = $this->configuracaoCartao();
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        return view('admin.alunos.cartoes-multi', compact('alunos', 'configuracao', 'anoLetivo') + ['download' => false]);
    }

    public function cartoesEstudantesDownload(Request $request)
    {
        set_time_limit(600);
        $this->verificarPermissao('gerir_alunos');

        try {
            $alunos = $this->cartoesEstudantesAlunos($request);
            if ($alunos->isEmpty()) {
                return redirect()->route('admin.cartoes-estudantes.index')->with('error', 'Nenhum estudante encontrado para os filtros ou seleção indicados.');
            }
            $configuracao = $this->configuracaoCartao();
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();

            $logotipoBase64 = null;
            if (!empty($configuracao['logotipo']) && Storage::disk('public')->exists($configuracao['logotipo'])) {
                $logotipoPath = Storage::disk('public')->path($configuracao['logotipo']);
                $logotipoData = file_get_contents($logotipoPath);
                $logotipoBase64 = 'data:' . mime_content_type($logotipoPath) . ';base64,' . base64_encode($logotipoData);
            }

            $alunosComFotos = $alunos->map(function ($aluno) {
                $fotoBase64 = null;
                if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
                    $fotoPath = Storage::disk('public')->path($aluno->foto);
                    $fotoData = file_get_contents($fotoPath);
                    $fotoBase64 = 'data:' . mime_content_type($fotoPath) . ';base64,' . base64_encode($fotoData);
                }
                $aluno->foto_base64 = $fotoBase64;
                $aluno->qr_code_base64 = $this->gerarQrCodeBase64($aluno->codigo_estudante);
                return $aluno;
            });

            $pdf = Pdf::loadView('admin.alunos.cartoes-multi', [
                'alunos' => $alunosComFotos,
                'configuracao' => $configuracao,
                'anoLetivo' => $anoLetivo,
                'logotipoBase64' => $logotipoBase64,
                'download' => true,
            ]);
            $pdf->setPaper([0, 0, 242.65, 153], 'portrait');
            $pdf->setOption('enable-local-file-access', true);
            $pdf->setOption('isHtml5ParserEnabled', true);
            $pdf->setOption('isRemoteEnabled', true);
            $pdf->setOption('chroot', public_path());
            $pdf->setOption('dpi', 150);
            $pdf->setOption('defaultFont', 'Arial');
            return $pdf->download('cartoes-estudantes.pdf');
        } catch (\Throwable $e) {
            \Log::error('cartoesEstudantesDownload: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return redirect()->route('admin.cartoes-estudantes.index')
                ->with('error', 'Erro ao gerar PDF dos cartões. Reduza o número de estudantes (filtre por turma/classe) e tente novamente.');
        }
    }

    private function cartoesEstudantesQuery(Request $request, $anoLetivoSelecionado)
    {
        $query = Aluno::query()->with('user');
        
        if ($request->filled('search')) {
            $s = $request->search;
            $query->where(function ($q) use ($s) {
                $q->where('codigo_estudante', 'like', "%{$s}%")
                  ->orWhereHas('user', fn ($uq) => $uq->where('name', 'like', "%{$s}%"));
            });
        }
        if ($request->filled('classe') && $anoLetivoSelecionado) {
            $query->whereExists(function ($q) use ($request, $anoLetivoSelecionado) {
                $q->select(DB::raw(1))
                  ->from('turma_aluno')
                  ->join('turmas', 'turma_aluno.turma_id', '=', 'turmas.id')
                  ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                  ->where('turmas.codigo', $request->classe)
                  ->where('turma_aluno.ano_letivo_id', $anoLetivoSelecionado->id);
            });
        }
        if ($request->filled('turma') && $anoLetivoSelecionado) {
            $query->whereExists(function ($q) use ($request, $anoLetivoSelecionado) {
                $q->select(DB::raw(1))
                  ->from('turma_aluno')
                  ->join('turmas', 'turma_aluno.turma_id', '=', 'turmas.id')
                  ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                  ->where('turmas.nome', $request->turma)
                  ->where('turma_aluno.ano_letivo_id', $anoLetivoSelecionado->id);
            });
        }
        return $query;
    }

    private function cartoesEstudantesAlunos(Request $request)
    {
        $ids = $request->get('ids', []);
        $ids = is_array($ids) ? $ids : array_filter(explode(',', $ids));
        
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtivo?->id);
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : $anoLetivoAtivo;
        
        if (!empty($ids)) {
            $query = Aluno::whereIn('id', $ids);
        } else {
            $query = $this->cartoesEstudantesQuery($request, $anoLetivoSelecionado);
        }
        
        return $query->with([
            'user:id,name',
            'turmas' => function ($q) use ($anoLetivoSelecionado) {
                $q->select('turmas.id', 'turmas.nome', 'turmas.codigo')
                  ->wherePivot('ano_letivo_id', $anoLetivoSelecionado?->id);
            }
        ])->orderBy('codigo_estudante')->get();
    }

    public function imprimirCracha(Professor $professor)
    {
        $this->verificarPermissao('gerir_professores');
        $professor->load('user');
        $configuracao = $this->configuracaoCartao();
        $logotipoBase64 = null;
        if (!empty($configuracao['logotipo']) && Storage::disk('public')->exists($configuracao['logotipo'])) {
            $path = Storage::disk('public')->path($configuracao['logotipo']);
            $logotipoBase64 = 'data:' . mime_content_type($path) . ';base64,' . base64_encode(file_get_contents($path));
        }
        return view('admin.professores.imprimir-cracha', compact('professor', 'configuracao', 'logotipoBase64'));
    }

    public function cartoesProfessoresIndex(Request $request)
    {
        $this->verificarPermissao('gerir_professores');
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtivo?->id);
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : $anoLetivoAtivo;

        $query = $this->cartoesProfessoresQuery($request, $anoLetivoSelecionado);
        $professores = $query->with([
            'user:id,name',
            'turmas' => function ($q) use ($anoLetivoSelecionado) {
                $q->select('turmas.id', 'turmas.nome', 'turmas.codigo')
                  ->wherePivot('ano_letivo_id', $anoLetivoSelecionado?->id);
            }
        ])->orderBy('numero_funcionario')->paginate(50)->withQueryString();

        $classes = Turma::where('ano_letivo_id', $anoLetivoSelecionado?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('codigo')
            ->sort()
            ->values();
        $turmas = Turma::where('ano_letivo_id', $anoLetivoSelecionado?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('nome')
            ->sort()
            ->values();
        $anosLetivos = AnoLetivo::getAnosLetivosDisponiveis();

        return view('admin.professores.cartoes-professores', compact(
            'professores', 'classes', 'turmas', 'anosLetivos', 'anoLetivoId', 'anoLetivoSelecionado'
        ));
    }

    public function cartoesProfessoresFiltros(Request $request)
    {
        $this->verificarPermissao('gerir_professores');
        $anoLetivoId = $request->get('ano_letivo_id');
        $ano = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : null;
        $classes = $ano
            ? Turma::where('ano_letivo_id', $ano->id)->where('ativa', true)->distinct()->pluck('codigo')->sort()->values()
            : collect();
        $turmas = $ano
            ? Turma::where('ano_letivo_id', $ano->id)->where('ativa', true)->distinct()->pluck('nome')->sort()->values()
            : collect();
        return response()->json(['classes' => $classes, 'turmas' => $turmas]);
    }

    public function cartoesProfessoresImprimir(Request $request)
    {
        $this->verificarPermissao('gerir_professores');
        $professores = $this->cartoesProfessoresProfessores($request);
        if ($professores->isEmpty()) {
            return redirect()->route('admin.cartoes-professores.index')
                ->with('error', 'Nenhum professor encontrado para os filtros ou seleção indicados.');
        }
        $configuracao = $this->configuracaoCartao();
        $logotipoBase64 = null;
        if (!empty($configuracao['logotipo']) && Storage::disk('public')->exists($configuracao['logotipo'])) {
            $path = Storage::disk('public')->path($configuracao['logotipo']);
            $logotipoBase64 = 'data:' . mime_content_type($path) . ';base64,' . base64_encode(file_get_contents($path));
        }
        $professores = $professores->map(function ($p) {
            $p->qr_code_base64 = $this->gerarQrCodeBase64('P:' . $p->id);
            return $p;
        });
        return view('admin.professores.cartoes-professores-multi', compact('professores', 'configuracao', 'logotipoBase64') + ['download' => false]);
    }

    private function cartoesProfessoresQuery(Request $request, $anoLetivoSelecionado)
    {
        $query = Professor::query()->with('user')->whereHas('user', fn ($q) => $q->whereNull('deleted_at'));

        if ($request->filled('search')) {
            $s = $request->search;
            $query->where(function ($q) use ($s) {
                $q->where('numero_funcionario', 'like', "%{$s}%")
                  ->orWhereHas('user', fn ($uq) => $uq->where('name', 'like', "%{$s}%"));
            });
        }
        if ($request->filled('classe') && $anoLetivoSelecionado) {
            $query->whereExists(function ($q) use ($request, $anoLetivoSelecionado) {
                $q->select(DB::raw(1))
                  ->from('turma_disciplina_professor')
                  ->join('turmas', 'turma_disciplina_professor.turma_id', '=', 'turmas.id')
                  ->whereColumn('turma_disciplina_professor.professor_id', 'professores.id')
                  ->where('turmas.codigo', $request->classe)
                  ->where('turma_disciplina_professor.ano_letivo_id', $anoLetivoSelecionado->id);
            });
        }
        if ($request->filled('turma') && $anoLetivoSelecionado) {
            $query->whereExists(function ($q) use ($request, $anoLetivoSelecionado) {
                $q->select(DB::raw(1))
                  ->from('turma_disciplina_professor')
                  ->join('turmas', 'turma_disciplina_professor.turma_id', '=', 'turmas.id')
                  ->whereColumn('turma_disciplina_professor.professor_id', 'professores.id')
                  ->where('turmas.nome', $request->turma)
                  ->where('turma_disciplina_professor.ano_letivo_id', $anoLetivoSelecionado->id);
            });
        }
        return $query;
    }

    private function cartoesProfessoresProfessores(Request $request)
    {
        $ids = $request->get('ids', []);
        $ids = is_array($ids) ? $ids : array_filter(explode(',', $ids));
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtivo?->id);
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : $anoLetivoAtivo;

        if (!empty($ids)) {
            $query = Professor::whereIn('id', $ids);
        } else {
            $query = $this->cartoesProfessoresQuery($request, $anoLetivoSelecionado);
        }
        return $query->with('user:id,name')->orderBy('numero_funcionario')->get()->unique('id')->values();
    }

    public function scannerPresencaProfessor()
    {
        $this->verificarPermissao('gerir_professores');
        return view('admin.presenca-professor.scanner');
    }

    public function marcarPresencaProfessor(Request $request)
    {
        $this->verificarPermissao('gerir_professores');
        $codigo = $request->input('codigo', '');
        if (!preg_match('/^P:(\d+)$/', trim($codigo), $m)) {
            return response()->json(['success' => false, 'message' => 'Código QR inválido.'], 422);
        }
        $professor = Professor::with('user')->find($m[1]);
        if (!$professor) {
            return response()->json(['success' => false, 'message' => 'Professor não encontrado.'], 404);
        }
        $agora = now();
        $limite = $agora->copy()->subMinutes(10);
        $jaMarcou = PresencaProfessor::where('professor_id', $professor->id)
            ->where('marcado_em', '>=', $limite)
            ->exists();
        if ($jaMarcou) {
            return response()->json([
                'success' => false,
                'message' => 'Já marcou a presença. Aguarde 10 minutos para marcar novamente.',
            ], 422);
        }
        PresencaProfessor::create([
            'professor_id' => $professor->id,
            'marcado_em' => $agora,
            'origem' => 'scanner',
        ]);
        return response()->json(['success' => true, 'nome' => $professor->user?->name ?? $professor->numero_funcionario]);
    }

    public function indexPresencaProfessor(Request $request)
    {
        $this->verificarPermissao('gerir_professores');
        $dataFim = $request->filled('data_fim') ? \Carbon\Carbon::parse($request->data_fim) : now();
        $dataInicio = $request->filled('data_inicio') ? \Carbon\Carbon::parse($request->data_inicio) : $dataFim->copy()->subDays(30);
        if ($dataInicio->gt($dataFim)) {
            $dataInicio = $dataFim->copy()->subDays(30);
        }
        $professorId = $request->filled('professor_id') ? (int) $request->professor_id : null;

        $query = PresencaProfessor::query()
            ->with(['professor.user:id,name'])
            ->whereBetween('marcado_em', [$dataInicio->startOfDay(), $dataFim->endOfDay()]);
        if ($professorId) {
            $query->where('professor_id', $professorId);
        }
        $registos = $query->orderBy('marcado_em')->get();

        $agrupado = [];
        foreach ($registos as $r) {
            $dia = $r->marcado_em->format('Y-m-d');
            $key = $dia . '_' . $r->professor_id;
            if (!isset($agrupado[$key])) {
                $agrupado[$key] = [
                    'dia' => $r->marcado_em,
                    'professor_id' => $r->professor_id,
                    'professor' => $r->professor,
                    'entrada' => $r->marcado_em,
                    'saida' => $r->marcado_em,
                ];
            } else {
                $agrupado[$key]['saida'] = $r->marcado_em;
            }
        }
        $lista = collect($agrupado)->sortByDesc('dia')->values()->all();

        $professores = Professor::query()->with('user:id,name')->orderBy('numero_funcionario')->get(['id', 'numero_funcionario']);

        return view('admin.presenca-professor.index', [
            'lista' => $lista,
            'professores' => $professores,
            'data_inicio' => $dataInicio->format('Y-m-d'),
            'data_fim' => $dataFim->format('Y-m-d'),
            'professor_id' => $professorId,
        ]);
    }

    private function gerarQrCodeBase64(string $data): ?string
    {
        try {
            $options = new QROptions([
                'outputType' => QROutputInterface::GDIMAGE_PNG,
                'scale' => 10,
            ]);
            return (new QRCode($options))->render($data);
        } catch (\Throwable $e) {
            return null;
        }
    }

    private function configuracaoCartao()
    {
        $logotipo = Configuracao::get('logotipo_escola', null);
        $cores = $this->extrairCoresDoLogotipo($logotipo);
        if (!$cores) {
            $tema = \App\Models\Tema::getTemaAtivo();
            $corPrincipal = $tema->cor_principal ?? '#1a237e';
            $corSecundaria = $tema->cor_secundaria ?? $this->lightenColor($corPrincipal, 40);
            $cores = ['principal' => $corPrincipal, 'secundaria' => $corSecundaria, 'destaque' => '#ffb300'];
        }
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $anoRaw = $anoLetivo ? ($anoLetivo->ano ?? (string) date('Y')) : (string) date('Y');
        $anoLectivo = (is_string($anoRaw) && strpos($anoRaw, '/') !== false)
            ? (explode('/', $anoRaw)[1] ?? explode('/', $anoRaw)[0] ?? date('Y'))
            : $anoRaw;
        return [
            'escola' => Configuracao::get('escola', 'COLÉGIO PREMIUM'),
            'logotipo' => $logotipo,
            'slogan' => Configuracao::get('slogan_escola', 'PER SCIENTIAM AD EXCELLENTIAM'),
            'cor_principal' => $cores['principal'],
            'cor_secundaria' => $cores['secundaria'],
            'cor_destaque' => $cores['destaque'],
            'ano_lectivo' => $anoLectivo,
            'validade' => '12/' . $anoLectivo,
            'contacto' => Configuracao::get('contacto_escola', '(258) 84 123 4567'),
            'website' => Configuracao::get('website_escola', 'www.colegiopremium.cdu.mz'),
        ];
    }
    
    private function extrairCoresDoLogotipo($logotipoPath)
    {
        if (!$logotipoPath) {
            return null;
        }
        
        $fullPath = storage_path('app/public/' . $logotipoPath);
        
        if (!file_exists($fullPath)) {
            return null;
        }
        
        // Verificar tipo de imagem
        $imageInfo = getimagesize($fullPath);
        if (!$imageInfo) {
            return null;
        }
        
        $imageType = $imageInfo[2];
        
        // Carregar imagem baseada no tipo
        switch ($imageType) {
            case IMAGETYPE_JPEG:
                $image = imagecreatefromjpeg($fullPath);
                break;
            case IMAGETYPE_PNG:
                $image = imagecreatefrompng($fullPath);
                break;
            case IMAGETYPE_GIF:
                $image = imagecreatefromgif($fullPath);
                break;
            default:
                return null;
        }
        
        if (!$image) {
            return null;
        }
        
        // Redimensionar para análise mais rápida (máximo 100x100)
        $width = imagesx($image);
        $height = imagesy($image);
        $maxSize = 100;
        
        if ($width > $maxSize || $height > $maxSize) {
            $ratio = min($maxSize / $width, $maxSize / $height);
            $newWidth = (int)($width * $ratio);
            $newHeight = (int)($height * $ratio);
            $resized = imagecreatetruecolor($newWidth, $newHeight);
            imagecopyresampled($resized, $image, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
            imagedestroy($image);
            $image = $resized;
            $width = $newWidth;
            $height = $newHeight;
        }
        
        // Extrair cores dominantes
        $colors = [];
        $colorCount = [];
        
        for ($x = 0; $x < $width; $x += 2) {
            for ($y = 0; $y < $height; $y += 2) {
                $rgb = imagecolorat($image, $x, $y);
                $r = ($rgb >> 16) & 0xFF;
                $g = ($rgb >> 8) & 0xFF;
                $b = $rgb & 0xFF;
                
                // Ignorar cores muito claras (fundo) e muito escuras
                $brightness = ($r + $g + $b) / 3;
                if ($brightness < 30 || $brightness > 240) {
                    continue;
                }
                
                // Agrupar cores similares
                $r = round($r / 10) * 10;
                $g = round($g / 10) * 10;
                $b = round($b / 10) * 10;
                
                $colorKey = sprintf('#%02x%02x%02x', $r, $g, $b);
                
                if (!isset($colorCount[$colorKey])) {
                    $colorCount[$colorKey] = 0;
                }
                $colorCount[$colorKey]++;
            }
        }
        
        imagedestroy($image);
        
        if (empty($colorCount)) {
            return null;
        }
        
        // Ordenar por frequência
        arsort($colorCount);
        $dominantColors = array_slice(array_keys($colorCount), 0, 5, true);
        
        // Escolher cores: mais escura como principal, mais clara como secundária
        $principal = $dominantColors[0];
        $secundaria = count($dominantColors) > 1 ? $dominantColors[1] : $this->lightenColor($principal, 50);
        
        // Se a secundária for muito escura, clarear
        $secundariaBrightness = $this->getBrightness($secundaria);
        if ($secundariaBrightness < 100) {
            $secundaria = $this->lightenColor($secundaria, 60);
        }
        
        // Cor de destaque (amarelo/dourado ou derivada)
        $destaque = count($dominantColors) > 2 ? $dominantColors[2] : '#ffb300';
        $destaqueBrightness = $this->getBrightness($destaque);
        if ($destaqueBrightness < 150) {
            $destaque = '#ffb300'; // Usar dourado padrão se a cor extraída for muito escura
        }
        
        return [
            'principal' => $principal,
            'secundaria' => $secundaria,
            'destaque' => $destaque,
        ];
    }
    
    private function getBrightness($hex)
    {
        $hex = str_replace('#', '', $hex);
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        return ($r + $g + $b) / 3;
    }
    
    private function lightenColor($hex, $percent)
    {
        $hex = str_replace('#', '', $hex);
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        
        $r = min(255, $r + ($percent * 255 / 100));
        $g = min(255, $g + ($percent * 255 / 100));
        $b = min(255, $b + ($percent * 255 / 100));
        
        return '#' . str_pad(dechex($r), 2, '0', STR_PAD_LEFT) . 
                   str_pad(dechex($g), 2, '0', STR_PAD_LEFT) . 
                   str_pad(dechex($b), 2, '0', STR_PAD_LEFT);
    }

    private function prepararDadosFichaInscricao(Aluno $aluno)
    {
        $aluno->load('user');
        
        // Buscar turma atual do aluno
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $turma = null;
        if ($anoLetivo) {
            $turma = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivo->id)
                ->first();
        }
        
        // Se não tiver turma no ano letivo atual, buscar a primeira turma
        if (!$turma) {
            $turma = $aluno->turmas()->first();
            if ($turma && $turma->pivot) {
                $anoLetivo = AnoLetivo::find($turma->pivot->ano_letivo_id);
            }
        }
        
        // Processar endereço do aluno
        $enderecoParts = $this->processarEndereco($aluno->endereco ?? '');
        
        // Processar endereço do encarregado
        $enderecoEncarregadoParts = $this->processarEndereco($aluno->endereco_encarregado ?? '');
        
        // Preparar logotipo em base64
        $logotipo = Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $logotipoBase64 = 'data:' . mime_content_type($logotipoPath) . ';base64,' . base64_encode($logotipoData);
        }
        
        // Preparar foto do aluno em base64
        $fotoBase64 = null;
        if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
            $fotoPath = Storage::disk('public')->path($aluno->foto);
            $fotoData = file_get_contents($fotoPath);
            $fotoBase64 = 'data:' . mime_content_type($fotoPath) . ';base64,' . base64_encode($fotoData);
        }
        
        return compact(
            'aluno', 
            'turma', 
            'anoLetivo', 
            'enderecoParts', 
            'enderecoEncarregadoParts',
            'logotipoBase64',
            'fotoBase64'
        );
    }
    
    public function visualizarFichaInscricao(Aluno $aluno)
    {
        $dados = $this->prepararDadosFichaInscricao($aluno);
        return view('admin.alunos.ficha-inscricao-view', $dados);
    }
    
    public function fichaInscricao(Aluno $aluno)
    {
        $dados = $this->prepararDadosFichaInscricao($aluno);
        
        $pdf = Pdf::loadView('admin.alunos.ficha-inscricao', $dados);
        
        // Gerar nome do arquivo baseado no nome do aluno
        $nomeAluno = $aluno->user->name ?? 'Aluno';
        // Remover acentos e caracteres especiais, substituir espaços por underscore
        // Usar transliteração para remover acentos
        $nomeArquivo = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $nomeAluno);
        if ($nomeArquivo === false) {
            // Se iconv falhar, usar método alternativo
            $nomeArquivo = mb_strtolower($nomeAluno, 'UTF-8');
            $nomeArquivo = preg_replace('/[^a-z0-9\s]/', '', $nomeArquivo);
        } else {
            $nomeArquivo = mb_strtolower($nomeArquivo, 'UTF-8');
        }
        // Remover caracteres especiais e espaços múltiplos
        $nomeArquivo = preg_replace('/[^a-z0-9\s]/', '', $nomeArquivo);
        $nomeArquivo = preg_replace('/\s+/', '_', trim($nomeArquivo));
        if (empty($nomeArquivo)) {
            $nomeArquivo = 'aluno';
        }
        // Primeira letra maiúscula
        $nomeArquivo = ucfirst($nomeArquivo) . '_Ficha-inscricao.pdf';
        
        return $pdf->download($nomeArquivo);
    }
    
    private function processarEndereco($endereco)
    {
        $parts = [
            'bairro' => '',
            'rua' => '',
            'numero' => '',
            'quarteirao' => '',
            'flat' => ''
        ];
        
        if (empty($endereco)) {
            return $parts;
        }
        
        // Tentar extrair informações do endereço
        // Formato esperado pode variar, então fazemos uma análise básica
        $enderecoLower = mb_strtolower($endereco);
        
        // Procurar por "bairro"
        if (preg_match('/bairro[:\s]+([^,]+)/i', $endereco, $matches)) {
            $parts['bairro'] = trim($matches[1]);
        }
        
        // Procurar por "rua" ou "avenida"
        if (preg_match('/(rua|avenida|av\.?)[:\s]+([^,]+)/i', $endereco, $matches)) {
            $parts['rua'] = trim($matches[2]);
        }
        
        // Procurar por número
        if (preg_match('/n[°º]?\s*(\d+)/i', $endereco, $matches)) {
            $parts['numero'] = trim($matches[1]);
        }
        
        // Procurar por quarteirão
        if (preg_match('/quarteir[ãa]o[:\s]+([^,]+)/i', $endereco, $matches)) {
            $parts['quarteirao'] = trim($matches[1]);
        }
        
        // Procurar por flat
        if (preg_match('/flat[:\s]+([^,]+)/i', $endereco, $matches)) {
            $parts['flat'] = trim($matches[1]);
        }
        
        return $parts;
    }

    public function folhetoAluno(Aluno $aluno, Request $request)
    {
        $aluno->load('user');
        $anoLetivo = \App\Models\AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivo) {
            return redirect()->route('admin.alunos.show', $aluno)
                ->with('error', 'Nenhum ano letivo ativo encontrado.');
        }

        // Buscar turma atual do aluno
        $turma = $aluno->turmas()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->first();

        if (!$turma) {
            return redirect()->route('admin.alunos.show', $aluno)
                ->with('error', 'Aluno não está matriculado em nenhuma turma no ano letivo ativo.');
        }

        // Buscar trimestre selecionado (default: 1º trimestre)
        $trimestreNumero = $request->get('trimestre', 1);
        $trimestre = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->where('numero', $trimestreNumero)
            ->first();

        if (!$trimestre) {
            return redirect()->route('admin.alunos.show', $aluno)
                ->with('error', 'Trimestre não encontrado.');
        }

        // Buscar disciplinas da turma
        $disciplinasIds = \DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        $disciplinas = \App\Models\Disciplina::whereIn('id', $disciplinasIds)
            ->orderBy('nome')
            ->get();

        // Buscar avaliações do trimestre selecionado
        $mediasPorDisciplina = [];
        $totalMedias = 0;
        $disciplinasComNota = 0;

        foreach ($disciplinas as $disciplina) {
            $avaliacao = \App\Models\Avaliacao::where('aluno_id', $aluno->id)
                ->where('disciplina_id', $disciplina->id)
                ->where('turma_id', $turma->id)
                ->where('trimestre_id', $trimestre->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->first();

            $media = null;
            if ($avaliacao && $avaliacao->mt !== null) {
                $media = round($avaliacao->mt, 0);
                $totalMedias += $media;
                $disciplinasComNota++;
            }

            $mediasPorDisciplina[$disciplina->id] = $media;
        }

        // Calcular média trimestral geral
        $mediaTrimestralGeral = $disciplinasComNota > 0 ? round($totalMedias / $disciplinasComNota, 0) : null;

        // Classificar média trimestral
        $classificacaoMedia = $this->classificarMedia($mediaTrimestralGeral);

        $configuracao = [
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        // Preparar logotipo em base64
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && \Illuminate\Support\Facades\Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = \Illuminate\Support\Facades\Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $mime = mime_content_type($logotipoPath);
            $logotipoBase64 = "data:{$mime};base64," . base64_encode($logotipoData);
        }

        return view('admin.alunos.folheto-aluno', compact(
            'aluno', 'turma', 'anoLetivo', 'disciplinas', 'trimestre',
            'mediasPorDisciplina', 'mediaTrimestralGeral', 'classificacaoMedia', 'configuracao', 'logotipoBase64'
        ));
    }

    public function folhetosTurma(Request $request)
    {
        $anoLetivo = \App\Models\AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivo) {
            return redirect()->route('admin.alunos.index')
                ->with('error', 'Nenhum ano letivo ativo encontrado.');
        }

        $turmaId = $request->get('turma_id');
        $trimestreNumero = $request->get('trimestre', 1);

        if (!$turmaId) {
            $turmas = \App\Models\Turma::where('ano_letivo_id', $anoLetivo->id)
                ->where('ativa', true)
                ->orderBy('nome')
                ->get();
            
            $trimestres = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
                ->orderBy('numero', 'asc')
                ->get();
            
            return view('admin.alunos.folhetos-turma-select', compact('turmas', 'trimestres', 'anoLetivo'));
        }

        $turma = \App\Models\Turma::findOrFail($turmaId);
        $trimestre = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->where('numero', $trimestreNumero)
            ->firstOrFail();

        // Buscar alunos da turma ordenados por nome
        $alunos = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->with('user')
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*')
            ->get();

        // Buscar disciplinas da turma
        $disciplinasIds = \DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        $disciplinas = \App\Models\Disciplina::whereIn('id', $disciplinasIds)
            ->orderBy('nome')
            ->get();

        // Buscar avaliações e calcular médias para cada aluno
        $folhetos = [];
        foreach ($alunos as $index => $aluno) {
            $mediasPorDisciplina = [];
            $totalMedias = 0;
            $disciplinasComNota = 0;

            foreach ($disciplinas as $disciplina) {
                $avaliacao = \App\Models\Avaliacao::where('aluno_id', $aluno->id)
                    ->where('disciplina_id', $disciplina->id)
                    ->where('turma_id', $turma->id)
                    ->where('trimestre_id', $trimestre->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->first();

                $media = null;
                if ($avaliacao && $avaliacao->mt !== null) {
                    $media = round($avaliacao->mt, 0);
                    $totalMedias += $media;
                    $disciplinasComNota++;
                }

                $mediasPorDisciplina[$disciplina->id] = $media;
            }

            // Calcular média trimestral geral
            $mediaTrimestralGeral = $disciplinasComNota > 0 ? round($totalMedias / $disciplinasComNota, 0) : null;

            // Classificar média trimestral
            $classificacaoMedia = $this->classificarMedia($mediaTrimestralGeral);

            $folhetos[] = [
                'numero' => $index + 1,
                'aluno' => $aluno,
                'mediasPorDisciplina' => $mediasPorDisciplina,
                'mediaTrimestralGeral' => $mediaTrimestralGeral,
                'classificacaoMedia' => $classificacaoMedia,
            ];
        }

        $configuracao = [
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        // Preparar logotipo em base64
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && \Illuminate\Support\Facades\Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = \Illuminate\Support\Facades\Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $mime = mime_content_type($logotipoPath);
            $logotipoBase64 = "data:{$mime};base64," . base64_encode($logotipoData);
        }

        return view('admin.alunos.folhetos-turma', compact(
            'turma', 'trimestre', 'anoLetivo', 'disciplinas', 'folhetos', 'configuracao', 'logotipoBase64'
        ));
    }

    public function downloadFolhetoAluno(Request $request)
    {
        $alunoId = $request->get('aluno_id');
        $trimestreNumero = $request->get('trimestre', 1);
        
        if (!$alunoId) {
            abort(404);
        }

        $aluno = Aluno::findOrFail($alunoId);
        $aluno->load('user');
        $anoLetivo = \App\Models\AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivo) {
            abort(404, 'Nenhum ano letivo ativo encontrado.');
        }

        // Buscar turma atual do aluno
        $turma = $aluno->turmas()
            ->wherePivot('ano_letivo_id', $anoLetivo->id)
            ->first();

        if (!$turma) {
            abort(404, 'Aluno não está matriculado em nenhuma turma.');
        }

        // Buscar trimestre
        $trimestre = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->where('numero', $trimestreNumero)
            ->firstOrFail();

        // Buscar disciplinas da turma
        $disciplinasIds = \DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        $disciplinas = \App\Models\Disciplina::whereIn('id', $disciplinasIds)
            ->orderBy('nome')
            ->get();

        // Buscar avaliações do trimestre
        $mediasPorDisciplina = [];
        $totalMedias = 0;
        $disciplinasComNota = 0;

        foreach ($disciplinas as $disciplina) {
            $avaliacao = \App\Models\Avaliacao::where('aluno_id', $aluno->id)
                ->where('disciplina_id', $disciplina->id)
                ->where('turma_id', $turma->id)
                ->where('trimestre_id', $trimestre->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->first();

            $media = null;
            if ($avaliacao && $avaliacao->mt !== null) {
                $media = round($avaliacao->mt, 0);
                $totalMedias += $media;
                $disciplinasComNota++;
            }

            $mediasPorDisciplina[$disciplina->id] = $media;
        }

        // Calcular média trimestral geral
        $mediaTrimestralGeral = $disciplinasComNota > 0 ? round($totalMedias / $disciplinasComNota, 0) : null;

        // Classificar média trimestral
        $classificacaoMedia = $this->classificarMedia($mediaTrimestralGeral);

        $configuracao = [
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        // Preparar logotipo em base64
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && \Illuminate\Support\Facades\Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = \Illuminate\Support\Facades\Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $mime = mime_content_type($logotipoPath);
            $logotipoBase64 = "data:{$mime};base64," . base64_encode($logotipoData);
        }

        return view('admin.alunos.folheto-aluno-download', compact(
            'aluno', 'turma', 'trimestre', 'anoLetivo', 'disciplinas',
            'mediasPorDisciplina', 'mediaTrimestralGeral', 'classificacaoMedia', 'configuracao', 'logotipoBase64'
        ));
    }

    public function destroyAluno(Aluno $aluno)
    {
        DB::beginTransaction();
        try {
            $user = $aluno->user;

            // Remover foto se existir
            if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
                Storage::disk('public')->delete($aluno->foto);
            }

            // Eliminar todos os registos relacionados (hard delete)
            DB::table('turma_aluno')->where('aluno_id', $aluno->id)->delete();
            DB::table('trabalho_aluno')->where('aluno_id', $aluno->id)->delete();
            DB::table('avaliacoes')->where('aluno_id', $aluno->id)->delete();
            DB::table('financeiro')->where('aluno_id', $aluno->id)->delete();
            DB::table('invoices')->where('aluno_id', $aluno->id)->delete();
            DB::table('payments')->where('aluno_id', $aluno->id)->delete();

            // Hard delete do aluno (remove o registo da tabela alunos)
            $aluno->forceDelete();

            // Hard delete do user (remove o registo da tabela users)
            if ($user) {
                $user->forceDelete();
            }

            DB::commit();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Aluno eliminado com sucesso!'
                ]);
            }

            return redirect()->route('admin.alunos.index')->with('success', 'Aluno eliminado com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro ao eliminar aluno: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()->with('error', 'Erro ao eliminar aluno: ' . $e->getMessage());
        }
    }

    public function toggleStatusAluno(Aluno $aluno)
    {
        if ($aluno->user) {
            $aluno->user->is_active = !$aluno->user->is_active;
            $aluno->user->save();
            
            $status = $aluno->user->is_active ? 'ativado' : 'bloqueado';
            return redirect()->route('admin.alunos.index')->with('success', "Aluno {$status} com sucesso!");
        }

        return redirect()->route('admin.alunos.index')->with('error', 'Não foi possível alterar o status do aluno.');
    }

    // Gestão de Professores
    public function indexProfessores(Request $request)
    {
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        
        $query = Professor::with(['user', 'turmas' => function($q) use ($anoLetivoAtivo) {
            if ($anoLetivoAtivo) {
                $q->wherePivot('ano_letivo_id', $anoLetivoAtivo->id);
            }
        }])
        ->join('users', 'professores.user_id', '=', 'users.id')
        ->whereNull('users.deleted_at');
        
        // Pesquisa por código ou nome
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('numero_funcionario', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filtro por classe (codigo da turma)
        if ($request->filled('classe')) {
            if ($anoLetivoAtivo) {
                $query->whereExists(function($q) use ($request, $anoLetivoAtivo) {
                    $q->select(DB::raw(1))
                      ->from('turma_disciplina_professor')
                      ->join('turmas', 'turma_disciplina_professor.turma_id', '=', 'turmas.id')
                      ->whereColumn('turma_disciplina_professor.professor_id', 'professores.id')
                      ->where('turmas.codigo', $request->classe)
                      ->where('turma_disciplina_professor.ano_letivo_id', $anoLetivoAtivo->id);
                });
            } else {
                $query->whereHas('turmas', function($q) use ($request) {
                    $q->where('turmas.codigo', $request->classe);
                });
            }
        }
        
        // Filtro por turma (nome da turma)
        if ($request->filled('turma')) {
            if ($anoLetivoAtivo) {
                $query->whereExists(function($q) use ($request, $anoLetivoAtivo) {
                    $q->select(DB::raw(1))
                      ->from('turma_disciplina_professor')
                      ->join('turmas', 'turma_disciplina_professor.turma_id', '=', 'turmas.id')
                      ->whereColumn('turma_disciplina_professor.professor_id', 'professores.id')
                      ->where('turmas.nome', $request->turma)
                      ->where('turma_disciplina_professor.ano_letivo_id', $anoLetivoAtivo->id);
                });
            } else {
                $query->whereHas('turmas', function($q) use ($request) {
                    $q->where('turmas.nome', $request->turma);
                });
            }
        }
        
        // Ordenar: Secundário primeiro, depois Primário, depois Ambos
        $query->orderByRaw("
            CASE 
                WHEN professores.nivel_ensino = 'SECUNDARIO' THEN 1
                WHEN professores.nivel_ensino = 'PRIMARIO' THEN 2
                ELSE 3
            END
        ")
        ->orderBy('users.name', 'asc');
        
        // Otimização: Paginação e select específico
        $professores = $query->select('professores.id', 'professores.user_id', 'professores.numero_funcionario', 'professores.telefone', 'professores.nivel_ensino', 'professores.salario_fixo', 'professores.valor_hora', 'users.name', 'users.email', 'users.is_active')
            ->with('user:id,name,email,is_active')
            ->distinct()
            ->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());
        
        // Buscar classes e turmas disponíveis para os filtros
        $classes = Turma::where('ano_letivo_id', $anoLetivoAtivo?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('codigo')
            ->sort()
            ->values();
            
        $turmas = Turma::where('ano_letivo_id', $anoLetivoAtivo?->id)
            ->where('ativa', true)
            ->distinct()
            ->pluck('nome')
            ->sort()
            ->values();
        
        return view('admin.professores.index', compact('professores', 'classes', 'turmas'));
    }

    public function createProfessor()
    {
        $disciplinas = Disciplina::where('ativa', true)->orderBy('nome')->get();
        $turmas = Turma::where('ativa', true)->with('anoLetivo')->orderBy('nome')->get();
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        return view('admin.professores.create', compact('disciplinas', 'turmas', 'anoLetivo'));
    }

    public function storeProfessor(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:users,email',
            'telefone' => 'nullable|string|max:20',
            'documento_identificacao' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'genero' => 'nullable|in:M,F,Outro',
            'nivel_ensino' => 'required|in:PRIMARIO,SECUNDARIO,AMBOS',
            'disciplinas' => 'nullable|array',
            'disciplinas.*' => 'exists:disciplinas,id',
            'turma_id' => 'nullable|exists:turmas,id',
            'salario_fixo' => 'nullable|numeric|min:0',
            'valor_hora' => 'nullable|numeric|min:0',
            'numero_dependentes' => 'nullable|integer|min:0|max:10',
        ]);

        $dados = $request->all();
        
        // Processar upload do documento se fornecido
        if ($request->hasFile('documento_identificacao')) {
            $arquivo = $request->file('documento_identificacao');
            $nomeArquivo = time() . '_' . uniqid() . '.' . $arquivo->getClientOriginalExtension();
            $caminho = $arquivo->storeAs('documentos/professores', $nomeArquivo, 'public');
            $dados['documento_identificacao'] = $caminho;
        }

        $professor = $this->professorRepository->create($dados);
        
        // Atribuir disciplinas se fornecidas
        if ($request->filled('disciplinas') && $request->filled('turma_id')) {
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();
            if ($anoLetivo) {
                $nivelProfessor = $request->nivel_ensino ?? 'AMBOS';
                $turma = Turma::find($request->turma_id);
                if ($turma && $nivelProfessor !== 'AMBOS' && $turma->nivel_ensino !== $nivelProfessor) {
                    return redirect()->back()
                        ->withInput()
                        ->withErrors(['turma_id' => 'A turma selecionada não pertence ao nível de ensino do professor.']);
                }
                foreach ($request->disciplinas as $disciplinaId) {
                    $disciplina = Disciplina::find($disciplinaId);
                    if ($disciplina && $nivelProfessor !== 'AMBOS' && $disciplina->nivel_ensino !== $nivelProfessor) {
                        return redirect()->back()
                            ->withInput()
                            ->withErrors(['disciplinas' => 'Algumas disciplinas não pertencem ao nível de ensino do professor.']);
                    }

                    // Verificar se já existe OUTRO professor com esta disciplina nesta turma
                    $outroProfessorComDisciplina = DB::table('turma_disciplina_professor')
                        ->where('turma_id', $request->turma_id)
                        ->where('disciplina_id', $disciplinaId)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->where('professor_id', '!=', $professor->id)
                        ->exists();

                    if ($outroProfessorComDisciplina) {
                        // Buscar o nome do professor que já está alocado
                        $professorExistente = DB::table('turma_disciplina_professor')
                            ->where('turma_id', $request->turma_id)
                            ->where('disciplina_id', $disciplinaId)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->join('professores', 'turma_disciplina_professor.professor_id', '=', 'professores.id')
                            ->join('users', 'professores.user_id', '=', 'users.id')
                            ->select('users.name as nome')
                            ->first();
                        
                        $nomeProfessorExistente = $professorExistente ? $professorExistente->nome : 'outro professor';
                        return redirect()->back()
                            ->withInput()
                            ->withErrors(['disciplinas' => "A disciplina {$disciplina->nome} já está atribuída ao professor {$nomeProfessorExistente} nesta turma. Uma turma não pode ter dois professores diferentes para a mesma disciplina."]);
                    }

                    // Verificar se já existe esta mesma alocação
                    $existe = DB::table('turma_disciplina_professor')
                        ->where('turma_id', $request->turma_id)
                        ->where('professor_id', $professor->id)
                        ->where('disciplina_id', $disciplinaId)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->exists();
                    
                    if (!$existe) {
                        // Verificar se a disciplina já está atribuída à turma
                        $disciplinaJaAtribuida = DB::table('turma_disciplina')
                            ->where('turma_id', $request->turma_id)
                            ->where('disciplina_id', $disciplinaId)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->exists();
                        
                        // Se não estiver atribuída, criar automaticamente
                        if (!$disciplinaJaAtribuida && $disciplina) {
                            $cargaHoraria = $disciplina->carga_horaria_semanal ?? 0;
                            DB::table('turma_disciplina')->insert([
                                'turma_id' => $request->turma_id,
                                'disciplina_id' => $disciplinaId,
                                'ano_letivo_id' => $anoLetivo->id,
                                'carga_horaria_semanal' => $cargaHoraria,
                                'created_at' => now(),
                                'updated_at' => now(),
                            ]);
                        }
                        
                        // Criar alocação do professor
                        DB::table('turma_disciplina_professor')->insert([
                            'turma_id' => $request->turma_id,
                            'professor_id' => $professor->id,
                            'disciplina_id' => $disciplinaId,
                            'ano_letivo_id' => $anoLetivo->id,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                }
            }
        }

        return redirect()->route('admin.professores.index')->with('success', 'Professor criado com sucesso!');
    }

    public function editProfessor(Professor $professor)
    {
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $nivelProfessor = $professor->nivel_ensino ?? 'AMBOS';
        $disciplinas = Disciplina::where('ativa', true)
            ->when($nivelProfessor !== 'AMBOS', function ($query) use ($nivelProfessor) {
                $query->where('nivel_ensino', $nivelProfessor);
            })
            ->get();
        $turmas = Turma::where('ano_letivo_id', $anoLetivo?->id)
            ->where('ativa', true)
            ->when($nivelProfessor !== 'AMBOS', function ($query) use ($nivelProfessor) {
                $query->where('nivel_ensino', $nivelProfessor);
            })
            ->get();
        
        // Buscar alocações atuais do professor
        $alocacoesAtuais = DB::table('turma_disciplina_professor')
            ->where('professor_id', $professor->id)
            ->where('ano_letivo_id', $anoLetivo?->id)
            ->get();
        
        return view('admin.professores.edit', compact('professor', 'disciplinas', 'turmas', 'alocacoesAtuais', 'anoLetivo'));
    }

    public function updateProfessor(Request $request, Professor $professor)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:users,email,' . $professor->user_id,
            'telefone' => 'nullable|string|max:20',
            'documento_identificacao' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'genero' => 'nullable|in:M,F,Outro',
            'turma_id' => 'nullable|exists:turmas,id',
            'nivel_ensino' => 'required|in:PRIMARIO,SECUNDARIO,AMBOS',
            'disciplinas' => 'nullable|array',
            'disciplinas.*' => 'exists:disciplinas,id',
            'salario_fixo' => 'nullable|numeric|min:0',
            'valor_hora' => 'nullable|numeric|min:0',
            'numero_dependentes' => 'nullable|integer|min:0|max:10',
        ]);

        $dados = $request->all();
        
        // Processar upload do documento se fornecido
        if ($request->hasFile('documento_identificacao')) {
            // Deletar documento antigo se existir
            if ($professor->documento_identificacao && Storage::disk('public')->exists($professor->documento_identificacao)) {
                Storage::disk('public')->delete($professor->documento_identificacao);
            }
            
            $arquivo = $request->file('documento_identificacao');
            $nomeArquivo = time() . '_' . uniqid() . '.' . $arquivo->getClientOriginalExtension();
            $caminho = $arquivo->storeAs('documentos/professores', $nomeArquivo, 'public');
            $dados['documento_identificacao'] = $caminho;
        } else {
            // Manter documento existente se não houver novo upload
            unset($dados['documento_identificacao']);
        }

        $this->professorRepository->update($professor, $dados);

        // Atribuir novas disciplinas se fornecidas
        if ($request->filled('disciplinas') && $request->filled('turma_id')) {
            $anoLetivo = AnoLetivo::getAnoLetivoAtual();
            if ($anoLetivo) {
                $nivelProfessor = $request->nivel_ensino ?? $professor->nivel_ensino ?? 'AMBOS';
                $turma = Turma::find($request->turma_id);
                if ($turma && $nivelProfessor !== 'AMBOS' && $turma->nivel_ensino !== $nivelProfessor) {
                    return redirect()->back()
                        ->withInput()
                        ->withErrors(['turma_id' => 'A turma selecionada não pertence ao nível de ensino do professor.']);
                }
                foreach ($request->disciplinas as $disciplinaId) {
                    $disciplina = Disciplina::find($disciplinaId);
                    if ($disciplina && $nivelProfessor !== 'AMBOS' && $disciplina->nivel_ensino !== $nivelProfessor) {
                        return redirect()->back()
                            ->withInput()
                            ->withErrors(['disciplinas' => 'Algumas disciplinas não pertencem ao nível de ensino do professor.']);
                    }

                    // Verificar se já existe OUTRO professor com esta disciplina nesta turma
                    $outroProfessorComDisciplina = DB::table('turma_disciplina_professor')
                        ->where('turma_id', $request->turma_id)
                        ->where('disciplina_id', $disciplinaId)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->where('professor_id', '!=', $professor->id)
                        ->exists();

                    if ($outroProfessorComDisciplina) {
                        // Buscar o nome do professor que já está alocado
                        $professorExistente = DB::table('turma_disciplina_professor')
                            ->where('turma_id', $request->turma_id)
                            ->where('disciplina_id', $disciplinaId)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->join('professores', 'turma_disciplina_professor.professor_id', '=', 'professores.id')
                            ->join('users', 'professores.user_id', '=', 'users.id')
                            ->select('users.name as nome')
                            ->first();
                        
                        $nomeProfessorExistente = $professorExistente ? $professorExistente->nome : 'outro professor';
                        return redirect()->back()
                            ->withInput()
                            ->withErrors(['disciplinas' => "A disciplina {$disciplina->nome} já está atribuída ao professor {$nomeProfessorExistente} nesta turma. Uma turma não pode ter dois professores diferentes para a mesma disciplina."]);
                    }

                    // Verificar se já existe esta mesma alocação
                    $existe = DB::table('turma_disciplina_professor')
                        ->where('turma_id', $request->turma_id)
                        ->where('professor_id', $professor->id)
                        ->where('disciplina_id', $disciplinaId)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->exists();
                    
                    if (!$existe) {
                        // Verificar se a disciplina já está atribuída à turma
                        $disciplinaJaAtribuida = DB::table('turma_disciplina')
                            ->where('turma_id', $request->turma_id)
                            ->where('disciplina_id', $disciplinaId)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->exists();
                        
                        // Se não estiver atribuída, criar automaticamente
                        if (!$disciplinaJaAtribuida && $disciplina) {
                            $cargaHoraria = $disciplina->carga_horaria_semanal ?? 0;
                            DB::table('turma_disciplina')->insert([
                                'turma_id' => $request->turma_id,
                                'disciplina_id' => $disciplinaId,
                                'ano_letivo_id' => $anoLetivo->id,
                                'carga_horaria_semanal' => $cargaHoraria,
                                'created_at' => now(),
                                'updated_at' => now(),
                            ]);
                        }
                        
                        // Criar alocação do professor
                        DB::table('turma_disciplina_professor')->insert([
                            'turma_id' => $request->turma_id,
                            'professor_id' => $professor->id,
                            'disciplina_id' => $disciplinaId,
                            'ano_letivo_id' => $anoLetivo->id,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                }
            }
        }

        return redirect()->route('admin.professores.index')->with('success', 'Professor atualizado com sucesso!');
    }

    public function showProfessor(Professor $professor)
    {
        $professor->load('user');
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        // Buscar disciplinas que o professor leciona (distintas)
        $disciplinas = DB::table('turma_disciplina_professor')
            ->where('professor_id', $professor->id)
            ->when($anoLetivo, function($query) use ($anoLetivo) {
                $query->where('ano_letivo_id', $anoLetivo->id);
            })
            ->join('disciplinas', 'turma_disciplina_professor.disciplina_id', '=', 'disciplinas.id')
            ->whereNull('disciplinas.deleted_at')
            ->select('disciplinas.id', 'disciplinas.nome', 'disciplinas.modelo_avaliacao')
            ->distinct()
            ->orderBy('disciplinas.nome')
            ->get();
        
        // Buscar alocações (turma + disciplina) do professor
        $alocacoes = [];
        if ($anoLetivo) {
            $alocacoes = DB::table('turma_disciplina_professor')
                ->where('turma_disciplina_professor.professor_id', $professor->id)
                ->where('turma_disciplina_professor.ano_letivo_id', $anoLetivo->id)
                ->join('turmas', 'turma_disciplina_professor.turma_id', '=', 'turmas.id')
                ->join('disciplinas', 'turma_disciplina_professor.disciplina_id', '=', 'disciplinas.id')
                ->whereNull('turmas.deleted_at')
                ->whereNull('disciplinas.deleted_at')
                ->select(
                    'turmas.id as turma_id',
                    'turmas.nome as turma_nome',
                    'turmas.codigo as turma_codigo',
                    'disciplinas.id as disciplina_id',
                    'disciplinas.nome as disciplina_nome'
                )
                ->orderBy('turmas.codigo')
                ->orderBy('turmas.nome')
                ->orderBy('disciplinas.nome')
                ->get();
            
            // Adicionar número de alunos para cada alocação
            foreach ($alocacoes as $alocacao) {
                $turma = Turma::find($alocacao->turma_id);
                if ($turma) {
                    $alocacao->total_alunos = $turma->alunos()
                        ->wherePivot('ano_letivo_id', $anoLetivo->id)
                        ->count();
                } else {
                    $alocacao->total_alunos = 0;
                }
            }
        }
        
        return view('admin.professores.show', compact('professor', 'disciplinas', 'alocacoes', 'anoLetivo'));
    }

    public function destroyProfessor(Professor $professor)
    {
        $user = $professor->user;

        DB::beginTransaction();
        try {
            // Remover ficheiro do documento de identificação se existir
            if ($professor->documento_identificacao && Storage::disk('public')->exists($professor->documento_identificacao)) {
                Storage::disk('public')->delete($professor->documento_identificacao);
            }

            // Eliminar todos os registos relacionados (hard delete)
            // A ordem respeita as FKs: primeiro tabelas que referenciam professor_id
            DB::table('turma_disciplina_professor')->where('professor_id', $professor->id)->delete();
            DB::table('horarios')->where('professor_id', $professor->id)->delete();
            DB::table('trabalhos')->where('professor_id', $professor->id)->delete();
            DB::table('professor_payments')->where('professor_id', $professor->id)->delete();
            DB::table('planos_trimestrais')->where('professor_id', $professor->id)->delete();

            // Hard delete do professor (remove o registo da tabela professores)
            $professor->forceDelete();

            // Hard delete do user (remove o registo da tabela users)
            if ($user) {
                $user->forceDelete();
            }

            DB::commit();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Professor eliminado com sucesso!'
                ]);
            }

            return redirect()->route('admin.professores.index')->with('success', 'Professor eliminado com sucesso!');
        } catch (\Throwable $e) {
            DB::rollBack();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro ao eliminar professor: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()->with('error', 'Erro ao eliminar professor: ' . $e->getMessage());
        }
    }

    public function toggleStatusProfessor(Professor $professor)
    {
        if ($professor->user) {
            $professor->user->is_active = !$professor->user->is_active;
            $professor->user->save();
            
            $status = $professor->user->is_active ? 'ativado' : 'bloqueado';
            return redirect()->route('admin.professores.index')->with('success', "Professor {$status} com sucesso!");
        }

        return redirect()->route('admin.professores.index')->with('error', 'Não foi possível alterar o status do professor.');
    }

    /**
     * Atualizar valor por hora do professor via AJAX
     */
    public function updateValorHora(Request $request, Professor $professor)
    {
        $request->validate([
            'valor_hora' => 'nullable|numeric|min:0',
        ]);

        $professor->valor_hora = $request->valor_hora ?: null;
        $professor->save();

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Valor por hora atualizado com sucesso!',
                'valor_hora' => $professor->valor_hora
            ]);
        }

        return redirect()->back()->with('success', 'Valor por hora atualizado com sucesso!');
    }

    // Gestão de Turmas
    public function indexTurmas(Request $request)
    {
        // Endpoint para AJAX - retornar turmas por ano letivo
        if ($request->ajax() || $request->wantsJson()) {
            $anoLetivoId = $request->get('ano_letivo_id');
            if ($anoLetivoId) {
                $turmas = Turma::where('ano_letivo_id', $anoLetivoId)
                    ->where('ativa', true)
                    ->orderBy('codigo')
                    ->orderBy('nome')
                    ->get()
                    ->map(function($turma) {
                        return [
                            'id' => $turma->id,
                            'codigo' => $turma->codigo,
                            'nome' => $turma->nome,
                            'nivel_ensino' => $turma->nivel_ensino
                        ];
                    });
                
                return response()->json(['turmas' => $turmas]);
            }
            return response()->json(['turmas' => []]);
        }
        
        $query = Turma::with(['anoLetivo', 'alunos']);
        
        // Filtro por Ano Letivo - sempre usar 2026 por padrão
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        if ($request->filled('ano_letivo_id')) {
            // Validar que o ano selecionado seja 2026 ou outro ano válido
            $anoSelecionado = AnoLetivo::find($request->ano_letivo_id);
            if ($anoSelecionado) {
                $anoRaw = $anoSelecionado->getRawOriginal('ano');
                // Só permitir se for 2026 ou outro ano válido (não 2025)
                if (strpos($anoRaw, '2026') !== false || $anoRaw === '2026' || 
                    (strpos($anoRaw, '2027') !== false || $anoRaw === '2027')) {
                    $query->where('ano_letivo_id', $request->ano_letivo_id);
                } else {
                    // Se não for válido, usar 2026
                    if ($anoLetivoAtivo) {
                        $query->where('ano_letivo_id', $anoLetivoAtivo->id);
                    }
                }
            } else {
                // Se não encontrar, usar 2026
                if ($anoLetivoAtivo) {
                    $query->where('ano_letivo_id', $anoLetivoAtivo->id);
                }
            }
        } else {
            // Sem filtro, sempre usar 2026
            if ($anoLetivoAtivo) {
                $query->where('ano_letivo_id', $anoLetivoAtivo->id);
            }
        }
        
        // Filtro por nível de ensino
        if ($request->filled('nivel_ensino')) {
            $query->where('nivel_ensino', $request->nivel_ensino);
        }
        
        // Pesquisa
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nome', 'like', "%{$search}%")
                  ->orWhere('codigo', 'like', "%{$search}%");
            });
        }
        
        // Otimização: Paginação e select específico
        $turmas = $query->select('turmas.id', 'turmas.nome', 'turmas.codigo', 'turmas.nivel_ensino', 'turmas.ano_letivo_id', 'turmas.ativa', 'turmas.capacidade_maxima')
            ->with('anoLetivo:id,ano')
            ->orderBy('nome')
            ->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());
        
        $anoLetivos = AnoLetivo::getAnosLetivosDisponiveis();
        
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        
        return view('admin.turmas.index', compact('turmas', 'anoLetivos', 'anoLetivoAtivo'));
    }

    public function createTurma()
    {
        $anoLetivos = AnoLetivo::getAnosLetivosDisponiveis();
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        return view('admin.turmas.create', compact('anoLetivos', 'anoLetivoAtivo'));
    }

    public function storeTurma(Request $request)
    {
        $request->validate([
            'nome' => 'required|string|in:A,B,C,D,E,F,G,H,I,J,K,L',
            'codigo' => 'required|string',
            'nivel_ensino' => 'required|in:PRIMARIO,SECUNDARIO',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'capacidade_maxima' => 'nullable|integer|min:1',
        ], [
            'nome.required' => 'A turma é obrigatória.',
            'nome.in' => 'A turma deve ser uma letra de A a L.',
            'codigo.required' => 'A classe é obrigatória.',
            'nivel_ensino.required' => 'O nível de ensino é obrigatório.',
            'nivel_ensino.in' => 'O nível de ensino selecionado é inválido.',
            'ano_letivo_id.required' => 'O ano letivo é obrigatório.',
            'ano_letivo_id.exists' => 'O ano letivo selecionado é inválido.',
        ]);

        $numeroClasse = (int) preg_replace('/\D+/', '', $request->codigo);
        if ($numeroClasse < 1) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['codigo' => 'Classe inválida.']);
        }

        if ($request->nivel_ensino === 'PRIMARIO' && $numeroClasse > 6) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['codigo' => 'Ensino primário aceita apenas classes de 1ª a 6ª.']);
        }

        if ($request->nivel_ensino === 'SECUNDARIO' && $numeroClasse < 7) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['codigo' => 'Ensino secundário aceita apenas classes de 7ª a 12ª.']);
        }

        // Verificar se já existe uma turma com a mesma classe e letra no mesmo ano letivo
        $turmaExistente = Turma::withTrashed()
            ->where('codigo', $request->codigo)
            ->where('nome', $request->nome)
            ->where('ano_letivo_id', $request->ano_letivo_id)
            ->first();
            
        if ($turmaExistente) {
            if ($turmaExistente->trashed()) {
                $turmaExistente->restore();
                // Processar capacidade máxima
                $capacidade = 35; // Valor padrão
                $capacidadeInput = $request->get('capacidade_maxima');
                if ($capacidadeInput !== null && $capacidadeInput !== '') {
                    $capacidadeInput = trim((string)$capacidadeInput);
                    if ($capacidadeInput !== '' && is_numeric($capacidadeInput)) {
                        $capacidadeInt = (int)$capacidadeInput;
                        if ($capacidadeInt > 0) {
                            $capacidade = $capacidadeInt;
                        }
                    }
                }
                
                $turmaExistente->update([
                    'nivel_ensino' => $request->nivel_ensino,
                    'capacidade_maxima' => $capacidade,
                    'ativa' => $request->has('Activa') ? true : false,
                ]);
                return redirect()->route('admin.turmas.index')
                    ->with('success', 'Turma restaurada com sucesso!');
            }

            return redirect()->back()
                ->withInput()
                ->withErrors(['nome' => 'Já existe uma turma ' . $request->nome . ' para a ' . $request->codigo . ' neste ano letivo.']);
        }

        // Processar capacidade máxima
        $capacidade = 35; // Valor padrão
        $capacidadeInput = $request->get('capacidade_maxima');
        if ($capacidadeInput !== null && $capacidadeInput !== '') {
            $capacidadeInput = trim((string)$capacidadeInput);
            if ($capacidadeInput !== '' && is_numeric($capacidadeInput)) {
                $capacidadeInt = (int)$capacidadeInput;
                if ($capacidadeInt > 0) {
                    $capacidade = $capacidadeInt;
                }
            }
        }

        Turma::create([
            'nome' => $request->nome,
            'codigo' => $request->codigo,
            'nivel_ensino' => $request->nivel_ensino,
            'ano_letivo_id' => $request->ano_letivo_id,
            'capacidade_maxima' => $capacidade,
            'ativa' => $request->has('Activa') ? true : false,
        ]);

        return redirect()->route('admin.turmas.index')->with('success', 'Turma criada com sucesso!');
    }

    public function editTurma(Turma $turma)
    {
        $anoLetivos = AnoLetivo::getAnosLetivosDisponiveis();
        return view('admin.turmas.edit', compact('turma', 'anoLetivos'));
    }

    public function updateTurma(Request $request, Turma $turma)
    {
        $request->validate([
            'nome' => 'required|string|in:A,B,C,D,E,F,G,H,I,J,K,L',
            'codigo' => 'required|string',
            'nivel_ensino' => 'required|in:PRIMARIO,SECUNDARIO',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'capacidade_maxima' => 'nullable|integer|min:1',
        ], [
            'nome.required' => 'A turma é obrigatória.',
            'nome.in' => 'A turma deve ser uma letra de A a L.',
            'codigo.required' => 'A classe é obrigatória.',
            'nivel_ensino.required' => 'O nível de ensino é obrigatório.',
            'nivel_ensino.in' => 'O nível de ensino selecionado é inválido.',
            'ano_letivo_id.required' => 'O ano letivo é obrigatório.',
            'ano_letivo_id.exists' => 'O ano letivo selecionado é inválido.',
        ]);

        $numeroClasse = (int) preg_replace('/\D+/', '', $request->codigo);
        if ($numeroClasse < 1) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['codigo' => 'Classe inválida.']);
        }

        if ($request->nivel_ensino === 'PRIMARIO' && $numeroClasse > 6) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['codigo' => 'Ensino primário aceita apenas classes de 1ª a 6ª.']);
        }

        if ($request->nivel_ensino === 'SECUNDARIO' && $numeroClasse < 7) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['codigo' => 'Ensino secundário aceita apenas classes de 7ª a 12ª.']);
        }

        // Verificar se já existe outra turma com a mesma classe e letra no mesmo ano letivo
        $turmaExistente = Turma::where('codigo', $request->codigo)
            ->where('nome', $request->nome)
            ->where('ano_letivo_id', $request->ano_letivo_id)
            ->where('id', '!=', $turma->id)
            ->first();
            
        if ($turmaExistente) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['nome' => 'Já existe uma turma ' . $request->nome . ' para a ' . $request->codigo . ' neste ano letivo.']);
        }

        // Processar capacidade máxima
        $capacidade = $turma->capacidade_maxima ?? 35; // Manter valor atual ou usar padrão
        $capacidadeInput = $request->get('capacidade_maxima');
        if ($capacidadeInput !== null && $capacidadeInput !== '') {
            $capacidadeInput = trim((string)$capacidadeInput);
            if ($capacidadeInput !== '' && is_numeric($capacidadeInput)) {
                $capacidadeInt = (int)$capacidadeInput;
                if ($capacidadeInt > 0) {
                    $capacidade = $capacidadeInt;
                }
            }
        }

        $turma->update([
            'nome' => $request->nome,
            'codigo' => $request->codigo,
            'nivel_ensino' => $request->nivel_ensino,
            'ano_letivo_id' => $request->ano_letivo_id,
            'capacidade_maxima' => $capacidade,
            'ativa' => $request->has('Activa') ? true : false,
        ]);

        return redirect()->route('admin.turmas.index')->with('success', 'Turma atualizada com sucesso!');
    }

    public function destroyTurma(Turma $turma)
    {
        DB::beginTransaction();
        try {
            // Eliminar todos os registos relacionados (hard delete)
            DB::table('turma_aluno')->where('turma_id', $turma->id)->delete();
            DB::table('turma_disciplina')->where('turma_id', $turma->id)->delete();
            DB::table('turma_disciplina_professor')->where('turma_id', $turma->id)->delete();
            DB::table('horarios')->where('turma_id', $turma->id)->delete();
            DB::table('eventos')->where('turma_id', $turma->id)->delete();
            DB::table('avaliacoes')->where('turma_id', $turma->id)->delete();
            DB::table('trabalhos')->where('turma_id', $turma->id)->delete();

            // Hard delete da turma (remove o registo da tabela turmas)
            $turma->forceDelete();

            DB::commit();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Turma eliminada com sucesso!'
                ]);
            }

            return redirect()->route('admin.turmas.index')->with('success', 'Turma eliminada com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro ao eliminar turma: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()->with('error', 'Erro ao eliminar turma: ' . $e->getMessage());
        }
    }

    public function showTurma(Turma $turma)
    {
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $alunos = $turma->alunos()->wherePivot('ano_letivo_id', $anoLetivo?->id)->with('user')->get();
        $professores = $turma->professores()->wherePivot('ano_letivo_id', $anoLetivo?->id)->with('user')->get();
        
        return view('admin.turmas.show', compact('turma', 'alunos', 'professores', 'anoLetivo'));
    }

    // Alocação de Alunos
    public function alocarAlunos(Turma $turma)
    {
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        $alunosTurma = $turma->alunos()->wherePivot('ano_letivo_id', $anoLetivo?->id)->with('user')->get();
        
        // Alunos disponíveis: não estão alocados a nenhuma turma no ano letivo ativo
        // Usa uma subquery para verificar na tabela pivot
        $alunosDisponiveis = Aluno::with('user')
            ->whereNotExists(function($query) use ($anoLetivo) {
                $query->select(DB::raw(1))
                      ->from('turma_aluno')
                      ->join('turmas', 'turma_aluno.turma_id', '=', 'turmas.id')
                      ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                      ->where('turma_aluno.ano_letivo_id', $anoLetivo?->id)
                      ->whereNull('turmas.deleted_at');
            })
            ->get();

        return view('admin.turmas.alocar-alunos', compact('turma', 'alunosTurma', 'alunosDisponiveis', 'anoLetivo'));
    }

    public function storeAlocacaoAlunos(Request $request, Turma $turma)
    {
        $request->validate([
            'aluno_id' => 'required|array',
            'aluno_id.*' => 'exists:alunos,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
        ]);

        // Verificar capacidade máxima da turma (35 alunos)
        $capacidadeMaxima = $turma->capacidade_maxima ?? 35;
        $alunosNaTurma = DB::table('turma_aluno')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $request->ano_letivo_id)
            ->count();
        
        $alunosParaAlocar = count($request->aluno_id);
        $vagasDisponiveis = $capacidadeMaxima - $alunosNaTurma;
        
        if ($alunosParaAlocar > $vagasDisponiveis) {
            return redirect()->back()
                ->with('error', "A turma {$turma->codigo} - {$turma->nome} tem apenas {$vagasDisponiveis} vaga(s) disponível(is). Você tentou alocar {$alunosParaAlocar} aluno(s).");
        }
        
        $alocados = 0;
        foreach ($request->aluno_id as $alunoId) {
            // Verificar se já existe alocação para este aluno, turma e ano letivo
            $existe = DB::table('turma_aluno')
                ->where('aluno_id', $alunoId)
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $request->ano_letivo_id)
                ->exists();
            
            if (!$existe) {
                // Verificar se ainda há vagas
                if ($alunosNaTurma + $alocados < $capacidadeMaxima) {
                    // Alocar à nova turma
                    $turma->alunos()->attach($alunoId, ['ano_letivo_id' => $request->ano_letivo_id]);
                    $alocados++;
                }
            }
        }

        return redirect()->route('admin.turmas.alocar-alunos', $turma)->with('success', 'Alunos alocados com sucesso!');
    }

    public function removerAluno(Request $request, Turma $turma, Aluno $aluno)
    {
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        $turma->alunos()->wherePivot('ano_letivo_id', $anoLetivo?->id)->detach($aluno->id);
        
        return redirect()->route('admin.turmas.alocar-alunos', $turma)->with('success', 'Aluno removido da turma com sucesso!');
    }




    // Gestão de Disciplinas
    public function indexDisciplinas(Request $request)
    {
        $query = Disciplina::query();
        
        // Filtro por nível de ensino
        if ($request->filled('nivel_ensino')) {
            $query->where('nivel_ensino', $request->nivel_ensino);
        }
        
        // Pesquisa
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nome', 'like', "%{$search}%")
                  ->orWhere('codigo', 'like', "%{$search}%");
            });
        }
        
        // Otimização: Paginação e select específico
        $disciplinas = $query->select('disciplinas.id', 'disciplinas.nome', 'disciplinas.codigo', 'disciplinas.nivel_ensino', 'disciplinas.carga_horaria_semanal', 'disciplinas.ativa')
            ->orderBy('nome')
            ->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());
        
        return view('admin.disciplinas.index', compact('disciplinas'));
    }

    public function createDisciplina()
    {
        return view('admin.disciplinas.create');
    }

    public function storeDisciplina(Request $request)
    {
        $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|unique:disciplinas,codigo',
            'nivel_ensino' => 'required|in:PRIMARIO,SECUNDARIO',
            'carga_horaria_horas' => 'required_without:carga_horaria_minutos|integer|min:0|max:40',
            'carga_horaria_minutos' => 'required_without:carga_horaria_horas|integer|min:0|max:59',
        ], [
            'nome.required' => 'O nome da disciplina é obrigatório.',
            'codigo.required' => 'O código da disciplina é obrigatório.',
            'codigo.unique' => 'Este código já está em uso.',
            'nivel_ensino.required' => 'O nível de ensino é obrigatório.',
            'nivel_ensino.in' => 'O nível de ensino selecionado é inválido.',
            'carga_horaria_horas.required_without' => 'A carga horária é obrigatória. Informe pelo menos horas ou minutos.',
            'carga_horaria_minutos.required_without' => 'A carga horária é obrigatória. Informe pelo menos horas ou minutos.',
        ]);

        // Calcular carga horária em minutos (obrigatória)
            $horas = (int) $request->input('carga_horaria_horas', 0);
            $minutos = (int) $request->input('carga_horaria_minutos', 0);
            $cargaHoraria = ($horas * 60) + $minutos;
        
        if ($cargaHoraria <= 0) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['carga_horaria_horas' => 'A carga horária deve ser maior que zero.']);
        }

        Disciplina::create([
            'nome' => $request->nome,
            'codigo' => $request->codigo,
            'modelo_avaliacao' => 'NACIONAL',
            'nivel_ensino' => $request->nivel_ensino,
            'carga_horaria_semanal' => $cargaHoraria,
            'ativa' => $request->has('Activa') ? true : false,
        ]);

        return redirect()->route('admin.disciplinas.index')->with('success', 'Disciplina criada com sucesso!');
    }

    public function editDisciplina(Disciplina $disciplina)
    {
        // Verificar se a disciplina foi deletada
        if ($disciplina->trashed()) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Esta disciplina foi eliminada e não pode ser editada.'
                ], 404);
            }
            return redirect()->route('admin.disciplinas.index')
                ->with('error', 'Esta disciplina foi eliminada e não pode ser editada.');
        }

        return view('admin.disciplinas.edit', compact('disciplina'));
    }

    public function updateDisciplina(Request $request, Disciplina $disciplina)
    {
        // Verificar se a disciplina foi deletada
        if ($disciplina->trashed()) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Esta disciplina foi eliminada e não pode ser atualizada.'
                ], 404);
            }
            return redirect()->route('admin.disciplinas.index')
                ->with('error', 'Esta disciplina foi eliminada e não pode ser atualizada.');
        }

        $request->validate([
            'nome' => 'required|string|max:255',
            'codigo' => 'required|string|unique:disciplinas,codigo,' . $disciplina->id,
            'nivel_ensino' => 'required|in:PRIMARIO,SECUNDARIO',
            'carga_horaria_horas' => 'required_without:carga_horaria_minutos|integer|min:0|max:40',
            'carga_horaria_minutos' => 'required_without:carga_horaria_horas|integer|min:0|max:59',
        ], [
            'nome.required' => 'O nome da disciplina é obrigatório.',
            'codigo.required' => 'O código da disciplina é obrigatório.',
            'codigo.unique' => 'Este código já está em uso.',
            'nivel_ensino.required' => 'O nível de ensino é obrigatório.',
            'nivel_ensino.in' => 'O nível de ensino selecionado é inválido.',
            'carga_horaria_horas.required_without' => 'A carga horária é obrigatória. Informe pelo menos horas ou minutos.',
            'carga_horaria_minutos.required_without' => 'A carga horária é obrigatória. Informe pelo menos horas ou minutos.',
        ]);

        // Calcular carga horária em minutos (obrigatória)
            $horas = (int) $request->input('carga_horaria_horas', 0);
            $minutos = (int) $request->input('carga_horaria_minutos', 0);
            $cargaHoraria = ($horas * 60) + $minutos;
        
        if ($cargaHoraria <= 0) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['carga_horaria_horas' => 'A carga horária deve ser maior que zero.']);
        }

        $disciplina->update([
            'nome' => $request->nome,
            'codigo' => $request->codigo,
            'modelo_avaliacao' => 'NACIONAL',
            'nivel_ensino' => $request->nivel_ensino,
            'carga_horaria_semanal' => $cargaHoraria,
            'ativa' => $request->has('Activa') ? true : false,
        ]);

        return redirect()->route('admin.disciplinas.index')->with('success', 'Disciplina atualizada com sucesso!');
    }

    public function destroyDisciplina(Disciplina $disciplina)
    {
        DB::beginTransaction();
        try {
            // Verificar se a disciplina já foi deletada
            if ($disciplina->trashed()) {
                if (request()->ajax() || request()->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Esta disciplina já foi eliminada.'
                    ], 404);
                }
                return redirect()->route('admin.disciplinas.index')
                    ->with('error', 'Esta disciplina já foi eliminada.');
            }

            // Eliminar todos os registos relacionados (hard delete)
            DB::table('turma_disciplina')->where('disciplina_id', $disciplina->id)->delete();
            DB::table('turma_disciplina_professor')->where('disciplina_id', $disciplina->id)->delete();
            DB::table('horarios')->where('disciplina_id', $disciplina->id)->delete();
            DB::table('avaliacoes')->where('disciplina_id', $disciplina->id)->delete();
            DB::table('trabalhos')->where('disciplina_id', $disciplina->id)->delete();
            DB::table('planos_trimestrais')->where('disciplina_id', $disciplina->id)->delete();
            DB::table('cambridge_grade_boundaries')->where('disciplina_id', $disciplina->id)->delete();

            // Hard delete da disciplina (remove o registo da tabela disciplinas)
            $disciplina->forceDelete();

            DB::commit();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Disciplina eliminada com sucesso!'
                ]);
            }

            return redirect()->route('admin.disciplinas.index')->with('success', 'Disciplina eliminada com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro ao eliminar disciplina: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()->with('error', 'Erro ao eliminar disciplina: ' . $e->getMessage());
        }
    }

    // Atribuição de Disciplinas à Turma
    public function atribuirDisciplinas(Turma $turma)
    {
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        // CORREÇÃO AUTOMÁTICA: Verificar se há professores alocados com disciplinas que não estão na tabela turma_disciplina
        // Se encontrar, criar automaticamente o registro em turma_disciplina
        if ($anoLetivo) {
            $professoresAlocados = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->select('disciplina_id', 'professor_id')
                ->distinct()
                ->get();
            
            foreach ($professoresAlocados as $alocacao) {
                // Verificar se a disciplina já está atribuída na tabela turma_disciplina
                $disciplinaJaAtribuida = DB::table('turma_disciplina')
                    ->where('turma_id', $turma->id)
                    ->where('disciplina_id', $alocacao->disciplina_id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->exists();
                
                if (!$disciplinaJaAtribuida) {
                    // Buscar a disciplina para obter a carga horária
                    $disciplina = Disciplina::find($alocacao->disciplina_id);
                    if ($disciplina) {
                        $cargaHoraria = $disciplina->carga_horaria_semanal ?? 0;
                        
                        // Criar registro na tabela turma_disciplina automaticamente
                        DB::table('turma_disciplina')->insert([
                            'turma_id' => $turma->id,
                            'disciplina_id' => $alocacao->disciplina_id,
                            'ano_letivo_id' => $anoLetivo->id,
                            'carga_horaria_semanal' => $cargaHoraria,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                }
            }
        }
        
        // Disciplinas já atribuídas à turma com seus professores
        $disciplinasAtribuidas = $turma->disciplinasAtribuidas()
            ->wherePivot('ano_letivo_id', $anoLetivo?->id)
            ->get();
        
        // Buscar professores alocados para cada disciplina atribuída
        $disciplinasComProfessores = [];
        foreach ($disciplinasAtribuidas as $disciplina) {
            $nivelDisciplina = $disciplina->nivel_ensino ?? 'PRIMARIO';
            
            $query = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('disciplina_id', $disciplina->id)
                ->where('ano_letivo_id', $anoLetivo?->id)
                ->join('professores', 'turma_disciplina_professor.professor_id', '=', 'professores.id')
                ->whereNull('professores.deleted_at')
                ->join('users', 'professores.user_id', '=', 'users.id')
                ->whereNull('users.deleted_at');
            
            // Filtrar por nível de ensino: professor deve ser do mesmo nível ou AMBOS
            if ($nivelDisciplina === 'PRIMARIO') {
                $query->where(function($q) {
                    $q->where('professores.nivel_ensino', 'PRIMARIO')
                      ->orWhere('professores.nivel_ensino', 'AMBOS');
                });
            } elseif ($nivelDisciplina === 'SECUNDARIO') {
                $query->where(function($q) {
                    $q->where('professores.nivel_ensino', 'SECUNDARIO')
                      ->orWhere('professores.nivel_ensino', 'AMBOS');
                });
            }
            
            $professorAlocado = $query
                ->select('users.name as professor_nome', 'professores.id as professor_id')
                ->first();
            
            $disciplinasComProfessores[] = [
                'disciplina' => $disciplina,
                'professor' => $professorAlocado
            ];
        }
        
        // Todas as disciplinas ativas compatíveis com o nível da turma
        $nivelEnsino = $turma->nivel_ensino ?? 'PRIMARIO';
        $todasDisciplinas = Disciplina::where('ativa', true)
            ->where('nivel_ensino', $nivelEnsino)
            ->orderBy('nome')
            ->get();
        
        // Filtrar apenas disciplinas não atribuídas
        $disciplinasIdsAtribuidas = $disciplinasAtribuidas->pluck('id')->toArray();
        $disciplinasDisponiveis = $todasDisciplinas->filter(function($disciplina) use ($disciplinasIdsAtribuidas) {
            return !in_array($disciplina->id, $disciplinasIdsAtribuidas);
        });
        
        // Para cada disciplina disponível, buscar professores que já lecionam essa disciplina
        $professoresPorDisciplina = [];
        foreach ($disciplinasDisponiveis as $disciplina) {
            // Filtrar professores pelo nível de ensino da disciplina
            // Se disciplina é PRIMARIO, mostrar professores PRIMARIO ou AMBOS
            // Se disciplina é SECUNDARIO, mostrar professores SECUNDARIO ou AMBOS
            $nivelDisciplina = $disciplina->nivel_ensino ?? 'PRIMARIO';
            
            // Buscar professores que já lecionam esta disciplina em qualquer turma
            $query = DB::table('turma_disciplina_professor')
                ->where('disciplina_id', $disciplina->id)
                ->where('ano_letivo_id', $anoLetivo?->id)
                ->join('professores', 'turma_disciplina_professor.professor_id', '=', 'professores.id')
                ->whereNull('professores.deleted_at')
                ->join('users', 'professores.user_id', '=', 'users.id')
                ->whereNull('users.deleted_at');
            
            // Filtrar por nível de ensino: professor deve ser do mesmo nível ou AMBOS
            if ($nivelDisciplina === 'PRIMARIO') {
                $query->where(function($q) {
                    $q->where('professores.nivel_ensino', 'PRIMARIO')
                      ->orWhere('professores.nivel_ensino', 'AMBOS');
                });
            } elseif ($nivelDisciplina === 'SECUNDARIO') {
                $query->where(function($q) {
                    $q->where('professores.nivel_ensino', 'SECUNDARIO')
                      ->orWhere('professores.nivel_ensino', 'AMBOS');
                });
            }
            
            $professoresDisciplina = $query
                ->select('professores.id', 'users.name')
                ->distinct()
                ->get()
                ->map(function($prof) {
                    return [
                        'id' => $prof->id,
                        'name' => $prof->name
                    ];
                });
            
            $professoresPorDisciplina[$disciplina->id] = $professoresDisciplina;
        }
        
        return view('admin.turmas.atribuir-disciplinas', compact('turma', 'disciplinasComProfessores', 'disciplinasDisponiveis', 'anoLetivo', 'professoresPorDisciplina'));
    }

    public function storeAtribuicaoDisciplinas(Request $request, Turma $turma)
    {
        $request->validate([
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'disciplinas' => 'required|array',
            'disciplinas.*.selecionada' => 'required|in:1',
            'disciplinas.*.professor_id' => 'required|exists:professores,id',
        ]);

        $anoLetivo = AnoLetivo::findOrFail($request->ano_letivo_id);
        $nivelEnsino = $turma->nivel_ensino ?? 'PRIMARIO';
        $disciplinasAtribuidas = 0;
        $erros = [];
        
        foreach ($request->disciplinas as $disciplinaId => $dados) {
            if (!isset($dados['selecionada']) || $dados['selecionada'] != '1') {
                continue;
            }

            $disciplina = Disciplina::find($disciplinaId);
            if (!$disciplina) {
                $erros[] = "Disciplina ID {$disciplinaId} não encontrada.";
                continue;
            }

            $professor = Professor::with('user')->find($dados['professor_id']);
            if (!$professor) {
                $erros[] = "Professor não encontrado para a disciplina {$disciplina->nome}.";
                continue;
            }

            // Validar nível de ensino da disciplina com a turma
            if ($disciplina->nivel_ensino !== $nivelEnsino) {
                $erros[] = "A disciplina {$disciplina->nome} não pertence ao nível de ensino da turma.";
                continue;
            }
            
            // Validar nível de ensino do professor com a disciplina
            $nivelDisciplina = $disciplina->nivel_ensino ?? 'PRIMARIO';
            $nivelProfessor = $professor->nivel_ensino ?? 'AMBOS';
            
            if ($nivelProfessor !== 'AMBOS') {
                if ($nivelDisciplina === 'PRIMARIO' && $nivelProfessor !== 'PRIMARIO') {
                    $erros[] = "O professor {$professor->user->name} é do ensino secundário e não pode lecionar a disciplina {$disciplina->nome} (primário).";
                    continue;
                }
                if ($nivelDisciplina === 'SECUNDARIO' && $nivelProfessor !== 'SECUNDARIO') {
                    $erros[] = "O professor {$professor->user->name} é do ensino primário e não pode lecionar a disciplina {$disciplina->nome} (secundário).";
                    continue;
                }
            }

            // Validar se o professor realmente dá esta disciplina
            $professorDaLeciona = DB::table('turma_disciplina_professor')
                ->where('professor_id', $professor->id)
                ->where('disciplina_id', $disciplina->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->exists();

            if (!$professorDaLeciona) {
                $erros[] = "O professor selecionado não leciona a disciplina {$disciplina->nome}.";
                continue;
            }

            // Verificar se já existe OUTRO professor com esta disciplina nesta turma
            $outroProfessorComDisciplina = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('disciplina_id', $disciplinaId)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->where('professor_id', '!=', $professor->id)
                ->exists();
            
            if ($outroProfessorComDisciplina) {
                // Buscar o nome do professor que já está alocado
                $professorExistente = DB::table('turma_disciplina_professor')
                    ->where('turma_id', $turma->id)
                    ->where('disciplina_id', $disciplinaId)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->join('professores', 'turma_disciplina_professor.professor_id', '=', 'professores.id')
                    ->join('users', 'professores.user_id', '=', 'users.id')
                    ->select('users.name as nome')
                    ->first();
                
                $nomeProfessorExistente = $professorExistente ? $professorExistente->nome : 'outro professor';
                $erros[] = "A disciplina {$disciplina->nome} já está atribuída ao professor {$nomeProfessorExistente} nesta turma. Uma turma não pode ter dois professores diferentes para a mesma disciplina.";
                continue;
            }

            // Verificar se já existe esta mesma alocação (mesmo professor, mesma disciplina)
            $alocacaoExistente = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('professor_id', $professor->id)
                ->where('disciplina_id', $disciplinaId)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->exists();

            if ($alocacaoExistente) {
                $erros[] = "O professor já está alocado à disciplina {$disciplina->nome} nesta turma.";
                continue;
            }
            
            // Verificar se a disciplina já está atribuída na tabela turma_disciplina
            $disciplinaJaAtribuida = DB::table('turma_disciplina')
                ->where('turma_id', $turma->id)
                ->where('disciplina_id', $disciplinaId)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->exists();

            // Usar a carga horária da disciplina (já definida)
            $cargaHoraria = $disciplina->carga_horaria_semanal ?? 0;

            // Criar registro na tabela turma_disciplina apenas se não existir
            if (!$disciplinaJaAtribuida) {
                DB::table('turma_disciplina')->insert([
                    'turma_id' => $turma->id,
                    'disciplina_id' => $disciplinaId,
                    'ano_letivo_id' => $anoLetivo->id,
                    'carga_horaria_semanal' => $cargaHoraria,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            // Criar registro na tabela turma_disciplina_professor
            DB::table('turma_disciplina_professor')->insert([
                'turma_id' => $turma->id,
                'professor_id' => $professor->id,
                'disciplina_id' => $disciplinaId,
                'ano_letivo_id' => $anoLetivo->id,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $disciplinasAtribuidas++;
        }

        if (count($erros) > 0 && $disciplinasAtribuidas == 0) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['disciplinas' => implode(' ', $erros)]);
        }

        $mensagem = $disciplinasAtribuidas > 0 
            ? "{$disciplinasAtribuidas} disciplina(s) atribuída(s) com sucesso!" 
            : 'Nenhuma disciplina foi atribuída.';

        if (count($erros) > 0) {
            $mensagem .= ' Alguns erros ocorreram: ' . implode(' ', $erros);
        }

        return redirect()->route('admin.turmas.atribuir-disciplinas', $turma)
            ->with('success', $mensagem);
    }

    public function removerDisciplina(Request $request, Turma $turma, Disciplina $disciplina)
    {
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        // Remover da tabela turma_disciplina
        DB::table('turma_disciplina')
            ->where('turma_id', $turma->id)
            ->where('disciplina_id', $disciplina->id)
            ->where('ano_letivo_id', $anoLetivo?->id)
            ->delete();
        
        // Também remover da tabela turma_disciplina_professor se houver
        DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('disciplina_id', $disciplina->id)
            ->where('ano_letivo_id', $anoLetivo?->id)
            ->delete();
        
        return redirect()->route('admin.turmas.atribuir-disciplinas', $turma)
            ->with('success', 'Disciplina removida com sucesso!');
    }

    // Inscrições - Mostrar todos os alunos da escola
    public function inscricoes(Request $request)
    {
        // Para renovações, usar anos letivos disponíveis para matrícula
        $anosLetivos = AnoLetivo::getAnosLetivosParaMatricula();
        $anoLetivoAtual = AnoLetivo::getAnoLetivoAtual();
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtual?->id);
        
        // Buscar alunos da escola com paginação e eager loading
        $query = Aluno::select('alunos.id', 'alunos.codigo_estudante', 'alunos.user_id')
            ->with([
                'user:id,name,is_active',
                'turmas' => function($q) use ($anoLetivoId) {
                    if ($anoLetivoId) {
                        $q->select('turmas.id', 'turmas.nome', 'turmas.codigo')
                          ->wherePivot('ano_letivo_id', $anoLetivoId);
                    }
                }
            ]);
        
        // Filtros de pesquisa
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('codigo_estudante', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filtrar alunos do ano letivo selecionado (se informado)
        if ($anoLetivoId) {
            $query->whereExists(function($q) use ($anoLetivoId) {
                $q->select(DB::raw(1))
                    ->from('turma_aluno')
                    ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                    ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
            });

            // Subqueries para status (evitar N+1 nas avaliações)
            $query->addSelect([
                'mfrd_min' => DB::table('avaliacoes')
                    ->selectRaw('MIN(mfrd)')
                    ->whereColumn('avaliacoes.aluno_id', 'alunos.id')
                    ->where('avaliacoes.ano_letivo_id', $anoLetivoId)
                    ->whereNotNull('avaliacoes.mfrd'),
                'mfrd_count' => DB::table('avaliacoes')
                    ->selectRaw('COUNT(*)')
                    ->whereColumn('avaliacoes.aluno_id', 'alunos.id')
                    ->where('avaliacoes.ano_letivo_id', $anoLetivoId)
                    ->whereNotNull('avaliacoes.mfrd'),
            ]);
        }

        // Excluir alunos já renovados no ano letivo atual
        if ($anoLetivoAtual && $anoLetivoId && $anoLetivoAtual->id != $anoLetivoId) {
            $query->whereNotExists(function($q) use ($anoLetivoAtual) {
                $q->select(DB::raw(1))
                    ->from('turma_aluno')
                    ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                    ->where('turma_aluno.ano_letivo_id', $anoLetivoAtual->id);
            });
        }

        $alunosPaginados = $query->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());

        // Classificar alunos (aprovado/reprovado/sem avaliação) sem N+1
        $alunosClassificados = $alunosPaginados->getCollection()->map(function($aluno) use ($anoLetivoId) {
            $turmaAnterior = $aluno->turmas->first();
            $status = 'sem_avaliacao';

            if ($anoLetivoId) {
                $countMfrd = (int) ($aluno->mfrd_count ?? 0);
                if ($countMfrd > 0) {
                    $status = \App\Models\Avaliacao::isAprovado($aluno->mfrd_min) ? 'aprovado' : 'reprovado';
                }
            }

            return [
                'aluno' => $aluno,
                'turma_anterior' => $turmaAnterior,
                'status' => $status,
                'ja_renovado' => false,
            ];
        });

        $alunosNaoRenovadosPaginated = $alunosPaginados->setCollection($alunosClassificados);
        
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : null;
        
        return view('admin.inscricoes.index', compact(
            'alunosNaoRenovadosPaginated',
            'anosLetivos',
            'anoLetivoId',
            'anoLetivoSelecionado',
            'anoLetivoAtual'
        ));
    }

    // Renovar matrícula de um aluno específico
    public function renovarAluno(Aluno $aluno, Request $request)
    {
        // Para renovações, usar anos letivos disponíveis para matrícula
        $anosLetivosDisponiveis = AnoLetivo::getAnosLetivosParaMatricula();
        $anoLetivoAtual = $anosLetivosDisponiveis->first();
        
        if (!$anoLetivoAtual) {
            return redirect()->route('admin.inscricoes.index')
                ->with('error', 'Nenhum ano letivo disponível para renovação.');
        }
        
        // Buscar turma atual do aluno no ano letivo anterior
        $anoLetivoAnteriorId = $request->get('ano_letivo_anterior_id');
        if (!$anoLetivoAnteriorId) {
            return redirect()->route('admin.inscricoes.index')
                ->with('error', 'Ano letivo anterior não especificado.');
        }
        
        $turmaAtual = $aluno->turmas()
            ->wherePivot('ano_letivo_id', $anoLetivoAnteriorId)
            ->first();
        
        if (!$turmaAtual) {
            return redirect()->route('admin.inscricoes.index')
                ->with('error', 'Aluno não possui turma no ano letivo anterior.');
        }
        
        // Verificar se aluno é aprovado ou reprovado
        $avaliacoes = $aluno->avaliacoes()
            ->where('ano_letivo_id', $anoLetivoAnteriorId)
            ->whereNotNull('mfrd')
            ->get();
        
        $isAprovado = false;
        if ($avaliacoes->isNotEmpty()) {
            // Aprovado se MFrD >= 10 (com arredondamento por excesso de 9.5-9.99 para 10)
            $isAprovado = $avaliacoes->every(function($avaliacao) {
                return \App\Models\Avaliacao::isAprovado($avaliacao->mfrd);
            });
        }
        
        // Extrair número da classe atual
        $classeAtual = $this->extrairNumeroClasse($turmaAtual->codigo);
        
        // Buscar turmas dos anos letivos disponíveis para matrícula
        $anosLetivosIds = $anosLetivosDisponiveis->pluck('id');
        
        if ($isAprovado) {
            // Aprovados: classe superior
            $classeDestino = $classeAtual + 1;
            $turmasDisponiveis = Turma::whereIn('ano_letivo_id', $anosLetivosIds)
                ->where('ativa', true)
                ->where(function($q) use ($classeDestino) {
                    $q->where('codigo', 'like', "{$classeDestino}%")
                      ->orWhere('codigo', 'like', "{$classeDestino}ª%")
                      ->orWhere('codigo', 'like', "{$classeDestino}º%")
                      ->orWhere('codigo', 'like', "{$classeDestino}ª Classe%")
                      ->orWhere('codigo', 'like', "{$classeDestino}º Classe%");
                })
                ->orderBy('codigo')
                ->orderBy('nome')
                ->get();
        } else {
            // Reprovados: mesma classe, turma diferente ou mesmo nome
            $turmasDisponiveis = Turma::whereIn('ano_letivo_id', $anosLetivosIds)
                ->where('ativa', true)
                ->where(function($q) use ($classeAtual, $turmaAtual) {
                    $q->where(function($subQ) use ($classeAtual) {
                        $subQ->where('codigo', 'like', "{$classeAtual}%")
                             ->orWhere('codigo', 'like', "{$classeAtual}ª%")
                             ->orWhere('codigo', 'like', "{$classeAtual}º%")
                             ->orWhere('codigo', 'like', "{$classeAtual}ª Classe%")
                             ->orWhere('codigo', 'like', "{$classeAtual}º Classe%");
                    });
                })
                ->where('id', '!=', $turmaAtual->id) // Turma diferente
                ->orderBy('codigo')
                ->orderBy('nome')
                ->get();
        }
        
        return view('admin.inscricoes.renovar', compact(
            'aluno',
            'turmaAtual',
            'turmasDisponiveis',
            'anoLetivoAtual',
            'anoLetivoAnteriorId',
            'isAprovado',
            'classeAtual',
            'anosLetivosDisponiveis'
        ));
    }

    public function processarRenovacaoAluno(Request $request, Aluno $aluno)
    {
        $request->validate([
            'turma_id' => 'required|exists:turmas,id',
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'ano_letivo_anterior_id' => 'required|exists:ano_letivo,id',
        ]);
        
        $anoLetivo = AnoLetivo::findOrFail($request->ano_letivo_id);
        $turma = Turma::findOrFail($request->turma_id);
        
        // Verificar se a turma pertence ao ano letivo
        if ($turma->ano_letivo_id != $anoLetivo->id) {
            return redirect()->back()
                ->with('error', 'A turma selecionada não pertence ao ano letivo atual.');
        }
        
        // Verificar se o ano letivo está disponível para matrícula
        $anosLetivosDisponiveis = AnoLetivo::getAnosLetivosParaMatricula();
        if (!$anosLetivosDisponiveis->pluck('id')->contains($anoLetivo->id)) {
            return redirect()->back()
                ->with('error', 'O ano letivo selecionado não está disponível para renovação.');
        }
        
        // Verificar se já está matriculado
        $existe = DB::table('turma_aluno')
            ->where('aluno_id', $aluno->id)
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->exists();
        
        if ($existe) {
            return redirect()->back()
                ->with('error', 'Aluno já está matriculado nesta turma.');
        }
        
        // Verificar capacidade máxima da turma (35 alunos)
        $capacidadeMaxima = $turma->capacidade_maxima ?? 35;
        $alunosNaTurma = DB::table('turma_aluno')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivo->id)
            ->count();
        
        if ($alunosNaTurma >= $capacidadeMaxima) {
            return redirect()->back()
                ->with('error', "A turma {$turma->codigo} - {$turma->nome} já atingiu a capacidade máxima de {$capacidadeMaxima} alunos.");
        }
        
        // Alocar aluno à nova turma (sem remover outras alocações)
        $aluno->turmas()->attach($turma->id, [
            'ano_letivo_id' => $anoLetivo->id
        ]);
        
        // Definir valor da mensalidade baseado na classe da turma (se não tiver valor personalizado)
        if (!$aluno->valor_mensalidade_personalizado || $aluno->valor_mensalidade_personalizado <= 0) {
            $financeiroService = app(\App\Services\FinanceiroService::class);
            $valorMensalidade = $financeiroService->getValorMensalidadePorTurma($turma);
            $aluno->update(['valor_mensalidade_personalizado' => $valorMensalidade]);
        }
        
        $anoLetivoAnteriorId = $request->input('ano_letivo_anterior_id');
        
        // O aluno agora está renovado e sairá automaticamente da lista de não renovados
        // porque a query filtra por !ja_renovado
        
        return redirect()->route('admin.inscricoes.index', ['ano_letivo_id' => $anoLetivoAnteriorId])
            ->with('success', "Matrícula renovada com sucesso! Aluno alocado à turma {$turma->codigo} - {$turma->nome}. O aluno foi movido para a lista de renovados.");
    }

    // Listar alunos com mais de 2 turmas ou classes
    public function alunosMultiplasTurmas(Request $request)
    {
        $anoLetivoAtual = AnoLetivo::getAnoLetivoAtual();
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtual?->id);
        
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        // Buscar todos os alunos com suas turmas
        $query = Aluno::with(['user', 'turmas' => function($q) use ($anoLetivoId) {
            if ($anoLetivoId) {
                $q->wherePivot('ano_letivo_id', $anoLetivoId);
            }
        }]);
        
        // Filtros
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('codigo_estudante', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }
        
        $alunos = $query->get();
        
        // Filtrar apenas alunos com mais de 2 turmas
        $alunosMultiplasTurmas = $alunos->filter(function($aluno) {
            return $aluno->turmas->count() > 2;
        })->map(function($aluno) {
            // Agrupar turmas por classe (codigo)
            $turmasPorClasse = $aluno->turmas->groupBy('codigo');
            $aluno->turmas_por_classe = $turmasPorClasse;
            $aluno->total_turmas = $aluno->turmas->count();
            $aluno->total_classes = $turmasPorClasse->count();
            return $aluno;
        });
        
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : null;
        
        return view('admin.alunos.multiplas-turmas', compact(
            'alunosMultiplasTurmas',
            'anosLetivos',
            'anoLetivoId',
            'anoLetivoSelecionado'
        ));
    }

    // Listar alunos que renovaram matrícula
    public function alunosRenovados(Request $request)
    {
        $anoLetivoAtual = AnoLetivo::getAnoLetivoAtual();
        $anoLetivoId = $request->get('ano_letivo_id', $anoLetivoAtual?->id);
        
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        // Buscar alunos que têm turma no ano letivo atual mas também tinham no ano anterior
        $query = Aluno::with(['user']);
        
        if ($anoLetivoId && $anoLetivoAtual && $anoLetivoId == $anoLetivoAtual->id) {
            // Buscar alunos que têm turma no ano letivo atual
            $query->whereExists(function($q) use ($anoLetivoId) {
                $q->select(DB::raw(1))
                  ->from('turma_aluno')
                  ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                  ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
            });
        }
        
        // Filtros
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('codigo_estudante', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }
        
        $alunos = $query->with(['turmas' => function($q) use ($anoLetivoId) {
            if ($anoLetivoId) {
                $q->wherePivot('ano_letivo_id', $anoLetivoId);
            }
        }])->get();
        
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : null;
        
        return view('admin.inscricoes.renovados', compact(
            'alunos',
            'anosLetivos',
            'anoLetivoId',
            'anoLetivoSelecionado'
        ));
    }

    // Função auxiliar para extrair número da classe
    private function extrairNumeroClasse($codigo)
    {
        // Tenta extrair número do código (ex: "1ª Classe", "2ª Classe", "1º", "2º", etc.)
        if (preg_match('/(\d+)/', $codigo, $matches)) {
            return (int)$matches[1];
        }
        return 1; // Default
    }

    // Renovação de Matrícula (mantido para compatibilidade, mas simplificado)
    public function renovacaoMatricula(Request $request)
    {
        $anoLetivoAtual = AnoLetivo::getAnoLetivoAtual();
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        $anoLetivoOrigemId = $request->get('ano_letivo_origem_id', $anoLetivoAtual?->id);
        $anoLetivoDestinoId = $request->get('ano_letivo_destino_id');
        
        $alunosParaRenovar = collect();
        $turmasOrigem = collect();
        
        if ($anoLetivoOrigemId) {
            $anoLetivoOrigem = AnoLetivo::find($anoLetivoOrigemId);
            if ($anoLetivoOrigem) {
                // Buscar alunos do ano letivo origem
                $alunosParaRenovar = Aluno::whereExists(function($q) use ($anoLetivoOrigemId) {
                    $q->select(DB::raw(1))
                      ->from('turma_aluno')
                      ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                      ->where('turma_aluno.ano_letivo_id', $anoLetivoOrigemId);
                })->with(['user', 'turmas' => function($q) use ($anoLetivoOrigemId) {
                    $q->wherePivot('ano_letivo_id', $anoLetivoOrigemId);
                }])->get();
                
                // Buscar turmas do ano letivo origem
                $turmasOrigem = Turma::where('ano_letivo_id', $anoLetivoOrigemId)
                    ->where('ativa', true)
                    ->orderBy('codigo')
                    ->orderBy('nome')
                    ->get();
            }
        }
        
        $turmasDestino = collect();
        if ($anoLetivoDestinoId) {
            $turmasDestino = Turma::where('ano_letivo_id', $anoLetivoDestinoId)
                ->where('ativa', true)
                ->orderBy('codigo')
                ->orderBy('nome')
                ->get();
        }
        
        return view('admin.alunos.renovacao-matricula', compact(
            'anosLetivos', 
            'anoLetivoAtual', 
            'anoLetivoOrigemId', 
            'anoLetivoDestinoId',
            'alunosParaRenovar',
            'turmasOrigem',
            'turmasDestino'
        ));
    }

    public function processarRenovacaoMatricula(Request $request)
    {
        $request->validate([
            'ano_letivo_origem_id' => 'required|exists:ano_letivo,id',
            'ano_letivo_destino_id' => 'required|exists:ano_letivo,id',
            'alunos' => 'required|array',
            'alunos.*.aluno_id' => 'required|exists:alunos,id',
            'alunos.*.turma_id' => 'required|exists:turmas,id',
        ]);

        $anoLetivoOrigem = AnoLetivo::findOrFail($request->ano_letivo_origem_id);
        $anoLetivoDestino = AnoLetivo::findOrFail($request->ano_letivo_destino_id);
        
        $renovados = 0;
        $erros = [];

        foreach ($request->alunos as $alunoData) {
            try {
                $aluno = Aluno::findOrFail($alunoData['aluno_id']);
                $turmaDestino = Turma::findOrFail($alunoData['turma_id']);
                
                // Verificar se a turma destino pertence ao ano letivo destino
                if ($turmaDestino->ano_letivo_id != $anoLetivoDestino->id) {
                    $erros[] = "Aluno " . ($aluno->user->name ?? 'Aluno') . ": A turma selecionada não pertence ao ano letivo destino.";
                    continue;
                }
                
                // Verificar se já existe alocação para este aluno na turma e ano letivo destino
                $existe = DB::table('turma_aluno')
                    ->where('aluno_id', $aluno->id)
                    ->where('turma_id', $turmaDestino->id)
                    ->where('ano_letivo_id', $anoLetivoDestino->id)
                    ->exists();
                
                if (!$existe) {
                    // Verificar capacidade máxima da turma (35 alunos)
                    $capacidadeMaxima = $turmaDestino->capacidade_maxima ?? 35;
                    $alunosNaTurma = DB::table('turma_aluno')
                        ->where('turma_id', $turmaDestino->id)
                        ->where('ano_letivo_id', $anoLetivoDestino->id)
                        ->count();
                    
                    if ($alunosNaTurma >= $capacidadeMaxima) {
                        $erros[] = "Aluno " . ($aluno->user->name ?? 'Aluno') . ": A turma {$turmaDestino->codigo} - {$turmaDestino->nome} já atingiu a capacidade máxima de {$capacidadeMaxima} alunos.";
                        continue;
                    }
                    
                    // Alocar aluno à turma destino no novo ano letivo
                    $aluno->turmas()->attach($turmaDestino->id, [
                        'ano_letivo_id' => $anoLetivoDestino->id
                    ]);
                    
                    // Definir valor da mensalidade baseado na classe da turma (se não tiver valor personalizado)
                    if (!$aluno->valor_mensalidade_personalizado || $aluno->valor_mensalidade_personalizado <= 0) {
                        $financeiroService = app(\App\Services\FinanceiroService::class);
                        $valorMensalidade = $financeiroService->getValorMensalidadePorTurma($turmaDestino);
                        $aluno->update(['valor_mensalidade_personalizado' => $valorMensalidade]);
                    }
                    
                    $renovados++;
                } else {
                    $erros[] = "Aluno " . ($aluno->user->name ?? 'Aluno') . ": Já está matriculado nesta turma no ano letivo destino.";
                }
            } catch (\Exception $e) {
                $erros[] = "Erro ao renovar aluno ID {$alunoData['aluno_id']}: " . $e->getMessage();
            }
        }

        $mensagem = "Renovação concluída! {$renovados} aluno(s) renovado(s) com sucesso.";
        if (!empty($erros)) {
            $mensagem .= " Erros: " . implode('; ', $erros);
        }

        return redirect()->route('admin.inscricoes.index')
            ->with('success', $mensagem)
            ->with('erros', $erros);
    }

    // Alunos Inscritos por Ano Letivo
    public function alunosInscritos(Request $request)
    {
        $anoLetivoId = $request->get('ano_letivo_id');
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivoId) {
            $anoLetivoId = $anoLetivoAtivo?->id;
        }
        
        $anosLetivos = AnoLetivo::orderBy('ano', 'desc')->get();
        
        $query = Aluno::with(['user', 'turmas' => function($q) use ($anoLetivoId) {
            if ($anoLetivoId) {
                $q->wherePivot('ano_letivo_id', $anoLetivoId);
            }
        }]);
        
        if ($anoLetivoId) {
            $query->whereExists(function($q) use ($anoLetivoId) {
                $q->select(DB::raw(1))
                  ->from('turma_aluno')
                  ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                  ->where('turma_aluno.ano_letivo_id', $anoLetivoId);
            });
        }
        
        // Filtros
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('codigo_estudante', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%");
                  });
            });
        }
        
        if ($request->filled('turma_id')) {
            $query->whereExists(function($q) use ($request, $anoLetivoId) {
                $q->select(DB::raw(1))
                  ->from('turma_aluno')
                  ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                  ->where('turma_aluno.turma_id', $request->turma_id);
                if ($anoLetivoId) {
                    $q->where('turma_aluno.ano_letivo_id', $anoLetivoId);
                }
            });
        }
        
        // Otimização: Paginação e select específico
        $alunos = $query->select('alunos.id', 'alunos.codigo_estudante', 'alunos.user_id')
            ->with('user:id,name,is_active')
            ->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());
        
        $turmas = collect();
        if ($anoLetivoId) {
            $turmas = Turma::where('ano_letivo_id', $anoLetivoId)
                ->where('ativa', true)
                ->orderBy('codigo')
                ->orderBy('nome')
                ->get();
        }
        
        $anoLetivoSelecionado = $anoLetivoId ? AnoLetivo::find($anoLetivoId) : null;
        
        return view('admin.alunos.inscritos', compact(
            'alunos', 
            'anosLetivos', 
            'anoLetivoId', 
            'anoLetivoSelecionado',
            'turmas'
        ));
    }

    // Configurações - Controle de Lançamento de Notas
    public function configuracoesLancamentoNotas()
    {
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        $trimestres = $anoLetivo ? $anoLetivo->trimestres()->orderBy('numero')->get() : collect();
        
        // Buscar configurações por trimestre (JSON)
        $lancamentoPorTrimestre = Configuracao::get('lancamento_notas_trimestres', []);
        if (is_string($lancamentoPorTrimestre)) {
            $lancamentoPorTrimestre = json_decode($lancamentoPorTrimestre, true) ?? [];
        }
        
        // Para cada trimestre, verificar se está liberado (padrão: true se não especificado)
        foreach ($trimestres as $trimestre) {
            if (!isset($lancamentoPorTrimestre[$trimestre->id])) {
                $lancamentoPorTrimestre[$trimestre->id] = true; // Padrão: liberado
            }
        }
        
        return view('admin.configuracoes-lancamento-notas', compact('trimestres', 'anoLetivo', 'lancamentoPorTrimestre'));
    }

    public function updateConfiguracoesLancamentoNotas(Request $request)
    {
        // Validação opcional - permite array vazio (todos trancados)
        $request->validate([
            'trimestres' => 'nullable|array',
        ]);

        // Buscar todos os trimestres do ano letivo ativo
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        if (!$anoLetivo) {
            return redirect()->back()->withErrors(['error' => 'Nenhum ano letivo ativo encontrado.']);
        }
        
        $trimestres = $anoLetivo->trimestres()->orderBy('numero')->get();
        
        // Atualiza controle de lançamento de notas por trimestre
        // Com o campo hidden, todos os trimestres sempre estarão no array
        // Se o valor é "1", está marcado (liberado = true)
        // Se o valor é "0" ou não existe, está desmarcado (liberado = false)
        $lancamentoPorTrimestre = [];
        foreach ($trimestres as $trimestre) {
            // Verifica se o valor é "1" (checkbox marcado)
            $lancamentoPorTrimestre[$trimestre->id] = isset($request->trimestres[$trimestre->id]) && $request->trimestres[$trimestre->id] == '1';
        }
        
        Configuracao::set('lancamento_notas_trimestres', $lancamentoPorTrimestre, 'json', 'Controla se o lançamento de notas está liberado (true) ou trancado (false) por trimestre');

        return redirect()->route('admin.configuracoes.lancamento-notas')->with('success', 'Configuração de lançamento de notas atualizada com sucesso!');
    }

    /**
     * Listar inscrições pendentes
     */
    public function inscricoesPendentes(Request $request)
    {
        $query = Aluno::with('user')
            ->where('status_inscricao', 'PENDENTE')
            ->orderBy('data_submissao', 'desc');

        $inscricoes = $query->paginate(20)->appends(request()->query());

        return view('admin.inscricoes.pendentes', compact('inscricoes'));
    }

    /**
     * Aprovar inscrição
     */
    public function aprovarInscricao(Aluno $aluno, Request $request)
    {
        $aluno->update([
            'status_inscricao' => 'APROVADO',
            'observacoes_inscricao' => $request->observacoes,
        ]);

        // Ativar usuário
        if ($aluno->user) {
            $aluno->user->update(['is_active' => true]);
        }

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Inscrição aprovada com sucesso!'
            ]);
        }

        return redirect()->route('admin.inscricoes.pendentes')
            ->with('success', 'Inscrição aprovada com sucesso!');
    }

    /**
     * Rejeitar inscrição
     */
    public function rejeitarInscricao(Aluno $aluno, Request $request)
    {
        $request->validate([
            'observacoes' => 'required|string|min:10',
        ]);

        $aluno->update([
            'status_inscricao' => 'REJEITADO',
            'observacoes_inscricao' => $request->observacoes,
        ]);

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Inscrição rejeitada.'
            ]);
        }

        return redirect()->route('admin.inscricoes.pendentes')
            ->with('success', 'Inscrição rejeitada.');
    }

    /**
     * Visualizar detalhes da inscrição
     */
    public function visualizarInscricao(Aluno $aluno)
    {
        $aluno->load('user');
        return view('admin.inscricoes.visualizar', compact('aluno'));
    }
}



