<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Professor;
use App\Models\ProfessorPayment;
use App\Models\Configuracao;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;

class ProfessorPaymentController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin,superadmin,funcionario']);
    }

    /**
     * Verifica se o usuário tem permissão para acessar o recurso
     */
    private function verificarPermissao()
    {
        $user = auth()->user();
        
        // Admin e Superadmin têm todas as permissões
        if ($user->isAdmin() || $user->isSuperadmin()) {
            return true;
        }
        
        // Funcionários precisam ter a permissão específica
        if ($user->isFuncionario()) {
            if (!$user->podeGerir('gerir_pagamentos_professores')) {
                abort(403, 'Acesso negado. Você não tem permissão para acessar pagamentos de professores.');
            }
        }
        
        return true;
    }

    /**
     * Listar pagamentos de professores
     */
    public function index(Request $request)
    {
        $this->verificarPermissao();
        
        $query = ProfessorPayment::with(['professor.user', 'createdBy'])
            ->whereHas('professor', function($q) {
                $q->whereNull('deleted_at');
            });

        // Filtros
        if ($request->has('professor_id') && $request->professor_id !== '' && $request->professor_id !== null) {
            $query->where('professor_id', $request->professor_id);
        }

        if ($request->has('mes') && $request->mes !== '' && $request->mes !== null) {
            $query->where('mes', $request->mes);
        }

        if ($request->has('ano') && $request->ano !== '' && $request->ano !== null) {
            $query->where('ano', $request->ano);
        }

        if ($request->has('status') && $request->status !== '' && $request->status !== null) {
            $query->where('status', $request->status);
        }

        $payments = $query->orderBy('ano', 'desc')
            ->orderBy('mes', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(25);

        // Atualizar pagamentos que não têm INSS e IRPS calculados
        foreach ($payments as $payment) {
            if (($payment->inss == 0 && $payment->irps == 0 && $payment->valor_liquido == 0) || 
                ($payment->inss === null || $payment->irps === null)) {
                $inss = $this->calcularINSS($payment->valor_total);
                $numeroDependentes = $payment->professor->numero_dependentes ?? 0;
                $irps = $this->calcularIRPS($payment->valor_total, $numeroDependentes);
                $valorLiquido = $payment->valor_total - $inss - $irps;
                
                $payment->update([
                    'inss' => $inss,
                    'irps' => $irps,
                    'valor_liquido' => $valorLiquido,
                ]);
            }
        }

        $professores = Professor::with('user')
            ->whereHas('user', function($q) {
                $q->whereNull('deleted_at');
            })
            ->orderBy('numero_funcionario')
            ->get();

        // Se for requisição AJAX, retornar JSON
        $isAjax = $request->ajax() || 
                  $request->wantsJson() || 
                  $request->header('X-Requested-With') === 'XMLHttpRequest' ||
                  $request->expectsJson();
        
        if ($isAjax) {
            return response()->json([
                'success' => true,
                'payments' => $payments->map(function($payment) {
                    return [
                        'id' => $payment->id,
                        'professor_id' => $payment->professor_id,
                        'professor_nome' => $payment->professor && $payment->professor->user ? $payment->professor->user->name : 'Professor eliminado',
                        'professor_codigo' => $payment->professor ? $payment->professor->numero_funcionario : '-',
                        'mes' => $payment->mes,
                        'ano' => $payment->ano,
                        'mes_abrev' => ['', 'Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set', 'Out', 'Nov', 'Dez'][$payment->mes],
                        'ano_abrev' => substr($payment->ano, -2),
                        'horas_trabalhadas' => $payment->horas_trabalhadas,
                        'valor_hora' => $payment->professor ? ($payment->professor->valor_hora ?? 0) : 0,
                        'numero_dependentes' => $payment->professor ? ($payment->professor->numero_dependentes ?? 0) : 0,
                        'salario_fixo' => $payment->professor ? ($payment->professor->salario_fixo ?? 0) : 0,
                        'nivel_ensino' => $payment->professor ? ($payment->professor->nivel_ensino ?? 'AMBOS') : 'AMBOS',
                        'valor_total' => $payment->valor_total,
                        'inss' => $payment->inss ?? 0,
                        'irps' => $payment->irps ?? 0,
                        'valor_liquido' => $payment->valor_liquido ?? ($payment->valor_total - ($payment->inss ?? 0) - ($payment->irps ?? 0)),
                        'status' => $payment->status,
                        'data_pagamento' => $payment->data_pagamento ? $payment->data_pagamento->format('d/m/y') : null,
                    ];
                }),
                'pagination' => [
                    'current_page' => $payments->currentPage(),
                    'last_page' => $payments->lastPage(),
                    'per_page' => $payments->perPage(),
                    'total' => $payments->total(),
                ],
                'professores' => $professores->map(function($prof) {
                    return [
                        'id' => $prof->id,
                        'nome' => $prof->user ? $prof->user->name : '-',
                        'codigo' => $prof->numero_funcionario,
                        'nivel_ensino' => $prof->nivel_ensino ?? 'AMBOS',
                        'salario_fixo' => $prof->salario_fixo ?? 0,
                        'valor_hora' => $prof->valor_hora ?? 0,
                        'numero_dependentes' => $prof->numero_dependentes ?? 0,
                    ];
                }),
            ]);
        }

        return view('admin.professor-payments.index', compact('payments', 'professores'));
    }

    /**
     * Criar novo pagamento
     */
    public function create(Request $request)
    {
        $this->verificarPermissao();
        
        $professorId = $request->get('professor_id');
        $professores = Professor::with('user')
            ->whereHas('user', function($q) {
                $q->whereNull('deleted_at');
            })
            ->orderBy('numero_funcionario')
            ->get();

        $professor = $professorId ? Professor::with('user')->find($professorId) : null;

        return view('admin.professor-payments.create', compact('professores', 'professor'));
    }

    /**
     * Calcular valor do pagamento
     */
    public function calcular(Request $request)
    {
        $this->verificarPermissao();
        
        $request->validate([
            'professor_id' => 'required|exists:professores,id',
            'mes' => 'required|integer|min:1|max:12',
            'ano' => 'required|integer|min:2020|max:2100',
            'horas_trabalhadas' => 'nullable|numeric|min:0',
        ]);

        $professor = Professor::findOrFail($request->professor_id);
        $horasTrabalhadas = $request->horas_trabalhadas;

        $valorTotal = $professor->calcularSalario($horasTrabalhadas);
        
        // Calcular INSS e IRPS
        $inss = $this->calcularINSS($valorTotal);
        $numeroDependentes = $professor->numero_dependentes ?? 0;
        $irps = $this->calcularIRPS($valorTotal, $numeroDependentes);
        $valorLiquido = $valorTotal - $inss - $irps;

        return response()->json([
            'success' => true,
            'valor_total' => number_format($valorTotal, 2, '.', ''),
            'inss' => number_format($inss, 2, '.', ''),
            'irps' => number_format($irps, 2, '.', ''),
            'valor_liquido' => number_format($valorLiquido, 2, '.', ''),
            'tipo_calculo' => $professor->nivel_ensino === 'PRIMARIO' ? 'fixo' : 'por_horas',
        ]);
    }

    /**
     * Armazenar pagamento
     */
    public function store(Request $request)
    {
        $this->verificarPermissao();
        
        $request->validate([
            'professor_id' => 'required|exists:professores,id',
            'mes' => 'required|integer|min:1|max:12',
            'ano' => 'required|integer|min:2020|max:2100',
            'horas_trabalhadas' => 'nullable|numeric|min:0',
            'valor_total' => 'required|numeric|min:0',
            'observacoes' => 'nullable|string|max:1000',
        ]);

        $professor = Professor::findOrFail($request->professor_id);

        // Verificar se já existe pagamento para este mês/ano
        $pagamentoExistente = ProfessorPayment::where('professor_id', $professor->id)
            ->where('mes', $request->mes)
            ->where('ano', $request->ano)
            ->first();

        if ($pagamentoExistente) {
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Já existe um pagamento registrado para este professor neste mês/ano.'
                ], 422);
            }
            return redirect()->back()
                ->withErrors(['error' => 'Já existe um pagamento registrado para este professor neste mês/ano.'])
                ->withInput();
        }

        // Validar horas para secundário
        if ($professor->nivel_ensino === 'SECUNDARIO' && !$request->filled('horas_trabalhadas')) {
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Horas trabalhadas são obrigatórias para professores do secundário.'
                ], 422);
            }
            return redirect()->back()
                ->withErrors(['horas_trabalhadas' => 'Horas trabalhadas são obrigatórias para professores do secundário.'])
                ->withInput();
        }

        // Calcular valor se necessário
        $valorTotal = $request->valor_total;
        if ($professor->nivel_ensino === 'SECUNDARIO' && $request->filled('horas_trabalhadas')) {
            $valorCalculado = $professor->calcularSalario($request->horas_trabalhadas);
            if (abs($valorCalculado - $valorTotal) > 0.01) {
                // Usar valor calculado se houver diferença
                $valorTotal = $valorCalculado;
            }
        }

        // Calcular INSS e IRPS
        $inss = $this->calcularINSS($valorTotal);
        $numeroDependentes = $professor->numero_dependentes ?? 0;
        $irps = $this->calcularIRPS($valorTotal, $numeroDependentes);
        $valorLiquido = $valorTotal - $inss - $irps;

        $payment = ProfessorPayment::create([
            'professor_id' => $professor->id,
            'mes' => $request->mes,
            'ano' => $request->ano,
            'horas_trabalhadas' => $request->horas_trabalhadas,
            'valor_total' => $valorTotal,
            'inss' => $inss,
            'irps' => $irps,
            'valor_liquido' => $valorLiquido,
            'status' => 'PENDENTE',
            'observacoes' => $request->observacoes,
            'created_by' => Auth::id(),
        ]);

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento registrado com sucesso!',
                'payment' => $payment
            ]);
        }

        return redirect()->route('admin.professor-payments.index')
            ->with('success', 'Pagamento registrado com sucesso!');
    }

    /**
     * Atualizar pagamento (horas, valor bruto, recalcular impostos)
     */
    public function update(Request $request, ProfessorPayment $payment)
    {
        $this->verificarPermissao();
        
        if ($payment->status === 'PAGO') {
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Não é possível atualizar um pagamento já pago.'
                ], 422);
            }
            return redirect()->back()
                ->withErrors(['error' => 'Não é possível atualizar um pagamento já pago.']);
        }

        $request->validate([
            'horas_trabalhadas' => 'nullable|numeric|min:0',
            'valor_total' => 'required|numeric|min:0',
        ]);

        $valorTotal = $request->valor_total;
        
        // Recalcular INSS e IRPS
        $inss = $this->calcularINSS($valorTotal);
        $numeroDependentes = $payment->professor ? ($payment->professor->numero_dependentes ?? 0) : 0;
        $irps = $this->calcularIRPS($valorTotal, $numeroDependentes);
        $valorLiquido = $valorTotal - $inss - $irps;

        $payment->update([
            'horas_trabalhadas' => $request->horas_trabalhadas,
            'valor_total' => $valorTotal,
            'inss' => $inss,
            'irps' => $irps,
            'valor_liquido' => $valorLiquido,
        ]);

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento atualizado com sucesso!',
                'payment' => $payment->fresh(),
                'inss' => number_format($inss, 2, '.', ''),
                'irps' => number_format($irps, 2, '.', ''),
                'valor_liquido' => number_format($valorLiquido, 2, '.', ''),
            ]);
        }

        return redirect()->back()
            ->with('success', 'Pagamento atualizado com sucesso!');
    }

    /**
     * Marcar como pago
     */
    public function marcarComoPago(Request $request, ProfessorPayment $payment)
    {
        $this->verificarPermissao();
        
        $payment->update([
            'status' => 'PAGO',
            'data_pagamento' => now(),
        ]);

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento marcado como pago!',
                'payment' => [
                    'id' => $payment->id,
                    'status' => $payment->status,
                    'data_pagamento' => $payment->data_pagamento->format('d/m/y'),
                ]
            ]);
        }

        return redirect()->back()
            ->with('success', 'Pagamento marcado como pago!');
    }

    /**
     * Marcar todos os pagamentos pendentes como pagos
     */
    public function marcarTodosComoPago(Request $request)
    {
        $this->verificarPermissao();
        
        $query = ProfessorPayment::where('status', 'PENDENTE');
        
        // Aplicar filtros se fornecidos
        if ($request->filled('mes')) {
            $query->where('mes', $request->mes);
        }
        
        if ($request->filled('ano')) {
            $query->where('ano', $request->ano);
        }
        
        if ($request->filled('professor_id')) {
            $query->where('professor_id', $request->professor_id);
        }
        
        $payments = $query->get();
        
        if ($payments->isEmpty()) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Não há pagamentos pendentes para marcar como pagos.'
                ]);
            }
            return redirect()->back()
                ->with('info', 'Não há pagamentos pendentes para marcar como pagos.');
        }
        
        $count = $payments->count();
        
        foreach ($payments as $payment) {
            $payment->update([
                'status' => 'PAGO',
                'data_pagamento' => now(),
            ]);
        }
        
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => "{$count} pagamento(s) marcado(s) como pago(s) com sucesso!",
                'count' => $count
            ]);
        }
        
        return redirect()->back()
            ->with('success', "{$count} pagamento(s) marcado(s) como pago(s) com sucesso!");
    }

    /**
     * Cancelar pagamento
     */
    public function cancelar(ProfessorPayment $payment)
    {
        $this->verificarPermissao();
        
        if ($payment->status === 'PAGO') {
            return redirect()->back()
                ->withErrors(['error' => 'Não é possível cancelar um pagamento já pago.']);
        }

        $payment->update([
            'status' => 'CANCELADO',
        ]);

        return redirect()->back()
            ->with('success', 'Pagamento cancelado!');
    }

    /**
     * Deletar pagamento
     */
    public function destroy(ProfessorPayment $payment)
    {
        $this->verificarPermissao();
        
        if ($payment->status === 'PAGO') {
            return redirect()->back()
                ->withErrors(['error' => 'Não é possível deletar um pagamento já pago.']);
        }

        $payment->delete();

        return redirect()->back()
            ->with('success', 'Pagamento deletado com sucesso!');
    }

    /**
     * Gerar folha de salário (todos os professores ou específico)
     */
    public function gerarFolhaSalario(Request $request)
    {
        $this->verificarPermissao();
        
        $professorId = $request->get('professor_id');
        $mes = $request->get('mes', date('m'));
        $ano = $request->get('ano', date('Y'));

        // Buscar pagamentos
        $query = ProfessorPayment::with(['professor.user', 'createdBy'])
            ->whereHas('professor', function($q) {
                $q->whereNull('deleted_at');
            })
            ->where('mes', $mes)
            ->where('ano', $ano)
            ->where('status', 'PAGO');

        if ($professorId) {
            $query->where('professor_id', $professorId);
        }

        $payments = $query->orderBy('professor_id')->get();

        if ($payments->isEmpty()) {
            return redirect()->back()
                ->with('error', 'Nenhum pagamento encontrado para o período selecionado.');
        }

        // Preparar dados para a folha
        $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
                 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
        
        $mesNome = $meses[(int)$mes];
        $totalGeral = $payments->sum('valor_total');

        // Buscar nome da escola e logotipo
        $nomeEscola = Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE');
        $logotipo = Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $mime = mime_content_type($logotipoPath);
            $logotipoBase64 = "data:{$mime};base64," . base64_encode($logotipoData);
        }

        // Filtrar pagamentos com professor válido
        $paymentsValidos = $payments->filter(function($payment) {
            return $payment->professor && $payment->professor->user;
        });

        if ($paymentsValidos->isEmpty()) {
            return redirect()->back()
                ->with('error', 'Nenhum pagamento válido encontrado (professores podem ter sido eliminados).');
        }

        $professorEspecifico = null;
        $nomeArquivo = 'folha-salario-todos-' . $mesNome . '-' . $ano . '.pdf';
        
        if ($professorId && $paymentsValidos->first()) {
            $professorEspecifico = $paymentsValidos->first()->professor;
            if ($professorEspecifico) {
                $nomeArquivo = 'folha-salario-' . $professorEspecifico->numero_funcionario . '-' . $mesNome . '-' . $ano . '.pdf';
            }
        }

        // Gerar PDF
        $pdf = Pdf::loadView('admin.professor-payments.folha-salario', [
            'payments' => $paymentsValidos,
            'mes' => $mes,
            'ano' => $ano,
            'mesNome' => $mesNome,
            'totalGeral' => $paymentsValidos->sum('valor_total'),
            'professorEspecifico' => $professorEspecifico,
            'nomeEscola' => $nomeEscola,
            'logotipoBase64' => $logotipoBase64,
        ]);

        return $pdf->download($nomeArquivo);
    }

    /**
     * Visualizar folha de salário (HTML para impressão)
     */
    public function visualizarFolhaSalario(Request $request)
    {
        $this->verificarPermissao();
        
        $professorId = $request->get('professor_id');
        $mes = $request->get('mes', date('m'));
        $ano = $request->get('ano', date('Y'));

        // Buscar pagamentos
        $query = ProfessorPayment::with(['professor.user', 'createdBy'])
            ->whereHas('professor', function($q) {
                $q->whereNull('deleted_at');
            })
            ->where('mes', $mes)
            ->where('ano', $ano)
            ->where('status', 'PAGO');

        if ($professorId) {
            $query->where('professor_id', $professorId);
        }

        $payments = $query->orderBy('professor_id')->get();

        if ($payments->isEmpty()) {
            return redirect()->back()
                ->with('error', 'Nenhum pagamento encontrado para o período selecionado.');
        }

        // Preparar dados para a folha
        $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
                 'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
        
        $mesNome = $meses[(int)$mes];
        
        // Filtrar pagamentos com professor válido
        $paymentsValidos = $payments->filter(function($payment) {
            return $payment->professor && $payment->professor->user;
        });

        if ($paymentsValidos->isEmpty()) {
            return redirect()->back()
                ->with('error', 'Nenhum pagamento válido encontrado (professores podem ter sido eliminados).');
        }

        $totalGeral = $paymentsValidos->sum('valor_total');

        // Buscar nome da escola e logotipo
        $nomeEscola = Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE');
        $logotipo = Configuracao::get('logotipo_escola', null);
        $logotipoUrl = null;
        
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoUrl = asset('storage/' . $logotipo);
        }

        $professorEspecifico = null;
        if ($professorId && $paymentsValidos->first()) {
            $professorEspecifico = $paymentsValidos->first()->professor;
        }

        return view('admin.professor-payments.folha-salario', [
            'payments' => $paymentsValidos,
            'mes' => $mes,
            'ano' => $ano,
            'mesNome' => $mesNome,
            'totalGeral' => $totalGeral,
            'professorEspecifico' => $professorEspecifico,
            'nomeEscola' => $nomeEscola,
            'logotipoBase64' => null,
            'logotipoUrl' => $logotipoUrl,
        ]);
    }

    /**
     * Gerar pagamentos do mês seguinte baseado no mês atual
     */
    public function gerarMesSeguinte(Request $request)
    {
        $this->verificarPermissao();
        
        $mesAtual = $request->get('mes', date('m'));
        $anoAtual = $request->get('ano', date('Y'));

        // Calcular mês e ano seguinte
        $dataSeguinte = \Carbon\Carbon::create($anoAtual, $mesAtual, 1)->addMonth();
        $mesSeguinte = $dataSeguinte->month;
        $anoSeguinte = $dataSeguinte->year;

        // Buscar pagamentos do mês atual (PAGO ou PENDENTE)
        $paymentsAtual = ProfessorPayment::with(['professor.user'])
            ->where('mes', $mesAtual)
            ->where('ano', $anoAtual)
            ->whereIn('status', ['PAGO', 'PENDENTE'])
            ->get();

        if ($paymentsAtual->isEmpty()) {
            return redirect()->back()
                ->with('error', 'Nenhum pagamento encontrado para o mês atual. Não é possível gerar o mês seguinte.');
        }

        $criados = 0;
        $ignorados = 0;
        $erros = [];

        DB::beginTransaction();
        try {
            foreach ($paymentsAtual as $paymentAtual) {
                // Verificar se o professor ainda existe
                if (!$paymentAtual->professor) {
                    $erros[] = "Erro: Professor ID {$paymentAtual->professor_id} não encontrado (foi eliminado).";
                    continue;
                }

                // Verificar se já existe pagamento para o mês seguinte
                $existe = ProfessorPayment::where('professor_id', $paymentAtual->professor_id)
                    ->where('mes', $mesSeguinte)
                    ->where('ano', $anoSeguinte)
                    ->exists();

                if ($existe) {
                    $ignorados++;
                    continue;
                }

                // Criar novo pagamento para o mês seguinte
                try {
                // Calcular INSS e IRPS
                $inss = $this->calcularINSS($paymentAtual->valor_total);
                $numeroDependentes = $paymentAtual->professor->numero_dependentes ?? 0;
                $irps = $this->calcularIRPS($paymentAtual->valor_total, $numeroDependentes);
                $valorLiquido = $paymentAtual->valor_total - $inss - $irps;
                    
                    ProfessorPayment::create([
                        'professor_id' => $paymentAtual->professor_id,
                        'mes' => $mesSeguinte,
                        'ano' => $anoSeguinte,
                        'horas_trabalhadas' => $paymentAtual->horas_trabalhadas,
                        'valor_total' => $paymentAtual->valor_total,
                        'inss' => $inss,
                        'irps' => $irps,
                        'valor_liquido' => $valorLiquido,
                        'status' => 'PENDENTE',
                        'observacoes' => 'Gerado automaticamente do mês anterior',
                        'created_by' => Auth::id(),
                    ]);
                    $criados++;
                } catch (\Exception $e) {
                    $nomeProfessor = $paymentAtual->professor && $paymentAtual->professor->user 
                        ? $paymentAtual->professor->user->name 
                        : "Professor ID {$paymentAtual->professor_id}";
                    $erros[] = "Erro ao criar pagamento para {$nomeProfessor}: " . $e->getMessage();
                }
            }

            DB::commit();

            $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
                     'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
            $mesNome = $meses[$mesSeguinte];

            $mensagem = "Pagamentos gerados para {$mesNome}/{$anoSeguinte}! {$criados} pagamento(s) criado(s).";
            if ($ignorados > 0) {
                $mensagem .= " {$ignorados} pagamento(s) já existente(s) foram ignorado(s).";
            }
            if (!empty($erros)) {
                $mensagem .= " Erros: " . implode('; ', array_slice($erros, 0, 3));
            }

            return redirect()->route('admin.professor-payments.index', ['mes' => $mesSeguinte, 'ano' => $anoSeguinte])
                ->with('success', $mensagem);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Erro ao gerar pagamentos: ' . $e->getMessage());
        }
    }

    /**
     * Calcular INSS (3% do salário bruto)
     */
    private function calcularINSS($valorBruto)
    {
        // INSS: 3% do salário bruto (descontado do funcionário)
        return round($valorBruto * 0.03, 2);
    }

    /**
     * Calcular IRPS (Imposto sobre Rendimento) conforme tabela oficial de Moçambique
     * Baseado na tabela de retenção de IRPS com intervalos mensais e número de dependentes
     * 
     * @param float $valorBrutoMensal Salário bruto mensal em MTs
     * @param int $numeroDependentes Número de dependentes (0, 1, 2, 3, 4 ou mais)
     * @return float Valor do IRPS a reter
     */
    private function calcularIRPS($valorBrutoMensal, $numeroDependentes = 0)
    {
        // Limitar número de dependentes a 4 ou mais (tratado como 4)
        if ($numeroDependentes > 4) {
            $numeroDependentes = 4;
        }
        
        // Tabela oficial de IRPS - Valores fixos a reter por intervalo e número de dependentes
        // Conforme tabela oficial de Moçambique
        $tabelaIRPS = [
            // Intervalo: Até 20.249,99 - Não há imposto a reter
            ['min' => 0, 'max' => 20249.99, 'dependentes' => [
                0 => 0, 1 => 0, 2 => 0, 3 => 0, 4 => 0
            ], 'coeficiente' => 0],
            
            // Intervalo: 20.250,00 até 20.749,99
            ['min' => 20250.00, 'max' => 20749.99, 'dependentes' => [
                0 => 0, 1 => null, 2 => null, 3 => null, 4 => null
            ], 'coeficiente' => 0.10],
            
            // Intervalo: 20.750,00 até 20.999,99
            ['min' => 20750.00, 'max' => 20999.99, 'dependentes' => [
                0 => 50, 1 => 0, 2 => null, 3 => null, 4 => null
            ], 'coeficiente' => 0.10],
            
            // Intervalo: 21.000,00 até 21.249,99
            ['min' => 21000.00, 'max' => 21249.99, 'dependentes' => [
                0 => 75, 1 => 25, 2 => 0, 3 => null, 4 => null
            ], 'coeficiente' => 0.10],
            
            // Intervalo: 21.250,00 até 21.749,99
            ['min' => 21250.00, 'max' => 21749.99, 'dependentes' => [
                0 => 100, 1 => 50, 2 => 25, 3 => 0, 4 => null
            ], 'coeficiente' => 0.10],
            
            // Intervalo: 21.750,00 até 22.249,99
            ['min' => 21750.00, 'max' => 22249.99, 'dependentes' => [
                0 => 150, 1 => 100, 2 => 75, 3 => 50, 4 => 0
            ], 'coeficiente' => 0.10],
            
            // Intervalo: 22.250,00 até 32.749,99
            ['min' => 22250.00, 'max' => 32749.99, 'dependentes' => [
                0 => 200, 1 => 150, 2 => 125, 3 => 100, 4 => 50
            ], 'coeficiente' => 0.15],
            
            // Intervalo: 32.750,00 até 60.749,99
            ['min' => 32750.00, 'max' => 60749.99, 'dependentes' => [
                0 => 1775.00, 1 => 1725.00, 2 => 1700.00, 3 => 1675.00, 4 => 1625.00
            ], 'coeficiente' => 0.20],
            
            // Intervalo: 60.750,00 até 144.749,99
            ['min' => 60750.00, 'max' => 144749.99, 'dependentes' => [
                0 => 7375.00, 1 => 7325.00, 2 => 7300.00, 3 => 7275.00, 4 => 7225.00
            ], 'coeficiente' => 0.25],
            
            // Intervalo: 144.750,00 em diante
            ['min' => 144750.00, 'max' => PHP_FLOAT_MAX, 'dependentes' => [
                0 => 28375.00, 1 => 28325.00, 2 => 28300.00, 3 => 28275.00, 4 => 28225.00
            ], 'coeficiente' => 0.32],
        ];
        
        // Encontrar o intervalo correspondente ao salário bruto
        $intervalo = null;
        foreach ($tabelaIRPS as $item) {
            if ($valorBrutoMensal >= $item['min'] && $valorBrutoMensal <= $item['max']) {
                $intervalo = $item;
                break;
            }
        }
        
        // Se não encontrou intervalo (salário muito alto), usar o último
        if (!$intervalo) {
            $intervalo = end($tabelaIRPS);
        }
        
        // Obter valor fixo a reter baseado no número de dependentes
        // Se o valor for null (indicado por '-' na tabela), significa que não há imposto a reter
        $valorFixo = $intervalo['dependentes'][$numeroDependentes] ?? null;
        
        // Se o valor é null ou se não há imposto a reter (valor fixo = 0 e coeficiente = 0)
        if ($valorFixo === null || ($valorFixo == 0 && $intervalo['coeficiente'] == 0)) {
            return 0;
        }
        
        // Calcular IRPS: Valor fixo + (coeficiente × diferença entre salário e limite inferior)
        $diferenca = $valorBrutoMensal - $intervalo['min'];
        $irps = $valorFixo + ($diferenca * $intervalo['coeficiente']);
        
        return round($irps, 2);
    }
}
