<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Configuracao;
use App\Models\FuncionarioServico;
use App\Models\FuncionarioServicoPayment;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class FuncionarioServicoPaymentController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin,superadmin']);
    }

    public function index(Request $request)
    {
        $query = FuncionarioServicoPayment::with(['funcionarioServico', 'createdBy'])
            ->whereHas('funcionarioServico', function ($q) {
                $q->whereNull('deleted_at');
            });

        if ($request->has('funcionario_id') && $request->funcionario_id !== '' && $request->funcionario_id !== null) {
            $query->where('funcionario_servico_id', $request->funcionario_id);
        }

        if ($request->has('mes') && $request->mes !== '' && $request->mes !== null) {
            $query->where('mes', $request->mes);
        }

        if ($request->has('ano') && $request->ano !== '' && $request->ano !== null) {
            $query->where('ano', $request->ano);
        }

        if ($request->has('status') && $request->status !== '' && $request->status !== null) {
            $query->where('status', $request->status);
        }

        $payments = $query->orderBy('ano', 'desc')
            ->orderBy('mes', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(25)
            ->appends($request->query());

        // Atualizar cálculos ausentes
        foreach ($payments as $payment) {
            if (($payment->inss == 0 && $payment->irps == 0 && $payment->valor_liquido == 0) ||
                ($payment->inss === null || $payment->irps === null || $payment->valor_liquido === null)) {
                $inss = $this->calcularINSS($payment->valor_total);
                $deps = $payment->funcionarioServico ? ($payment->funcionarioServico->numero_dependentes ?? 0) : 0;
                $irps = $this->calcularIRPS($payment->valor_total, $deps);
                $liq = $payment->valor_total - $inss - $irps;
                $payment->update(['inss' => $inss, 'irps' => $irps, 'valor_liquido' => $liq]);
            }
        }

        $funcionarios = FuncionarioServico::query()
            ->whereNull('deleted_at')
            ->orderBy('nome')
            ->get();

        $isAjax = $request->ajax() ||
            $request->wantsJson() ||
            $request->expectsJson() ||
            $request->header('X-Requested-With') === 'XMLHttpRequest';

        if ($isAjax) {
            return response()->json([
                'success' => true,
                'payments' => $payments->map(function (FuncionarioServicoPayment $payment) {
                    $f = $payment->funcionarioServico;
                    return [
                        'id' => $payment->id,
                        'funcionario_id' => $payment->funcionario_servico_id,
                        'funcionario_nome' => $f ? $f->nome : 'Funcionário eliminado',
                        // Funcionários de serviços não têm "código" próprio como professores/administração.
                        // Usamos um identificador amigável para não aparecer "undefined" no frontend.
                        'funcionario_codigo' => $f ? ('#' . $f->id) : ('#' . $payment->funcionario_servico_id),
                        'cargo' => $f ? ($f->cargo ?? '-') : '-',
                        'mes' => $payment->mes,
                        'ano' => $payment->ano,
                        'mes_abrev' => ['', 'Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set', 'Out', 'Nov', 'Dez'][$payment->mes] ?? '',
                        'ano_abrev' => substr((string) $payment->ano, -2),
                        'numero_dependentes' => $f ? ($f->numero_dependentes ?? 0) : 0,
                        'salario_base' => $f ? ($f->salario_base ?? 0) : 0,
                        'valor_total' => $payment->valor_total,
                        'inss' => $payment->inss ?? 0,
                        'irps' => $payment->irps ?? 0,
                        'valor_liquido' => $payment->valor_liquido ?? ((float) $payment->valor_total - (float) ($payment->inss ?? 0) - (float) ($payment->irps ?? 0)),
                        'status' => $payment->status,
                        'data_pagamento' => $payment->data_pagamento ? $payment->data_pagamento->format('d/m/y') : null,
                    ];
                }),
                'pagination' => [
                    'current_page' => $payments->currentPage(),
                    'last_page' => $payments->lastPage(),
                    'per_page' => $payments->perPage(),
                    'total' => $payments->total(),
                ],
                'funcionarios' => $funcionarios->map(function (FuncionarioServico $f) {
                    return [
                        'id' => $f->id,
                        'nome' => $f->nome,
                        'cargo' => $f->cargo,
                        'numero_dependentes' => $f->numero_dependentes ?? 0,
                        'salario_base' => $f->salario_base ?? 0,
                    ];
                }),
            ]);
        }

        return view('admin.funcionario-payments.index', compact('payments', 'funcionarios'));
    }

    /**
     * Gerar folha de salário (todos os funcionários ou específico)
     */
    public function gerarFolhaSalario(Request $request)
    {
        $funcionarioId = $request->get('funcionario_id');
        $mes = $request->get('mes', date('m'));
        $ano = $request->get('ano', date('Y'));

        $query = FuncionarioServicoPayment::with(['funcionarioServico', 'createdBy'])
            ->whereHas('funcionarioServico', function ($q) {
                $q->whereNull('deleted_at');
            })
            ->where('mes', $mes)
            ->where('ano', $ano)
            ->where('status', 'PAGO');

        if ($funcionarioId) {
            $query->where('funcionario_servico_id', $funcionarioId);
        }

        $payments = $query->orderBy('funcionario_servico_id')->get();
        if ($payments->isEmpty()) {
            return redirect()->back()->with('error', 'Nenhum pagamento encontrado para o período selecionado.');
        }

        $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
            'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
        $mesNome = $meses[(int) $mes] ?? (string) $mes;

        $nomeEscola = Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE');
        $logotipo = Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;

        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $mime = mime_content_type($logotipoPath);
            $logotipoBase64 = "data:{$mime};base64," . base64_encode($logotipoData);
        }

        $paymentsValidos = $payments->filter(fn ($p) => (bool) $p->funcionarioServico);
        if ($paymentsValidos->isEmpty()) {
            return redirect()->back()->with('error', 'Nenhum pagamento válido encontrado (funcionários podem ter sido eliminados).');
        }

        $funcionarioEspecifico = null;
        $nomeArquivo = 'folha-salario-funcionarios-' . $mesNome . '-' . $ano . '.pdf';
        if ($funcionarioId && $paymentsValidos->first()) {
            $funcionarioEspecifico = $paymentsValidos->first()->funcionarioServico;
            if ($funcionarioEspecifico) {
                $nomeArquivo = 'folha-salario-funcionario-' . $funcionarioEspecifico->id . '-' . $mesNome . '-' . $ano . '.pdf';
            }
        }

        $pdf = Pdf::loadView('admin.payments.folha-salario-funcionarios', [
            'payments' => $paymentsValidos,
            'mes' => $mes,
            'ano' => $ano,
            'mesNome' => $mesNome,
            'totalGeral' => $paymentsValidos->sum('valor_total'),
            'nomeEscola' => $nomeEscola,
            'logotipoBase64' => $logotipoBase64,
            'titulo' => 'FOLHA DE SALÁRIO - FUNCIONÁRIOS',
            'entidadeLabel' => 'Funcionário',
            'entidadeEspecifica' => $funcionarioEspecifico,
            'tipoEntidade' => 'servico',
        ]);

        return $pdf->download($nomeArquivo);
    }

    /**
     * Visualizar folha de salário (HTML para impressão)
     */
    public function visualizarFolhaSalario(Request $request)
    {
        $funcionarioId = $request->get('funcionario_id');
        $mes = $request->get('mes', date('m'));
        $ano = $request->get('ano', date('Y'));

        $query = FuncionarioServicoPayment::with(['funcionarioServico', 'createdBy'])
            ->whereHas('funcionarioServico', function ($q) {
                $q->whereNull('deleted_at');
            })
            ->where('mes', $mes)
            ->where('ano', $ano)
            ->where('status', 'PAGO');

        if ($funcionarioId) {
            $query->where('funcionario_servico_id', $funcionarioId);
        }

        $payments = $query->orderBy('funcionario_servico_id')->get();
        if ($payments->isEmpty()) {
            return redirect()->back()->with('error', 'Nenhum pagamento encontrado para o período selecionado.');
        }

        $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
            'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
        $mesNome = $meses[(int) $mes] ?? (string) $mes;

        $paymentsValidos = $payments->filter(fn ($p) => (bool) $p->funcionarioServico);
        if ($paymentsValidos->isEmpty()) {
            return redirect()->back()->with('error', 'Nenhum pagamento válido encontrado (funcionários podem ter sido eliminados).');
        }

        $nomeEscola = Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE');
        $logotipo = Configuracao::get('logotipo_escola', null);
        $logotipoUrl = null;
        if ($logotipo && Storage::disk('public')->exists($logotipo)) {
            $logotipoUrl = asset('storage/' . $logotipo);
        }

        $funcionarioEspecifico = null;
        if ($funcionarioId && $paymentsValidos->first()) {
            $funcionarioEspecifico = $paymentsValidos->first()->funcionarioServico;
        }

        return view('admin.payments.folha-salario-funcionarios', [
            'payments' => $paymentsValidos,
            'mes' => $mes,
            'ano' => $ano,
            'mesNome' => $mesNome,
            'totalGeral' => $paymentsValidos->sum('valor_total'),
            'nomeEscola' => $nomeEscola,
            'logotipoBase64' => null,
            'logotipoUrl' => $logotipoUrl,
            'titulo' => 'FOLHA DE SALÁRIO - FUNCIONÁRIOS',
            'entidadeLabel' => 'Funcionário',
            'entidadeEspecifica' => $funcionarioEspecifico,
            'tipoEntidade' => 'servico',
        ]);
    }

    /**
     * Gerar pagamentos do mês seguinte baseado no mês atual
     */
    public function gerarMesSeguinte(Request $request)
    {
        $mesAtual = $request->get('mes', date('m'));
        $anoAtual = $request->get('ano', date('Y'));

        $dataSeguinte = \Carbon\Carbon::create($anoAtual, $mesAtual, 1)->addMonth();
        $mesSeguinte = $dataSeguinte->month;
        $anoSeguinte = $dataSeguinte->year;

        $paymentsAtual = FuncionarioServicoPayment::with(['funcionarioServico'])
            ->where('mes', $mesAtual)
            ->where('ano', $anoAtual)
            ->whereIn('status', ['PAGO', 'PENDENTE'])
            ->get();

        if ($paymentsAtual->isEmpty()) {
            return redirect()->back()->with('error', 'Nenhum pagamento encontrado para o mês atual. Não é possível gerar o mês seguinte.');
        }

        $criados = 0;
        $ignorados = 0;
        $erros = [];

        DB::beginTransaction();
        try {
            foreach ($paymentsAtual as $paymentAtual) {
                if (!$paymentAtual->funcionarioServico) {
                    $erros[] = "Erro: Funcionário ID {$paymentAtual->funcionario_servico_id} não encontrado (foi eliminado).";
                    continue;
                }

                $existe = FuncionarioServicoPayment::where('funcionario_servico_id', $paymentAtual->funcionario_servico_id)
                    ->where('mes', $mesSeguinte)
                    ->where('ano', $anoSeguinte)
                    ->exists();

                if ($existe) {
                    $ignorados++;
                    continue;
                }

                try {
                    $inss = $this->calcularINSS($paymentAtual->valor_total);
                    $deps = $paymentAtual->funcionarioServico->numero_dependentes ?? 0;
                    $irps = $this->calcularIRPS($paymentAtual->valor_total, $deps);
                    $liq = $paymentAtual->valor_total - $inss - $irps;

                    FuncionarioServicoPayment::create([
                        'funcionario_servico_id' => $paymentAtual->funcionario_servico_id,
                        'mes' => $mesSeguinte,
                        'ano' => $anoSeguinte,
                        'valor_total' => $paymentAtual->valor_total,
                        'inss' => $inss,
                        'irps' => $irps,
                        'valor_liquido' => $liq,
                        'status' => 'PENDENTE',
                        'observacoes' => 'Gerado automaticamente do mês anterior',
                        'created_by' => Auth::id(),
                    ]);
                    $criados++;
                } catch (\Exception $e) {
                    $nome = $paymentAtual->funcionarioServico ? $paymentAtual->funcionarioServico->nome : "Funcionário ID {$paymentAtual->funcionario_servico_id}";
                    $erros[] = "Erro ao criar pagamento para {$nome}: " . $e->getMessage();
                }
            }

            DB::commit();

            $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho',
                'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
            $mesNome = $meses[$mesSeguinte] ?? $mesSeguinte;

            $mensagem = "Pagamentos gerados para {$mesNome}/{$anoSeguinte}! {$criados} pagamento(s) criado(s).";
            if ($ignorados > 0) $mensagem .= " {$ignorados} pagamento(s) já existente(s) foram ignorado(s).";
            if (!empty($erros)) $mensagem .= " Erros: " . implode('; ', array_slice($erros, 0, 3));

            return redirect()->route('admin.funcionario-payments.index', ['mes' => $mesSeguinte, 'ano' => $anoSeguinte])
                ->with('success', $mensagem);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Erro ao gerar pagamentos: ' . $e->getMessage());
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'funcionario_id' => 'required|exists:funcionarios_servicos,id',
            'mes' => 'required|integer|min:1|max:12',
            'ano' => 'required|integer|min:2020|max:2100',
            'valor_total' => 'required|numeric|min:0',
            'observacoes' => 'nullable|string|max:1000',
        ]);

        $funcionario = FuncionarioServico::findOrFail($request->funcionario_id);

        $existe = FuncionarioServicoPayment::where('funcionario_servico_id', $funcionario->id)
            ->where('mes', $request->mes)
            ->where('ano', $request->ano)
            ->exists();

        if ($existe) {
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Já existe um pagamento para este funcionário neste mês/ano.'], 422);
            }
            return redirect()->back()->withErrors(['error' => 'Já existe um pagamento para este funcionário neste mês/ano.'])->withInput();
        }

        $valor = (float) $request->valor_total;
        $inss = $this->calcularINSS($valor);
        $deps = $funcionario->numero_dependentes ?? 0;
        $irps = $this->calcularIRPS($valor, $deps);
        $liq = $valor - $inss - $irps;

        $payment = FuncionarioServicoPayment::create([
            'funcionario_servico_id' => $funcionario->id,
            'mes' => $request->mes,
            'ano' => $request->ano,
            'valor_total' => $valor,
            'inss' => $inss,
            'irps' => $irps,
            'valor_liquido' => $liq,
            'status' => 'PENDENTE',
            'observacoes' => $request->observacoes,
            'created_by' => Auth::id(),
        ]);

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Pagamento criado com sucesso!', 'payment' => $payment->fresh(['funcionarioServico'])]);
        }

        return redirect()->route('admin.funcionario-payments.index')->with('success', 'Pagamento criado com sucesso!');
    }

    public function update(Request $request, FuncionarioServicoPayment $payment)
    {
        if ($payment->status === 'PAGO') {
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Não é possível atualizar um pagamento já pago.'], 422);
            }
            return redirect()->back()->withErrors(['error' => 'Não é possível atualizar um pagamento já pago.']);
        }

        $request->validate([
            'valor_total' => 'required|numeric|min:0',
        ]);

        $valor = (float) $request->valor_total;
        $inss = $this->calcularINSS($valor);
        $deps = $payment->funcionarioServico ? ($payment->funcionarioServico->numero_dependentes ?? 0) : 0;
        $irps = $this->calcularIRPS($valor, $deps);
        $liq = $valor - $inss - $irps;

        $payment->update([
            'valor_total' => $valor,
            'inss' => $inss,
            'irps' => $irps,
            'valor_liquido' => $liq,
        ]);

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento atualizado com sucesso!',
                'inss' => number_format($inss, 2, '.', ''),
                'irps' => number_format($irps, 2, '.', ''),
                'valor_liquido' => number_format($liq, 2, '.', ''),
            ]);
        }

        return redirect()->back()->with('success', 'Pagamento atualizado com sucesso!');
    }

    public function marcarComoPago(Request $request, FuncionarioServicoPayment $payment)
    {
        $payment->update([
            'status' => 'PAGO',
            'data_pagamento' => now(),
        ]);

        if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento marcado como pago!',
                'payment' => [
                    'id' => $payment->id,
                    'status' => $payment->status,
                    'data_pagamento' => $payment->data_pagamento->format('d/m/y'),
                ],
            ]);
        }

        return redirect()->back()->with('success', 'Pagamento marcado como pago!');
    }

    public function marcarTodosComoPago(Request $request)
    {
        $query = FuncionarioServicoPayment::where('status', 'PENDENTE');

        if ($request->has('mes') && $request->mes !== '' && $request->mes !== null) {
            $query->where('mes', $request->mes);
        }
        if ($request->has('ano') && $request->ano !== '' && $request->ano !== null) {
            $query->where('ano', $request->ano);
        }
        if ($request->has('funcionario_id') && $request->funcionario_id !== '' && $request->funcionario_id !== null) {
            $query->where('funcionario_servico_id', $request->funcionario_id);
        }

        $payments = $query->get();
        if ($payments->isEmpty()) {
            if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
                return response()->json(['success' => false, 'message' => 'Não há pagamentos pendentes para marcar como pagos.']);
            }
            return redirect()->back()->with('info', 'Não há pagamentos pendentes para marcar como pagos.');
        }

        $count = $payments->count();
        foreach ($payments as $p) {
            $p->update(['status' => 'PAGO', 'data_pagamento' => now()]);
        }

        if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
            return response()->json(['success' => true, 'message' => "{$count} pagamento(s) marcado(s) como pago(s) com sucesso!", 'count' => $count]);
        }

        return redirect()->back()->with('success', "{$count} pagamento(s) marcado(s) como pago(s) com sucesso!");
    }

    private function calcularINSS($valorBruto): float
    {
        return round(((float) $valorBruto) * 0.03, 2);
    }

    // Mesma tabela oficial usada em professores
    private function calcularIRPS($valorBrutoMensal, $numeroDependentes = 0): float
    {
        if ($numeroDependentes > 4) $numeroDependentes = 4;

        $tabelaIRPS = [
            ['min' => 0, 'max' => 20249.99, 'dependentes' => [0 => 0, 1 => 0, 2 => 0, 3 => 0, 4 => 0], 'coeficiente' => 0],
            ['min' => 20250.00, 'max' => 20749.99, 'dependentes' => [0 => 0, 1 => null, 2 => null, 3 => null, 4 => null], 'coeficiente' => 0.10],
            ['min' => 20750.00, 'max' => 20999.99, 'dependentes' => [0 => 50, 1 => 0, 2 => null, 3 => null, 4 => null], 'coeficiente' => 0.10],
            ['min' => 21000.00, 'max' => 21249.99, 'dependentes' => [0 => 75, 1 => 25, 2 => 0, 3 => null, 4 => null], 'coeficiente' => 0.10],
            ['min' => 21250.00, 'max' => 21749.99, 'dependentes' => [0 => 100, 1 => 50, 2 => 25, 3 => 0, 4 => null], 'coeficiente' => 0.10],
            ['min' => 21750.00, 'max' => 22249.99, 'dependentes' => [0 => 150, 1 => 100, 2 => 75, 3 => 50, 4 => 0], 'coeficiente' => 0.10],
            ['min' => 22250.00, 'max' => 32749.99, 'dependentes' => [0 => 200, 1 => 150, 2 => 125, 3 => 100, 4 => 50], 'coeficiente' => 0.15],
            ['min' => 32750.00, 'max' => 60749.99, 'dependentes' => [0 => 1775.00, 1 => 1725.00, 2 => 1700.00, 3 => 1675.00, 4 => 1625.00], 'coeficiente' => 0.20],
            ['min' => 60750.00, 'max' => 144749.99, 'dependentes' => [0 => 7375.00, 1 => 7325.00, 2 => 7300.00, 3 => 7275.00, 4 => 7225.00], 'coeficiente' => 0.25],
            ['min' => 144750.00, 'max' => PHP_FLOAT_MAX, 'dependentes' => [0 => 28375.00, 1 => 28325.00, 2 => 28300.00, 3 => 28275.00, 4 => 28225.00], 'coeficiente' => 0.32],
        ];

        $intervalo = null;
        foreach ($tabelaIRPS as $item) {
            if ($valorBrutoMensal >= $item['min'] && $valorBrutoMensal <= $item['max']) {
                $intervalo = $item;
                break;
            }
        }
        if (!$intervalo) $intervalo = end($tabelaIRPS);

        $valorFixo = $intervalo['dependentes'][$numeroDependentes] ?? null;
        if ($valorFixo === null || ($valorFixo == 0 && $intervalo['coeficiente'] == 0)) return 0;

        $diferenca = $valorBrutoMensal - $intervalo['min'];
        return round($valorFixo + ($diferenca * $intervalo['coeficiente']), 2);
    }
}

