<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\FuncionarioServico;
use Illuminate\Http\Request;

class FuncionarioServicoController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin,superadmin']);
    }

    public function index(Request $request)
    {
        $query = FuncionarioServico::query();

        if ($request->filled('search')) {
            $s = trim($request->search);
            $query->where(function ($q) use ($s) {
                $q->where('nome', 'like', "%{$s}%")
                    ->orWhere('cargo', 'like', "%{$s}%")
                    ->orWhere('telefone', 'like', "%{$s}%");
            });
        }

        if ($request->filled('cargo')) {
            $query->where('cargo', $request->cargo);
        }

        if ($request->filled('status')) {
            if ($request->status === 'ATIVO') {
                $query->where('is_active', true);
            } elseif ($request->status === 'INATIVO') {
                $query->where('is_active', false);
            }
        }

        $funcionarios = $query
            ->orderBy('nome')
            ->paginate(25)
            ->appends($request->query());

        // Buscar cargos únicos do banco para o filtro
        $cargosExistentes = FuncionarioServico::distinct()
            ->whereNotNull('cargo')
            ->where('cargo', '!=', '')
            ->orderBy('cargo')
            ->pluck('cargo')
            ->toArray();

        return view('admin.funcionarios-servicos.index', compact('funcionarios', 'cargosExistentes'));
    }

    public function create()
    {
        return view('admin.funcionarios-servicos.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'nome' => 'required|string|max:255',
            'cargo' => 'required|string|max:255',
            'salario_base' => 'required|numeric|min:0',
            'numero_dependentes' => 'nullable|integer|min:0|max:50',
            'telefone' => 'nullable|string|max:50',
            'endereco' => 'nullable|string|max:255',
        ]);

        FuncionarioServico::create([
            'nome' => $request->nome,
            'cargo' => $request->cargo,
            'salario_base' => $request->salario_base,
            'numero_dependentes' => $request->numero_dependentes ?? 0,
            'telefone' => $request->telefone,
            'endereco' => $request->endereco,
            'is_active' => true,
        ]);

        return redirect()->route('admin.funcionarios-servicos.index')
            ->with('success', 'Funcionário criado com sucesso!');
    }

    public function show(FuncionarioServico $funcionarioServico)
    {
        return view('admin.funcionarios-servicos.show', compact('funcionarioServico'));
    }

    public function edit(FuncionarioServico $funcionarioServico)
    {
        return view('admin.funcionarios-servicos.edit', compact('funcionarioServico'));
    }

    public function update(Request $request, FuncionarioServico $funcionarioServico)
    {
        $request->validate([
            'nome' => 'required|string|max:255',
            'cargo' => 'required|string|max:255',
            'salario_base' => 'required|numeric|min:0',
            'numero_dependentes' => 'nullable|integer|min:0|max:50',
            'telefone' => 'nullable|string|max:50',
            'endereco' => 'nullable|string|max:255',
        ]);

        $funcionarioServico->update([
            'nome' => $request->nome,
            'cargo' => $request->cargo,
            'salario_base' => $request->salario_base,
            'numero_dependentes' => $request->numero_dependentes ?? 0,
            'telefone' => $request->telefone,
            'endereco' => $request->endereco,
        ]);

        return redirect()->route('admin.funcionarios-servicos.index')
            ->with('success', 'Funcionário atualizado com sucesso!');
    }

    public function destroy(FuncionarioServico $funcionarioServico)
    {
        $funcionarioServico->delete();
        return redirect()->route('admin.funcionarios-servicos.index')
            ->with('success', 'Funcionário eliminado com sucesso!');
    }

    public function toggleStatus(FuncionarioServico $funcionarioServico)
    {
        $funcionarioServico->update([
            'is_active' => !$funcionarioServico->is_active,
        ]);

        $status = $funcionarioServico->is_active ? 'ativado' : 'desativado';
        return redirect()->back()->with('success', "Funcionário {$status} com sucesso!");
    }

    private function cargosPadrao(): array
    {
        return [
            'ADMINISTRAÇÃO',
            'LIMPEZA',
            'SEGURANÇA',
            'PORTARIA',
            'MANUTENÇÃO',
            'COZINHA',
            'MOTORISTA',
            'JARDINAGEM',
            'OUTRO',
        ];
    }
}

