<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Configuracao;
use App\Models\Funcionario;
use App\Models\FuncionarioPayment;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class FuncionarioPaymentController extends Controller
{
    public function __construct()
    {
        // Por enquanto, apenas Admin e Superadmin (funcionário pode ser liberado depois via permissão específica)
        $this->middleware(['auth', 'role:admin,superadmin']);
    }

    /**
     * Listar pagamentos de funcionários
     */
    public function index(Request $request)
    {
        $query = FuncionarioPayment::with(['funcionario.user', 'createdBy'])
            ->whereHas('funcionario', function ($q) {
                $q->whereNull('deleted_at');
            });

        // Filtros
        if ($request->has('funcionario_id') && $request->funcionario_id !== '' && $request->funcionario_id !== null) {
            $query->where('funcionario_id', $request->funcionario_id);
        }

        if ($request->has('mes') && $request->mes !== '' && $request->mes !== null) {
            $query->where('mes', $request->mes);
        }

        if ($request->has('ano') && $request->ano !== '' && $request->ano !== null) {
            $query->where('ano', $request->ano);
        }

        if ($request->has('status') && $request->status !== '' && $request->status !== null) {
            $query->where('status', $request->status);
        }

        $payments = $query->orderBy('ano', 'desc')
            ->orderBy('mes', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(25)
            ->appends($request->query());

        // Atualizar pagamentos que não têm INSS e IRPS calculados
        foreach ($payments as $payment) {
            if (($payment->inss == 0 && $payment->irps == 0 && $payment->valor_liquido == 0) ||
                ($payment->inss === null || $payment->irps === null || $payment->valor_liquido === null)) {
                $inss = $this->calcularINSS($payment->valor_total);
                $numeroDependentes = $payment->funcionario ? ($payment->funcionario->numero_dependentes ?? 0) : 0;
                $irps = $this->calcularIRPS($payment->valor_total, $numeroDependentes);
                $valorLiquido = $payment->valor_total - $inss - $irps;

                $payment->update([
                    'inss' => $inss,
                    'irps' => $irps,
                    'valor_liquido' => $valorLiquido,
                ]);
            }
        }

        $funcionarios = Funcionario::with('user')
            ->whereHas('user', function ($q) {
                $q->whereNull('deleted_at');
            })
            ->orderBy('numero_funcionario')
            ->get();

        // Se for requisição AJAX, retornar JSON (para paginação/filtros dinâmicos)
        $isAjax = $request->ajax() ||
            $request->wantsJson() ||
            $request->expectsJson() ||
            $request->header('X-Requested-With') === 'XMLHttpRequest';

        if ($isAjax) {
            return response()->json([
                'success' => true,
                'payments' => $payments->map(function (FuncionarioPayment $payment) {
                    $func = $payment->funcionario;
                    return [
                        'id' => $payment->id,
                        'funcionario_id' => $payment->funcionario_id,
                        'funcionario_nome' => $func && $func->user ? $func->user->name : 'Funcionário eliminado',
                        'funcionario_codigo' => $func ? ($func->numero_funcionario ?? '-') : '-',
                        'cargo' => $func ? ($func->cargo ?? '-') : '-',
                        'mes' => $payment->mes,
                        'ano' => $payment->ano,
                        'mes_abrev' => ['', 'Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set', 'Out', 'Nov', 'Dez'][$payment->mes] ?? '',
                        'ano_abrev' => substr((string) $payment->ano, -2),
                        'numero_dependentes' => $func ? ($func->numero_dependentes ?? 0) : 0,
                        'valor_total' => $payment->valor_total,
                        'inss' => $payment->inss ?? 0,
                        'irps' => $payment->irps ?? 0,
                        'valor_liquido' => $payment->valor_liquido ?? ((float) $payment->valor_total - (float) ($payment->inss ?? 0) - (float) ($payment->irps ?? 0)),
                        'status' => $payment->status,
                        'data_pagamento' => $payment->data_pagamento ? $payment->data_pagamento->format('d/m/y') : null,
                    ];
                }),
                'pagination' => [
                    'current_page' => $payments->currentPage(),
                    'last_page' => $payments->lastPage(),
                    'per_page' => $payments->perPage(),
                    'total' => $payments->total(),
                ],
                'funcionarios' => $funcionarios->map(function (Funcionario $func) {
                    return [
                        'id' => $func->id,
                        'nome' => $func->user ? $func->user->name : '-',
                        'codigo' => $func->numero_funcionario,
                        'cargo' => $func->cargo ?? '-',
                        'numero_dependentes' => $func->numero_dependentes ?? 0,
                        'salario_base' => $func->salario_base ?? 0,
                    ];
                }),
            ]);
        }

        return view('admin.funcionario-payments.index', compact('payments', 'funcionarios'));
    }

    /**
     * Armazenar pagamento
     */
    public function store(Request $request)
    {
        $request->validate([
            'funcionario_id' => 'required|exists:funcionarios,id',
            'mes' => 'required|integer|min:1|max:12',
            'ano' => 'required|integer|min:2020|max:2100',
            'valor_total' => 'required|numeric|min:0',
            'observacoes' => 'nullable|string|max:1000',
        ]);

        $funcionario = Funcionario::with('user')->findOrFail($request->funcionario_id);

        // Verificar se já existe pagamento para este mês/ano
        $existe = FuncionarioPayment::where('funcionario_id', $funcionario->id)
            ->where('mes', $request->mes)
            ->where('ano', $request->ano)
            ->exists();

        if ($existe) {
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Já existe um pagamento registrado para este funcionário neste mês/ano.',
                ], 422);
            }

            return redirect()->back()
                ->withErrors(['error' => 'Já existe um pagamento registrado para este funcionário neste mês/ano.'])
                ->withInput();
        }

        $valorTotal = (float) $request->valor_total;
        $inss = $this->calcularINSS($valorTotal);
        $numeroDependentes = $funcionario->numero_dependentes ?? 0;
        $irps = $this->calcularIRPS($valorTotal, $numeroDependentes);
        $valorLiquido = $valorTotal - $inss - $irps;

        $payment = FuncionarioPayment::create([
            'funcionario_id' => $funcionario->id,
            'mes' => $request->mes,
            'ano' => $request->ano,
            'valor_total' => $valorTotal,
            'inss' => $inss,
            'irps' => $irps,
            'valor_liquido' => $valorLiquido,
            'status' => 'PENDENTE',
            'observacoes' => $request->observacoes,
            'created_by' => Auth::id(),
        ]);

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento de funcionário registrado com sucesso!',
                'payment' => $payment->fresh(['funcionario.user']),
            ]);
        }

        return redirect()->route('admin.funcionario-payments.index', $request->only(['funcionario_id', 'mes', 'ano', 'status']))
            ->with('success', 'Pagamento de funcionário registrado com sucesso!');
    }

    /**
     * Atualizar pagamento (valor bruto e recalcular impostos)
     */
    public function update(Request $request, FuncionarioPayment $payment)
    {
        if ($payment->status === 'PAGO') {
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Não é possível atualizar um pagamento já pago.',
                ], 422);
            }

            return redirect()->back()->withErrors(['error' => 'Não é possível atualizar um pagamento já pago.']);
        }

        $request->validate([
            'valor_total' => 'required|numeric|min:0',
        ]);

        $valorTotal = (float) $request->valor_total;
        $inss = $this->calcularINSS($valorTotal);
        $numeroDependentes = $payment->funcionario ? ($payment->funcionario->numero_dependentes ?? 0) : 0;
        $irps = $this->calcularIRPS($valorTotal, $numeroDependentes);
        $valorLiquido = $valorTotal - $inss - $irps;

        $payment->update([
            'valor_total' => $valorTotal,
            'inss' => $inss,
            'irps' => $irps,
            'valor_liquido' => $valorLiquido,
        ]);

        if ($request->expectsJson() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento atualizado com sucesso!',
                'payment' => $payment->fresh(['funcionario.user']),
                'inss' => number_format($inss, 2, '.', ''),
                'irps' => number_format($irps, 2, '.', ''),
                'valor_liquido' => number_format($valorLiquido, 2, '.', ''),
            ]);
        }

        return redirect()->back()->with('success', 'Pagamento atualizado com sucesso!');
    }

    /**
     * Marcar como pago
     */
    public function marcarComoPago(Request $request, FuncionarioPayment $payment)
    {
        $payment->update([
            'status' => 'PAGO',
            'data_pagamento' => now(),
        ]);

        if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Pagamento marcado como pago!',
                'payment' => [
                    'id' => $payment->id,
                    'status' => $payment->status,
                    'data_pagamento' => $payment->data_pagamento->format('d/m/y'),
                ],
            ]);
        }

        return redirect()->back()->with('success', 'Pagamento marcado como pago!');
    }

    /**
     * Marcar todos os pagamentos pendentes como pagos (aplicando filtros se fornecidos)
     */
    public function marcarTodosComoPago(Request $request)
    {
        $query = FuncionarioPayment::where('status', 'PENDENTE');

        if ($request->has('mes') && $request->mes !== '' && $request->mes !== null) {
            $query->where('mes', $request->mes);
        }

        if ($request->has('ano') && $request->ano !== '' && $request->ano !== null) {
            $query->where('ano', $request->ano);
        }

        if ($request->has('funcionario_id') && $request->funcionario_id !== '' && $request->funcionario_id !== null) {
            $query->where('funcionario_id', $request->funcionario_id);
        }

        $payments = $query->get();

        if ($payments->isEmpty()) {
            if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Não há pagamentos pendentes para marcar como pagos.',
                ]);
            }

            return redirect()->back()->with('info', 'Não há pagamentos pendentes para marcar como pagos.');
        }

        $count = $payments->count();
        foreach ($payments as $payment) {
            $payment->update([
                'status' => 'PAGO',
                'data_pagamento' => now(),
            ]);
        }

        if ($request->ajax() || $request->wantsJson() || $request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => "{$count} pagamento(s) marcado(s) como pago(s) com sucesso!",
                'count' => $count,
            ]);
        }

        return redirect()->back()->with('success', "{$count} pagamento(s) marcado(s) como pago(s) com sucesso!");
    }

    /**
     * Calcular INSS (3% do salário bruto)
     */
    private function calcularINSS($valorBruto)
    {
        return round(((float) $valorBruto) * 0.03, 2);
    }

    /**
     * Calcular IRPS (tabela oficial - mesma regra usada para professores)
     */
    private function calcularIRPS($valorBrutoMensal, $numeroDependentes = 0)
    {
        if ($numeroDependentes > 4) {
            $numeroDependentes = 4;
        }

        $tabelaIRPS = [
            ['min' => 0, 'max' => 20249.99, 'dependentes' => [0 => 0, 1 => 0, 2 => 0, 3 => 0, 4 => 0], 'coeficiente' => 0],
            ['min' => 20250.00, 'max' => 20749.99, 'dependentes' => [0 => 0, 1 => null, 2 => null, 3 => null, 4 => null], 'coeficiente' => 0.10],
            ['min' => 20750.00, 'max' => 20999.99, 'dependentes' => [0 => 50, 1 => 0, 2 => null, 3 => null, 4 => null], 'coeficiente' => 0.10],
            ['min' => 21000.00, 'max' => 21249.99, 'dependentes' => [0 => 75, 1 => 25, 2 => 0, 3 => null, 4 => null], 'coeficiente' => 0.10],
            ['min' => 21250.00, 'max' => 21749.99, 'dependentes' => [0 => 100, 1 => 50, 2 => 25, 3 => 0, 4 => null], 'coeficiente' => 0.10],
            ['min' => 21750.00, 'max' => 22249.99, 'dependentes' => [0 => 150, 1 => 100, 2 => 75, 3 => 50, 4 => 0], 'coeficiente' => 0.10],
            ['min' => 22250.00, 'max' => 32749.99, 'dependentes' => [0 => 200, 1 => 150, 2 => 125, 3 => 100, 4 => 50], 'coeficiente' => 0.15],
            ['min' => 32750.00, 'max' => 60749.99, 'dependentes' => [0 => 1775.00, 1 => 1725.00, 2 => 1700.00, 3 => 1675.00, 4 => 1625.00], 'coeficiente' => 0.20],
            ['min' => 60750.00, 'max' => 144749.99, 'dependentes' => [0 => 7375.00, 1 => 7325.00, 2 => 7300.00, 3 => 7275.00, 4 => 7225.00], 'coeficiente' => 0.25],
            ['min' => 144750.00, 'max' => PHP_FLOAT_MAX, 'dependentes' => [0 => 28375.00, 1 => 28325.00, 2 => 28300.00, 3 => 28275.00, 4 => 28225.00], 'coeficiente' => 0.32],
        ];

        $intervalo = null;
        foreach ($tabelaIRPS as $item) {
            if ($valorBrutoMensal >= $item['min'] && $valorBrutoMensal <= $item['max']) {
                $intervalo = $item;
                break;
            }
        }

        if (!$intervalo) {
            $intervalo = end($tabelaIRPS);
        }

        $valorFixo = $intervalo['dependentes'][$numeroDependentes] ?? null;

        if ($valorFixo === null || ($valorFixo == 0 && $intervalo['coeficiente'] == 0)) {
            return 0;
        }

        $diferenca = $valorBrutoMensal - $intervalo['min'];
        $irps = $valorFixo + ($diferenca * $intervalo['coeficiente']);

        return round($irps, 2);
    }
}

