<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Aluno;

class DiagnosticarAcessoAluno extends Command
{
    protected $signature = 'aluno:diagnosticar-acesso {codigo?}';
    protected $description = 'Diagnostica problemas de acesso de alunos';

    public function handle()
    {
        $codigo = $this->argument('codigo');
        
        if ($codigo) {
            $aluno = Aluno::where('codigo_estudante', $codigo)->first();
            if (!$aluno) {
                $this->error("Aluno com código '{$codigo}' não encontrado.");
                return 1;
            }
            $this->diagnosticarAluno($aluno);
        } else {
            // Diagnosticar todos os alunos
            $alunos = Aluno::with('user')->limit(10)->get();
            $this->info("Diagnosticando primeiros 10 alunos...\n");
            
            foreach ($alunos as $aluno) {
                $this->diagnosticarAluno($aluno);
                $this->line("---");
            }
        }
        
        return 0;
    }
    
    private function diagnosticarAluno(Aluno $aluno)
    {
        $this->info("Aluno: {$aluno->codigo_estudante}");
        
        if (!$aluno->user) {
            $this->error("  ✗ SEM USER associado!");
            return;
        }
        
        $user = $aluno->user;
        $this->line("  User ID: {$user->id}");
        $this->line("  Nome: {$user->name}");
        $this->line("  Email: {$user->email}");
        $this->line("  is_active: " . var_export($user->is_active, true));
        $this->line("  Tipo: {$user->tipo}");
        
        // Verificar roles
        $roles = $user->roles->pluck('slug')->toArray();
        $this->line("  Roles: " . implode(', ', $roles ?: ['NENHUMA']));
        
        if (!in_array('aluno', $roles)) {
            $this->error("  ✗ NÃO TEM ROLE 'aluno'!");
            $this->line("  → Adicionando role 'aluno'...");
            
            $roleAluno = \App\Models\Role::where('slug', 'aluno')->first();
            if ($roleAluno) {
                if (!$user->roles->contains($roleAluno->id)) {
                    $user->roles()->attach($roleAluno->id);
                    $this->info("  ✓ Role 'aluno' adicionada!");
                }
            } else {
                $this->error("  ✗ Role 'aluno' não existe no banco!");
            }
        } else {
            $this->info("  ✓ Tem role 'aluno'");
        }
        
        // Verificar se pode acessar
        $podeAcessar = $user->is_active && in_array('aluno', $roles);
        if ($podeAcessar) {
            $this->info("  ✓ PODE ACESSAR como aluno");
        } else {
            $this->error("  ✗ NÃO PODE ACESSAR como aluno");
            if (!$user->is_active) {
                $this->line("    - Motivo: is_active = false");
            }
            if (!in_array('aluno', $roles)) {
                $this->line("    - Motivo: não tem role 'aluno'");
            }
        }
    }
}
