<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\Role;

class DesbloquearAlunosEProfessores extends Command
{
    protected $signature = 'users:desbloquear-alunos-professores';
    protected $description = 'Desbloqueia todos os alunos e professores';

    public function handle()
    {
        // Buscar IDs das roles de aluno e professor
        $roleAluno = Role::where('slug', 'aluno')->first();
        $roleProfessor = Role::where('slug', 'professor')->first();

        if (!$roleAluno && !$roleProfessor) {
            $this->error('Roles de aluno e professor não encontradas.');
            return 1;
        }

        $roleIds = [];
        if ($roleAluno) {
            $roleIds[] = $roleAluno->id;
            $this->info("Role Aluno encontrada: ID {$roleAluno->id}");
        }
        if ($roleProfessor) {
            $roleIds[] = $roleProfessor->id;
            $this->info("Role Professor encontrada: ID {$roleProfessor->id}");
        }

        // Verificar quantos usuários têm essas roles
        $totalUsers = DB::table('users')
            ->whereIn('id', function($query) use ($roleIds) {
                $query->select('user_id')
                    ->from('user_role')
                    ->whereIn('role_id', $roleIds);
            })
            ->count();

        $this->info("Total de usuários encontrados: {$totalUsers}");

        // Verificar quantos estão bloqueados
        $bloqueados = DB::table('users')
            ->whereIn('id', function($query) use ($roleIds) {
                $query->select('user_id')
                    ->from('user_role')
                    ->whereIn('role_id', $roleIds);
            })
            ->where(function($q) {
                $q->where('is_active', false)
                  ->orWhere('is_active', 0)
                  ->orWhereNull('is_active');
            })
            ->count();

        $this->info("Usuários bloqueados: {$bloqueados}");

        // Atualizar TODOS os usuários que têm role de aluno ou professor
        // Forçar is_active = 1 mesmo para os que já estão desbloqueados
        $usersAtualizados = DB::table('users')
            ->whereIn('id', function($query) use ($roleIds) {
                $query->select('user_id')
                    ->from('user_role')
                    ->whereIn('role_id', $roleIds);
            })
            ->update([
                'is_active' => 1, // Forçar 1 explicitamente
                'updated_at' => now()
            ]);
        
        $this->info("Atualizados {$usersAtualizados} registros no banco de dados.");

        $this->info("✓ Desbloqueados {$usersAtualizados} alunos e professores com sucesso!");

        // Verificar novamente
        $aindaBloqueados = DB::table('users')
            ->whereIn('id', function($query) use ($roleIds) {
                $query->select('user_id')
                    ->from('user_role')
                    ->whereIn('role_id', $roleIds);
            })
            ->where(function($q) {
                $q->where('is_active', false)
                  ->orWhereNull('is_active');
            })
            ->count();

        if ($aindaBloqueados > 0) {
            $this->warn("Ainda há {$aindaBloqueados} usuários bloqueados. Verificando...");
            
            // Listar alguns exemplos
            $exemplos = DB::table('users')
                ->whereIn('id', function($query) use ($roleIds) {
                    $query->select('user_id')
                        ->from('user_role')
                        ->whereIn('role_id', $roleIds);
                })
                ->where(function($q) {
                    $q->where('is_active', false)
                      ->orWhereNull('is_active');
                })
                ->select('id', 'name', 'is_active')
                ->limit(5)
                ->get();

            foreach ($exemplos as $user) {
                $this->line("  - ID: {$user->id}, Nome: {$user->name}, is_active: " . ($user->is_active ?? 'NULL'));
            }
        } else {
            $this->info("✓ Todos os usuários foram desbloqueados com sucesso!");
        }

        return 0;
    }
}
